/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#ifndef MICROFEED_THREADS_H
#define MICROFEED_THREADS_H

/**
 * @addtogroup common libmicrofeed-common - Common modules used both in the publisher and in the subscriber side
 * @{
 * @addtogroup MicrofeedThread
 *
 * A thread and a mutex.
 *
 * This module can be also utilized outside of the library implementation.
 * In addition, if the default implementation of thread and mutex is not sufficient, you can change
 * the functions with #microfeed_thread_set_functions.
 *
 * @{
 */

/**
 * Opaque data type representing thread.
 */
typedef struct _MicrofeedThread MicrofeedThread;

typedef struct _MicrofeedThreadPool MicrofeedThreadPool;

/**
 * Opaque data type representing mutex.
 */
typedef struct _MicrofeedMutex MicrofeedMutex;

typedef void* (*MicrofeedThreadFunction)(void* data);

typedef void (*MicrofeedThreadExitCallback)(MicrofeedThread* thread, void* user_data);

typedef struct {
	void* (*thread_new)(void* (*function)(void* data), void* data);
	void (*thread_free)(void* thread_implementation);
	void* (*thread_get_current)(void);
	void (*thread_send_signal)(void* thread_implementation, int signal_number);
	void (*thread_join)(void* thread_implementation);
	void* (*mutex_new)(void);
	void (*mutex_free)(void* mutex_implementation);
	void (*mutex_lock)(void* mutex_implementation);
	void (*mutex_unlock)(void* mutex_implementation);
} MicrofeedThreadFunctions;

void microfeed_thread_set_functions(MicrofeedThreadFunctions* thread_functions);

void microfeed_thread_init(void);
void microfeed_thread_cleanup(void);

MicrofeedThread* microfeed_thread_new(MicrofeedThreadFunction function, void* data);
MicrofeedThread* microfeed_thread_new_with_exit_callback(MicrofeedThreadFunction function, void* data, MicrofeedThreadExitCallback exit_callback, void* user_data);
void microfeed_thread_free(MicrofeedThread* thread);
MicrofeedThread* microfeed_thread_ref(MicrofeedThread* thread);
void microfeed_thread_unref(MicrofeedThread* thread);

MicrofeedThread* microfeed_thread_get_current(void);
void microfeed_thread_send_signal(MicrofeedThread* thread, int signal_number);
unsigned long microfeed_thread_get_id(MicrofeedThread* thread);

MicrofeedThreadPool* microfeed_thread_pool_new(unsigned int maximum_thread_count);
MicrofeedThreadPool* microfeed_thread_pool_new_with_exit_callback(unsigned int max_threads, MicrofeedThreadExitCallback exit_callback, void* user_data);
void microfeed_thread_pool_free(MicrofeedThreadPool* thread_pool);
MicrofeedThread* microfeed_thread_pool_queue_thread(MicrofeedThreadPool* thread_pool, MicrofeedThreadFunction function, void* data);
MicrofeedThread* microfeed_thread_pool_queue_thread_with_exit_callback(MicrofeedThreadPool* thread_pool, MicrofeedThreadFunction function, void* data, MicrofeedThreadExitCallback exit_callback, void* user_data);
unsigned int microfeed_thread_pool_get_started_thread_count(MicrofeedThreadPool* thread_pool);
unsigned int microfeed_thread_pool_get_waiting_thread_count(MicrofeedThreadPool* thread_pool);
void microfeed_thread_pool_set_maximum_thread_count(MicrofeedThreadPool* thread_pool, unsigned int maximum_thread_count);

MicrofeedMutex* microfeed_mutex_new(void);
void microfeed_mutex_free(MicrofeedMutex* mutex);
void microfeed_mutex_lock(MicrofeedMutex* mutex);
void microfeed_mutex_unlock(MicrofeedMutex* mutex);

/**
 * @}
 * @}
 */

#endif
