/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-common/microfeedmisc.h>

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <errno.h>
#include <libgen.h>
#include <sys/types.h>
#include <fcntl.h>
#include <time.h>

int microfeed_debug_memory = 0;

void* microfeed_memory_allocate_bytes(size_t size) {
	void* pointer;
	
	if (!(pointer = malloc(size))) {
		fprintf(stderr, "ERROR: Failed to allocate memory.\n");
		
		exit(126);
	}
	memset(pointer, 0, size);
		
	return pointer;
}

void* microfeed_memory_allocate_impl(size_t size, const char* name) {
	void* pointer;
	
	pointer = microfeed_memory_allocate_bytes(size);
	if (microfeed_debug_memory) {
		printf("%p ALLOC   %s (%ld)\n", pointer, name, (long)size);
	}
	
	return pointer;
}

/* This is not working and not used...
void* microfeed_memory_allocate_with_strings_impl(size_t size, ...) {
	void* pointer;
	va_list argp;
	char* string;
	int count, i;
	size_t length;
	size_t reserved;
	
	pointer = microfeed_memory_allocate_bytes(size);

	va_start(argp, size);
	reserved = 0;
	for (count = 0; (string = va_arg(argp, char*)); count++) {
		length = strlen(string) + 1;
		((char**)pointer)[count] = ((char*)pointer) + size + reserved;
		reserved += length;
	}
	va_end(argp);

	if (!(pointer = realloc(pointer, size + reserved))) {
		fprintf(stderr, "ERROR: Failed to allocate memory!\n");
		
		exit(126);
	}
	
	va_start(argp, size);
	reserved = 0;
	for (i = 0; i < count - 1; i++) {
		string = va_arg(argp, char*);
		memcpy(((char**)pointer)[i], string, ((char**)pointer)[i + 1] - ((char**)pointer)[i]);
	}
	string = va_arg(argp, char*);
	memcpy(((char**)pointer)[i], string, length);
	va_end(argp);

	return pointer;
}
*/

void microfeed_memory_free(void* pointer) {
	if (microfeed_debug_memory) {
		printf("%p FREE\n", pointer);
	}

	free(pointer);
}

char* microfeed_util_string_concatenate(const char* s, ...) {
	char* result;
	size_t size;
	size_t length;
	va_list argp;

	size = strlen(s);
	length = size + 1;
	result = (char*)malloc(length);
	memcpy(result, s, length);
	va_start(argp, s);
	while ((s = va_arg(argp, const char*))) {
		length = strlen(s);
		result = realloc(result, size + length + 1);
		memcpy(result + size, s, length + 1);
		size += length;
	}
	va_end(argp);

	return result;	
}

char* microfeed_util_string_append(char* s, ...) {
	char* result;
	size_t size;
	size_t length;
	va_list argp;

	size = strlen(s);
	length = size + 1;
	result = (char*)malloc(length);
	memcpy(result, s, length);
	va_start(argp, s);
	while ((s = va_arg(argp, char*))) {
		length = strlen(s);
		result = realloc(result, size + length + 1);
		memcpy(result + size, s, length + 1);
		size += length;
	}
	va_end(argp);
	
	free(s);

	return result;	
}

char* microfeed_util_string_base64_encode(const char* s, size_t length) {
	static const char base64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	char* result;
	size_t result_length;
	const unsigned char* from;
	const unsigned char* end;
	char* to;
		
	result_length = ((length + 2) / 3) * 4;
	result = (char*)malloc(result_length + 1);
	result[result_length] = 0;
	for (from = (const unsigned char*)s, to = result, end = from + length; from + 2 < end; from += 3, to += 4) {
		to[0] = base64[from[0] >> 2];
		to[1] = base64[((from[0] & 0x3) << 4) | ((from[1] & 0xf0) >> 4)];
		to[2] = base64[((from[1] & 0xf) << 2) | ((from[2] & 0xc0) >> 6)];
		to[3] = base64[from[2] & 0x3f];
	}
	if (from + 1 < end) {
		to[0] = base64[from[0] >> 2];
		to[1] = base64[((from[0] & 0x3) << 4) | ((from[1] & 0xf0) >> 4)];
		to[2] = base64[((from[1] & 0xf) << 2)];
		to[3] = '=';
	} else if (from < end) {
		to[0] = base64[from[0] >> 2];
		to[1] = base64[(from[0] & 0x3) << 4];
		to[2] = '=';
		to[3] = '=';		
	}
	
	return result;	
}


size_t microfeed_util_string_starts_with(const char* s, const char* prefix) {
	size_t length = 0;
	const char* p;
	
	p = prefix;
	while (*s && *p && *s == *p) {
		s++;
		p++;
	}
	if (*p == 0) {
		length = p - prefix;
	}
	
	return length;
}

static int hexadecimal_char_to_int(char c) {
	int retvalue = -1;

	if (c >= '0' && c <= '9') {
		retvalue = c - '0';
	} else if (c >= 'a' && c <= 'f') {
		retvalue = c - 'a' + 10;
	} else if (c >= 'A' && c <= 'F') {
		retvalue = c - 'A' + 10;
	}
	
	return retvalue;
}

char* microfeed_util_string_percent_encoding_escape(const char* s) {
	static const char hex[] = "0123456789ABCDEF";
	char* result;
	size_t length;
	const char* from;
	char* to;
	size_t offset;
	
	length = strlen(s) + 1;
	result = (char*)malloc(length);
	
	for (to = result, from = s; *from; from++, to++) {
		if ((*from >= 'a' && *from <= 'z') || (*from >= 'A' && *from <= 'Z') || (*from >= '0' && *from <= '9') ||
		    *from == '-' || *from == '_' || *from == '.' || *from == '~') {
			*to = *from;
		} else {
			length += 2;
			offset = to - result;
		    result = realloc(result, length);
		    to = result + offset;
		    *to++ = '%';
		    *to++ = hex[((unsigned char)*from) >> 4];
		    *to = hex[((unsigned char)*from) & 0xf];
		}
	}
	*to = 0;
	
	return result;
}

char* microfeed_util_string_percent_encoding_unescape(const char* s) {
	char* result;
	const char* from;
	char* to;
	int tempvalue;
	int value;

	result = (char*)malloc(strlen(s) + 1);
	for (to = result, from = s; *from; from++, to++) {
		if (*from == '%') {
			if ((tempvalue = hexadecimal_char_to_int(from[1])) == -1) {
				free(result);
				result = NULL;
				break;
			}
			if ((value = hexadecimal_char_to_int(from[2])) == -1) {
				free(result);
				result = NULL;
				break;
			}
			value |= tempvalue << 4;
			if ((value >= 'a' && value <= 'z') || (value >= 'A' && value <= 'Z') || (value >= '0' && value <= '9') ||
			    value == '-' || value == '_' || value == '.' || value == '~') {
				*to = value;
			} else {
				*to++ = *from++;
				*to++ = *from++;
				*to = *from;
			}
		} else {
			*to = *from;
		}
	}
	if (result) {
		*to++ = 0;
		result = (char*)realloc(result, result - to);
	}

	return result;
}

char* microfeed_util_create_unique_identifier(void) {
	struct timespec now;
	char buffer[1024];
	
	if (!(clock_gettime(CLOCK_REALTIME, &now))) {
		snprintf(buffer, 1024, "%lx.%lx", (long)now.tv_sec, now.tv_nsec);
	} else {
		buffer[0] = 0;
	}
	
	return strdup(buffer);
}

int microfeed_util_create_directory_recursively(const char* directory) {
	int retval = 0;
	char* s1;
	char* s2;
	
	
	if (mkdir(directory, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) == 0 || errno == EEXIST) {
		retval = 1;
	} if (errno == ENOENT) {
		s1 = strdup(directory);
		s2 = dirname(s1);
		if (strcmp(s2, ".") && strcmp(s2, "/")) {
			if (microfeed_util_create_directory_recursively(s2) && mkdir(directory, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) == 0) {
				retval = 1;
			}
		}
		free(s1);
	}	
	
	return retval;
}


int microfeed_util_file_copy(const char* source_path, const char* destination_path) {
	int retval = 0;
	int source;
	int destination;
	struct stat stat_buffer;
	char* buffer;
	ssize_t left;
	ssize_t wrote;
	size_t position;
	
	if ((source = open(source_path, O_RDONLY)) >= 0) {
		if ((destination = open(destination_path, O_WRONLY | O_CREAT | O_EXCL, 0666))) {
			if (fstat(destination, &stat_buffer) == 0) {
				buffer = alloca(stat_buffer.st_blksize);
				while ((left = read(source, buffer, stat_buffer.st_blksize)) > 0) {
					position = 0;
					while ((wrote = write(destination, buffer + position, left)) != left && wrote >= 0) {
						position += wrote;
						left -= wrote;
					}
					if (wrote < 0) {
						break;
					}
				}
				if (left == 0 && wrote >= 0) {
					retval = 1;
				}
			}
			close(destination);
			if (retval == 0) {
				unlink(destination_path);
			}
		}
		close(source);
	}
	
	return retval;
}

int microfeed_util_file_exists(const char* path) {
	int retval = 0;
	int fd;
	
	if ((fd = open(path, O_RDONLY)) >= 0) {
		retval = 1;
		close(fd);
	}
	
	return retval;
}

int microfeed_util_file_create(const char* path, const void* data, size_t data_size) {
	int retval = 0;
	int fd;
	ssize_t wrote;
	
	if ((fd = open(path, O_WRONLY | O_CREAT | O_TRUNC, 0666)) >= 0) {
		while ((wrote = write(fd, data, data_size)) != data_size && wrote >= 0) {
			data += wrote;
			data_size -= wrote;		
		}
		if (wrote >= 0) {
			retval = 1;
		}

		close(fd);
	}

	return retval;
}

void microfeed_assert_failed(const char* text) {
	fprintf(stderr, "\n*****\nAssertion failed: %s\n\n*****\n\n", text);
	abort();
}
