/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QFile>
#include <QtCore/QTimer>
#include "speedalarm.h"
#include "settings.h"
#include "odometer.h"
#include "poialerts.h"
#include "mediaplayer.h"


SpeedAlarm::SpeedAlarm(): QObject(0), loaded_(false),
enabled_(false), isOver_(false), timer_(0)
{
}

SpeedAlarm::~SpeedAlarm()
{
    delete timer_;
}

SpeedAlarm& SpeedAlarm::instance()
{
    static SpeedAlarm instance;
    return instance;
}

void SpeedAlarm::start()
{
    if(!loaded_)
    {
        loadConfig();
    }

    if(enabled_)
    {
        connect(&(Odometer::instance()), SIGNAL(dataUpdated()), this, SLOT(onDataUpdated()));
    }

}

void SpeedAlarm::loadConfig()
{
    loaded_ = true;

    bool enabled = Settings::instance().value("alarm_enabled", false).toBool();

    if(enabled)
    {
        enabled_ = true;

        QString sound = Settings::instance().value("alarm_sound", "").toString();

        if(sound.isEmpty())
        {
            enabled_ = false;
            return;
        }

        QString soundDir = MediaPlayer::getSoundDir();

        if(QFile::exists(soundDir + sound))
        {
            soundFile_ = soundDir + sound;
        }
        else if(QFile::exists(MediaPlayer::getLocalSoundDir() + sound))
        {
            soundFile_ = MediaPlayer::getLocalSoundDir() + sound;
        }
        else
        {
            enabled_ = false;
            return;
        }

        threshold_ = Settings::instance().value("alarm_threshold", 0).toDouble();

        start();
    }
    else
    {
        end();
        enabled_ = false;
    }
}

void SpeedAlarm::end()
{
    if(enabled_)
    {
        disconnect(&(Odometer::instance()), SIGNAL(dataUpdated()), this, SLOT(onDataUpdated()));
    }
}

void SpeedAlarm::onDataUpdated()
{
    if(Odometer::instance().getLatestFix().kmSpeed > threshold_)
    {
        if(!isOver_)
        {
            MediaPlayer::play(soundFile_);
            isOver_ = true;
            emit speedExceedChanged(true);
            startAlarmTimer();
        }
    }
    else
    {
        if(isOver_)
        {
            emit speedExceedChanged(false);
        }

        isOver_ = false;
    }
}

void SpeedAlarm::startAlarmTimer()
{
    if(timer_)
    {
        timer_->stop();
        delete timer_;
        timer_ = 0;
    }

    timer_ = new QTimer;
    timer_->setInterval(ALARM_INTERVAL * 1000);
    connect(timer_, SIGNAL(timeout()), this, SLOT(playAlarmSound()));
    timer_->start();
}

void SpeedAlarm::playAlarmSound()
{
    if(isOver_)
    {
        MediaPlayer::play(soundFile_);
    }
    else
    {
        delete timer_;
        timer_ = 0;
    }
}
