/**midi_editor.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is for clutter based gui.
This is the midi editor.
*/

#include <tangle.h>
#include <glib-object.h>
#include <clutter/clutter.h>
#include "midi_editor.h"
#include "midi_editor_notes.h"
#include "../chum/chum.h"

#include "jammo-texture.h"

#include "config.h"

#include "../meam/jammo-meam.h"
#include "../meam/jammo-editing-track.h"
#include "../meam/jammo-instrument-track.h"

/*
 * TODO LIST:
 *
 * ZOOM VIEW - Hide letters away from the piano roll
 * GRID - Disable note stacking when created
 * GRID - Disable note stacking when moved
 * GRID - Extend notes if they are added next to each other
 * BACKGROUND - Add background image
 * BUTTONS - Add correct button images
 * NOTES - when one note is activated, draw yellow border to surround it - remember to disable if deactivated
 * NOTES - make arrow on top, to tell where next note is positioned outside of this view
 * NOTES - arrow image to this same "finder"
 * SONG - Making complete song
 * SONG - Playing and stopping song
 * SONG - bringing complete song inside this editor
 *
 */

ClutterColor complete_white = { 255,255,255,255 };
ClutterColor trans_color = { 0, 0, 0, 0x00 };
ClutterColor dark_color =  { 0, 0, 0, 0xdd };
ClutterColor white_color = { 0, 0, 0, 0xbb };
ClutterColor blue_color =  { 0, 0, 0xff, 0xff };

ClutterColor sing_green_color =  { 158, 219, 116, 0xff }; /*This is for background of singing game*/

ClutterColor light_blue_color = { 134, 175, 231, 255 };

ClutterColor black_color = { 0, 0, 0, 0xff };
ClutterColor green_color = { 0, 0xff, 0, 0xff };


ClutterColor *get_transparency_color(){ return &trans_color; }
ClutterColor *get_white_color(){        return &white_color; }
ClutterColor *get_dark_color(){         return &dark_color;  }
ClutterColor *get_blue_color(){         return &blue_color;  }
ClutterColor *get_sing_green_color(){         return &sing_green_color;  }
ClutterColor *get_black_color(){		return &black_color; }
ClutterColor *get_green_color(){		return &green_color; }
ClutterColor *get_light_blue_color(){	return &light_blue_color; }
ClutterColor *get_complete_white_color(){	return &complete_white; }

// >> dragging grid attributes

gboolean grid_is_dragging;
gboolean zoomed_out;
gfloat x_mouse_position_f, y_mouse_position_f;
gint x_start_of_dragging, y_start_of_dragging, x_drag_offset, y_drag_offset;

// >> dragging note attributes

gboolean note_is_dragging;
gfloat x_mouse_position_f_note, y_mouse_position_f_note, x_start_position_note, y_start_position_note;
gint x_start_of_dragging_note, y_start_of_dragging_note, x_drag_offset_note, y_drag_offset_note;

//Grid and note related parameters
//NOTE: NOT FINAL

const int GRID_NOTE_SIZE = 64;
const int GRID_NUMBER_OF_NOTES = 24;
const int GRID_SIZE = 24*(64);
const int GRID_STARTPOS_X = 100;
const int GRID_STARTPOS_Y = 0;
const double GRID_SCALE_OUT = 1.0;
const double GRID_SCALE_IN = 0.25;

// this indicates how our notes is positioned
char NOTE_TABLE[] = { 'B', 'a', 'A', 'g', 'G', 'f', 'F', 'E', 'd', 'D', 'c', 'C' };

// these are meant only for printing
char NOTE_TABLE_BIG_LETTERS[] = { 'B', 'A', 'G', 'F', 'E', 'D', 'C', '\0' };
char NOTE_TABLE_SMALL_LETTERS[] = { 'a', 'g', 'f', 'd', 'c', '\0'};

// Parameters related to tempo and time :: TODO

//const int TEMPO_TEENAGERS = x
//const int TEMPO_CHILDS = x
//const int TEMPO_BABIES = x

const int RANGE_OF_OCTAVE = 12;
const double NOTE_DURATION_IN_SECONDS = 0.25; // need to be calculated later according to the correct tempo?

gfloat GRID_POSITION_X, GRID_POSITION_Y;

// menu parameters

const int MENU_STARTPOS_X = 0;
const int MENU_STARTPOS_Y = 384;

// to keep track where users finger is

gfloat FINGER_POS_X;
gfloat FINGER_POS_Y;

// to keep track which note have been clicked last time.. x and y (corner..)

gfloat LAST_NOTE_X, LAST_NOTE_Y;


//Pointer to pianoLeft - note name texts
ClutterActor* pianoTexts;
//Pointer to piano key group, needed for synchronization with grid_motion
ClutterActor* pianoLeft;
//Pointer to grid..
ClutterActor* grid;
//Pointer to bottom menu
ClutterActor* menuBottom;
//Pointer to trash button
ClutterActor* button_trash;
//Pointer to note ( we need this to select note )
ClutterActor* tempNotePointer;
ClutterActor* button_pencil;
ClutterActor* pointToAnotherNote;

NoteArray* note_array_test;

/*
 * actionType attribute holds value which tells us, what kind of button we have selected and what actions we can do with it?
 *
 * Example: 0 - Normal action
 * 			1 - Pencil action
 * 			2 - etc..
 */

gint actionType=0;

// not sure if we need these.. but maybe if we add some more functionality!
void setActionType(gint i){
	actionType=i;
}

gint getActionType(){
	return actionType;
}


//Scales the grid and piano roll to emulate zoom
void zoom_pressed(ClutterActor *zoom_button, ClutterEvent *event, gpointer data){

	gdouble scale_x, scale_y;
	clutter_actor_get_scale(grid, &scale_x, &scale_y);

	if (scale_x!=GRID_SCALE_IN) 	// zooming out
	{
		clutter_actor_hide(pianoTexts);
		clutter_actor_set_scale(grid, GRID_SCALE_IN, GRID_SCALE_IN);
		clutter_actor_set_scale(pianoLeft, GRID_SCALE_OUT, GRID_SCALE_IN);

		zoomed_out = TRUE;

		clutter_actor_set_opacity(button_trash, 128);
		clutter_actor_set_opacity(button_pencil, 128);

		if ( getActionType())
			setActionType(0);

		clutter_actor_set_position (grid, GRID_STARTPOS_X, GRID_STARTPOS_Y);
		clutter_actor_set_position (pianoLeft, 0, 0);
	}
	else // zooming in
	{
		clutter_actor_show(pianoTexts);
		clutter_actor_set_scale(grid, GRID_SCALE_OUT, GRID_SCALE_OUT);
		clutter_actor_set_scale(pianoLeft, GRID_SCALE_OUT, GRID_SCALE_OUT);
		zoomed_out = FALSE;
		clutter_actor_set_opacity(button_trash, 255);
		clutter_actor_set_opacity(button_pencil, 255);
	}

}

//Returns the view to the start of the grid
void finder_pressed(ClutterActor *finder, ClutterEvent *event, gpointer data) {

	clutter_actor_set_position (grid, GRID_STARTPOS_X, GRID_STARTPOS_Y);
	clutter_actor_set_position (pianoLeft, 0, 0);

}

/*
 * This function calls pencil, and we allow user to make new notes.
 */

void pencil_pressed(ClutterActor *finder, ClutterEvent *event, gpointer data) {

	//printf("\n PENCIL CALLED!!!");

	if ( getActionType())
		setActionType(0);

	else
		if(!zoomed_out)
			setActionType(1);

}

/*
 * This function snaps note to grid
 */

void snapNoteToGrid(gint *snapX, gint *snapY) {

	gfloat calculateHelperX, calculateHelperY;

	GRID_POSITION_X = clutter_actor_get_x(CLUTTER_ACTOR(grid));
	GRID_POSITION_Y = clutter_actor_get_y(CLUTTER_ACTOR(grid));

	//printf("\n\nGRID_POSITION: x %f ja y %f", GRID_POSITION_X,GRID_POSITION_Y);
	//printf("\n\nFINGER_POSITION: x %f ja y %f", FINGER_POS_X,FINGER_POS_Y);

	calculateHelperX = (FINGER_POS_X-GRID_POSITION_X)/GRID_NOTE_SIZE;
	calculateHelperY = (FINGER_POS_Y-GRID_POSITION_Y)/GRID_NOTE_SIZE;

	//printf("\n\nCALCULATE_HELPER: x %f ja y %f", calculateHelperX,calculateHelperY);

	gint xCoord, yCoord;

	xCoord = (gint)calculateHelperX;
	yCoord = (gint)calculateHelperY;

	// printf("\n\nCOORD 1: x %d ja y %d", xCoord,yCoord);

	 xCoord = xCoord * GRID_NOTE_SIZE;
	 yCoord = yCoord * GRID_NOTE_SIZE;

	 // printf("\n\nCOORD 2: x %d ja y %d", xCoord,yCoord);

	 *snapX = xCoord;
	 *snapY = yCoord;

}

/*
 * This functions makes sure that we always save last spot where we have been clicked
 */

void setFingerPos(ClutterEvent *event) {
	clutter_event_get_coords (event, &FINGER_POS_X, &FINGER_POS_Y);
}


/*
 * This function deletes note, it is called when note is dragged to trash
 */

void deleteNote(ClutterActor *actor) {

	clutter_actor_destroy (actor);
	printf("\nnote deleted");

}


void note_pressed(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	clutter_actor_get_position(actor, &LAST_NOTE_X, &LAST_NOTE_Y);
	clutter_event_get_coords (event, &x_start_position_note, &y_start_position_note);

	if (clutter_actor_get_opacity(actor)==128){

		gfloat x_start_of_dragging_f_note,y_start_of_dragging_f_note;
	    clutter_event_get_coords (event, &x_start_of_dragging_f_note, &y_start_of_dragging_f_note);

	    x_start_of_dragging_note=(gint)x_start_of_dragging_f_note;
	    y_start_of_dragging_note=(gint)y_start_of_dragging_f_note;

	    note_is_dragging=TRUE;
	    grid_is_dragging=FALSE;

	    x_drag_offset_note = (clutter_actor_get_x (CLUTTER_ACTOR(actor)) - (int)x_start_of_dragging_note);
	    y_drag_offset_note = (clutter_actor_get_y (CLUTTER_ACTOR(actor)) - (int)y_start_of_dragging_note);

	    //even this object is not focused it still gets the mouse events
	    clutter_grab_pointer(CLUTTER_ACTOR(actor));

	}
	else {
			clutter_event_get_coords (event,&x_start_position_note, &y_start_position_note);
			tempNotePointer = actor;
			grid_is_dragging=TRUE;

			clutter_grab_pointer(CLUTTER_ACTOR(grid));
	}
}

void note_motion(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	 if (CLUTTER_IS_ACTOR(actor) && note_is_dragging ) {
	      //printf("drag\n");
	      clutter_event_get_coords (event,&x_mouse_position_f_note, &y_mouse_position_f_note);
	      //printf("X=%f, Y=%f\n",x_mouse_position, y_mouse_position);
	      gint new_x_note, new_y_note;
	      gint x_mouse_position_note=(gint)x_mouse_position_f_note;
	      gint y_mouse_position_note=(gint)y_mouse_position_f_note;

	      new_x_note=x_mouse_position_note + x_drag_offset_note;
	      new_y_note=y_mouse_position_note + y_drag_offset_note;

	      clutter_actor_set_position (CLUTTER_ACTOR(actor), new_x_note, new_y_note);

	 }

	 //else if( grid_is_dragging ){
	//	  grid_motion(actor, event, data);
	 //}

}


void note_released(ClutterActor *actor, ClutterEvent *event, gpointer data) {

	setFingerPos(event);

	gfloat trash_x, trash_y;

	// TODO: make here "function" which takes note from this position where we are dragging the another note..
	// we need this to make sure that we wont stack notes!

	clutter_actor_get_position(button_trash, &trash_x, &trash_y );

	gfloat trash_right_x = trash_x + clutter_actor_get_width(button_trash);
	gfloat trash_top_y = trash_y + clutter_actor_get_height(button_trash);

	if (note_is_dragging) {

		clutter_actor_set_opacity(actor,255);

		if( FINGER_POS_X >= trash_x && FINGER_POS_X <= trash_right_x && (480-FINGER_POS_Y) >= trash_y && (480-FINGER_POS_Y) <= trash_top_y ){
			deleteNote(actor);
		}
		// if the place is on the piano or black area >>> TODO : OR there is another note at the same position.. swap note back to start position
		else if (FINGER_POS_X <= GRID_STARTPOS_X || FINGER_POS_Y >= MENU_STARTPOS_Y)
		{
			clutter_actor_set_position (CLUTTER_ACTOR(actor), LAST_NOTE_X, LAST_NOTE_Y);
		}
		else
		{

			 gint finalNotePosX, finalNotePosY;
			 snapNoteToGrid(&finalNotePosX, &finalNotePosY);

			 clutter_actor_set_position(CLUTTER_ACTOR(actor), finalNotePosX, finalNotePosY);
		}

	}

	tempNotePointer = NULL;
	clutter_ungrab_pointer(); //ungrab
	note_is_dragging = FALSE;
	grid_is_dragging = FALSE;

}

void play_note(){
	printf("play_note pressed\n");

	JammoSequencer* sequencer = jammo_sequencer_new();
	JammoInstrumentTrack* instrument_track = jammo_instrument_track_new(1); //0= flute, 1=drumkit ,2=ud
	jammo_instrument_track_set_realtime(instrument_track,TRUE);
	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(instrument_track));

	//Instrument track doesn't work if there aren't another track with one sample.
	JammoEditingTrack* editing_track = jammo_editing_track_new();
	JammoSample* sample;
	gchar* filepath1 = g_strdup_printf("%s/themes/forest/loop1.wav",DATA_DIR);
	sample = jammo_sample_new_from_file(filepath1);
	jammo_editing_track_add_sample(editing_track, sample, 0);
	jammo_sequencer_add_track(sequencer, JAMMO_TRACK(editing_track));

	jammo_sequencer_play(sequencer);

	jammo_instrument_track_set_and_play_note_realtime(instrument_track,'e',1);
}

/*
 * This function will create note and snap it to grid
 */

void createNote(void) {

	// setFingerPos(event);

	 ClutterActor* note = clutter_rectangle_new_with_color(get_blue_color());
	 clutter_rectangle_set_border_color(CLUTTER_RECTANGLE(note),get_white_color());
	 clutter_rectangle_set_border_width(CLUTTER_RECTANGLE(note), 1);
	 clutter_actor_set_size(note, GRID_NOTE_SIZE, GRID_NOTE_SIZE);

	 gint finalNotePosX, finalNotePosY;
	 snapNoteToGrid(&finalNotePosX, &finalNotePosY);

	 clutter_actor_set_position(CLUTTER_ACTOR(note), finalNotePosX, finalNotePosY);
	 clutter_actor_show(note);
	 clutter_container_add_actor(CLUTTER_CONTAINER(grid), note);
	 clutter_actor_set_reactive(note, TRUE);

	// printf("\nYou created note to position: %d %d", finalNotePosX, finalNotePosY);
	// printf("Another test: %d %d\n", finalNotePosX/GRID_NOTE_SIZE, finalNotePosY/GRID_NOTE_SIZE);

	 double noteStartTime, noteEndTime;
	 double noteOctaveCalculator = 0.0;
	 int noteOctave=0;
	 char noteName;

	 noteOctave = finalNotePosY/GRID_NOTE_SIZE;
	 noteName = NOTE_TABLE[noteOctave];
	 noteOctaveCalculator = noteOctave / RANGE_OF_OCTAVE;
	 noteOctave = (gint)noteOctaveCalculator;

	 if (noteOctave == 0)
		 noteOctave = 1;

	 else
		 noteOctave++;

	 noteEndTime = NOTE_DURATION_IN_SECONDS * ((finalNotePosX/GRID_NOTE_SIZE)+1);
	 noteStartTime = noteEndTime - NOTE_DURATION_IN_SECONDS;

	 add_note_to_collection(note_array_test, new_note(noteName, noteOctave, noteStartTime, noteEndTime));
	 note_collection_debug_print(note_array_test);

	 g_signal_connect (note, "button-press-event", G_CALLBACK (note_pressed),NULL);
	 g_signal_connect (note, "button-release-event", G_CALLBACK (note_released),NULL);
	 g_signal_connect (note, "motion-event", G_CALLBACK (note_motion), NULL);

}


void grid_motion(ClutterActor *actor, ClutterEvent *event, gpointer data) {

  if (CLUTTER_IS_ACTOR(actor) && grid_is_dragging) {
	 // clutter_event_get_coords (event, &FINGER_POS_X, &FINGER_POS_Y);
      //printf("drag\n");
      clutter_event_get_coords (event,&x_mouse_position_f, &y_mouse_position_f);
      //printf("X=%f, Y=%f\n",x_mouse_position, y_mouse_position);
      gint new_x, new_y;
      gint x_mouse_position=(gint)x_mouse_position_f;
      gint y_mouse_position=(gint)y_mouse_position_f;

      new_x=x_mouse_position + x_drag_offset;
      new_y=y_mouse_position + y_drag_offset;

      //Debug print for snaps
      //gfloat gridPosY = clutter_actor_get_y(actor);
      //printf("%f\n", gridPosY);

      //Preliminary grid boundary point calculation
      //NOTE: WILL CHANGE AFTER FINAL NOTE SIZE SPECS
      int lowest_point_y = -GRID_SIZE+(6*GRID_NOTE_SIZE);
      int highest_point_y = 0;
      gfloat piano_x = clutter_actor_get_x(pianoLeft);

      //Restrict grid's movement in zoomed out view to horizontal only, restrict once piano roll is reached
      if(zoomed_out){
    	  if(new_x >= 100){
    		  clutter_actor_set_position(CLUTTER_ACTOR(actor), GRID_STARTPOS_X, GRID_STARTPOS_Y);
    	  }
    	  else{
    		  clutter_actor_set_position (CLUTTER_ACTOR(actor), new_x, GRID_STARTPOS_Y);
    		  clutter_actor_set_position (CLUTTER_ACTOR(pianoLeft), 0, 0);
		  }
      }
      //Restrict grid movement on the y-axis by setting up bottom and top limiters
      else {
    	  if(new_y >= lowest_point_y && new_y <= highest_point_y) {
    		  clutter_actor_set_position (CLUTTER_ACTOR(actor), new_x, new_y);
    		  clutter_actor_set_position (CLUTTER_ACTOR(pianoLeft), piano_x, new_y);
    	  }
    	  //If past the highest point (the top of the grid)
    	  else if(new_y >= highest_point_y) {
    		  clutter_actor_set_position(CLUTTER_ACTOR(actor), new_x, highest_point_y);
    		  clutter_actor_set_position(CLUTTER_ACTOR(pianoLeft), piano_x, highest_point_y);
    	  }
    	  //If past the lowest point (the bottom of the grid)
    	  else {

    		  clutter_actor_set_position (CLUTTER_ACTOR(actor), new_x, lowest_point_y);
    		  clutter_actor_set_position (CLUTTER_ACTOR(pianoLeft), piano_x, lowest_point_y);

    	  }

    	  //Restrict grid's left to right movement from moving past the piano
    	  //Restrict X if past piano and Y past the highest point
    	  if(new_x >= 100 && new_y >= highest_point_y ) {
    		  clutter_actor_set_position(CLUTTER_ACTOR(actor), 100, highest_point_y);
    	  }
    	  //Restrict X if past piano
    	  else if(new_x >= 100 && new_y >= lowest_point_y && new_y <= highest_point_y){
    		  clutter_actor_set_position(CLUTTER_ACTOR(actor), 100, new_y);
    	  }
    	  //Restrict X if past piano and Y past the lowest point
    	  else if(new_x >= 100 && new_y <= lowest_point_y ){
    		  clutter_actor_set_position(CLUTTER_ACTOR(actor), 100, lowest_point_y);
    	  }
      }
  }
}


void grid_pressed (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	setFingerPos(event);

	if (getActionType()) {
		createNote();
	}

	else {

	  gfloat x_start_of_dragging_f,y_start_of_dragging_f;
	  clutter_event_get_coords (event, &x_start_of_dragging_f, &y_start_of_dragging_f);

	  x_start_of_dragging=(gint)x_start_of_dragging_f;
	  y_start_of_dragging=(gint)y_start_of_dragging_f;

	  grid_is_dragging=TRUE;

	  x_drag_offset = (clutter_actor_get_x (CLUTTER_ACTOR(actor)) - (int)x_start_of_dragging);
	  y_drag_offset = (clutter_actor_get_y (CLUTTER_ACTOR(actor)) - (int)y_start_of_dragging);

	}

}


void grid_released (ClutterActor *actor, ClutterEvent *event, gpointer data) {

	setFingerPos(event);

	// NOTE!! make here some sort of better check for finger "touching"..
	// this cant be done without nokia n900..?

	if (!zoomed_out && FINGER_POS_X==x_start_position_note && FINGER_POS_Y==y_start_position_note && tempNotePointer!=NULL){

		if (clutter_actor_get_opacity(tempNotePointer)==255) {
			clutter_actor_set_opacity(tempNotePointer,128);

			if (pointToAnotherNote != tempNotePointer && pointToAnotherNote != NULL) {
			clutter_actor_set_opacity(pointToAnotherNote,255);
			}

		pointToAnotherNote = tempNotePointer;

		}

	}

	clutter_ungrab_pointer(); //ungrab
	tempNotePointer=NULL;
	note_is_dragging = FALSE;
	grid_is_dragging = FALSE;

}

/*
 * This is function which can draw different texts in midi editor
 */

void drawText(char *myText, int x, int y, ClutterActor* myContainer, ClutterColor* myColor){

	  /* Add a non-editable text actor to the stage: */
	  ClutterActor *text = clutter_text_new ();

	  /* Setup text properties */
	  clutter_text_set_color (CLUTTER_TEXT (text), myColor);
	  clutter_text_set_text (CLUTTER_TEXT (text), myText);
	  clutter_text_set_font_name (CLUTTER_TEXT (text), "Sans 12");
	  clutter_text_set_editable (CLUTTER_TEXT (text), FALSE);
	  clutter_text_set_line_wrap (CLUTTER_TEXT (text), FALSE);

	  clutter_actor_set_position (text, x, y);
	  clutter_container_add_actor (CLUTTER_CONTAINER (myContainer), text);
	  clutter_actor_show (text);

}

/*
 * This method will draw piano roll to the left, including black/white keys & letters of the current notes
 */

void drawPianoRoll(ClutterActor* stage) {

	   //Placeholder for piano key group
		pianoLeft=clutter_group_new();
		clutter_actor_set_size(pianoLeft,100,480);
		clutter_actor_set_position(pianoLeft,0, 0);
		clutter_actor_set_reactive(pianoLeft, TRUE);
		clutter_actor_show(pianoLeft);
		clutter_container_add_actor(CLUTTER_CONTAINER(stage), pianoLeft);

		//Placeholder for note names
		pianoTexts=clutter_group_new();
		clutter_actor_set_size(pianoTexts,100,480);
		clutter_actor_set_position(pianoTexts,0, 0);
		clutter_actor_set_reactive(pianoTexts, FALSE);
		clutter_actor_show(pianoTexts);
		clutter_container_add_actor(CLUTTER_CONTAINER(pianoLeft), pianoTexts);

/*
		  for(white_pos = 0; white_pos < GRID_NUMBER_OF_NOTES; white_pos++) { // white keys

		  ClutterActor *box = clutter_rectangle_new_with_color(GINT_TO_POINTER(get_blue_color()));
		  clutter_actor_set_size(box, 100, GRID_NOTE_SIZE);
		  clutter_actor_set_position(box, 0, white_pos*(GRID_NOTE_SIZE));
		  clutter_rectangle_set_border_color(CLUTTER_RECTANGLE(box), GINT_TO_POINTER(get_black_color()));
		  clutter_rectangle_set_border_width(CLUTTER_RECTANGLE(box), 1);
		  clutter_actor_set_reactive(box, FALSE);
		  clutter_actor_show(box);
		  clutter_container_add_actor(CLUTTER_CONTAINER(pianoLeft), box);

			  if (letter_pos==7)
				  letter_pos=0;

		  gchar *j = NOTE_TABLE_BIG_LETTERS[letter_pos];
		  drawText(&j, 80, white_pos*(GRID_NOTE_SIZE)+20, pianoLeft, get_black_color());

		  letter_pos++;

		  }
*/

	  int keyLooper;
	  int letterPosWhite=0;
	  int letterPosBlack=0;
	  int calculateHelper=0;
	  int extraGap=0;
	  int extraGapSecond=0;
	  int timeToMakeGap=4;
	  int loop=1;
	  int z=0;
	  int blackKeyFinalPos=0;
	  gboolean changeGapMode=FALSE;

	 // for (black_pos = 1; black_pos < 18; black_pos++){

		  // This is the main loop, inside this we draw black & white keys to the piano roll
		  // also we draw letters on them

		  for (keyLooper = 1; keyLooper < GRID_NUMBER_OF_NOTES; keyLooper++) { // black keys

		  calculateHelper++;

		  if (calculateHelper >1) {

			  extraGap += 64;

			  if (calculateHelper == timeToMakeGap) { // if there is going to be a gap, we make it
				extraGap += 64; // increasing the gap for black keys
				extraGapSecond = 64; // length of the gap for white key
				loop = 2; // value to stack two white keys
				//extraGap += 128;
				calculateHelper = 1;

				if (changeGapMode == FALSE) { // These are only meant for black keys

					timeToMakeGap = 3;
					changeGapMode = TRUE;
				}

				else { // These are only meant for black keys

					timeToMakeGap = 4;
					changeGapMode = FALSE;
				}

			  } else { // we are here if there is no need for stacking two white keys
				  loop = 1;
				  extraGapSecond = 0;
			  }

		  }

		  // we need this value to calculate white key position
		  blackKeyFinalPos = keyLooper*(GRID_NOTE_SIZE)+extraGap;

			  ClutterActor *black_key = clutter_rectangle_new_with_color(GINT_TO_POINTER(get_black_color()));
			  //clutter_rectangle_set_border_color(CLUTTER_RECTANGLE(black_key),get_blue_color());
			  //clutter_rectangle_set_border_width(CLUTTER_RECTANGLE(black_key), 1);
			  //clutter_actor_set_size(black_key, 60, GRID_NOTE_SIZE/2);
			  clutter_actor_set_size(black_key, 100, GRID_NOTE_SIZE);
			  //clutter_actor_set_position(black_key, 0, 48*black_pos+extraGap);
			  clutter_actor_set_position(black_key, 0, blackKeyFinalPos);
			  clutter_actor_set_opacity(black_key,255);
			  clutter_actor_set_reactive(black_key, FALSE);
			  clutter_actor_show(black_key);
			  clutter_container_add_actor(CLUTTER_CONTAINER(pianoLeft), black_key);

			  // If we are in the end of array we jump back to start
			  if (letterPosWhite==5)
			 	 letterPosWhite=0;

					  // We draw small letter on black keys here!!
			 		  gchar *h = NOTE_TABLE_SMALL_LETTERS[letterPosWhite];
			 		  drawText(&h, 80, keyLooper*(GRID_NOTE_SIZE)+20+extraGap, pianoTexts, get_complete_white_color());
			 		  // drawText(&h, 40, 48*black_pos+extraGap+8, pianoLeft, get_blue_color());

			 		  // We need this for loop to make two white keys at the right time
			 		  for(z = 0; z<loop; z++) {
					  ClutterActor *box = clutter_rectangle_new_with_color(GINT_TO_POINTER(get_complete_white_color()));
					  clutter_actor_set_size(box, 100, GRID_NOTE_SIZE);
					  clutter_actor_set_position(box, 0, (blackKeyFinalPos-GRID_NOTE_SIZE)-extraGapSecond);
					  clutter_rectangle_set_border_color(CLUTTER_RECTANGLE(box), GINT_TO_POINTER(get_black_color()));
					  clutter_rectangle_set_border_width(CLUTTER_RECTANGLE(box), 1);
					  clutter_actor_set_reactive(box, FALSE);
					  clutter_actor_show(box);
					  clutter_container_add_actor(CLUTTER_CONTAINER(pianoLeft), box);

					  // We draw big letter on this white key here
					  gchar *j = NOTE_TABLE_BIG_LETTERS[letterPosBlack];
					  drawText(&j, 80, (blackKeyFinalPos-GRID_NOTE_SIZE)-extraGapSecond+20, pianoTexts, get_black_color());

					  // already made one key? - so lets set gap to 0
					  	 if (extraGapSecond != 0)
					  	     extraGapSecond = 0;

					  // We increase value, so we can choose next letter in the array (NOTE_TABLE_BIG_LETTERS)
					  	 letterPosBlack++;

				 	  // If we are in the end of array we jump back to start
				 		 if (letterPosBlack==7)
				 			 letterPosBlack=0;

			 		  }

			 		// We increase value, so we can choose next letter in the array (NOTE_TABLE_SMALL_LETTERS)
			 		letterPosWhite++;

	  }

		  clutter_actor_raise_top(pianoTexts);

}





void drawMenuBar(ClutterActor* stage) {

	  //Menu bar group
	  //NOTE: Might be a good idea to separate into it's own function
	  menuBottom=clutter_group_new();
	  clutter_actor_set_size(menuBottom,800,96);
	  clutter_actor_set_position(menuBottom, MENU_STARTPOS_X, MENU_STARTPOS_Y);
	  clutter_actor_set_reactive(menuBottom, TRUE);
	  clutter_actor_show(menuBottom);
	  clutter_container_add_actor(CLUTTER_CONTAINER(stage), menuBottom);

	  //Background for the menu bar
	  ClutterActor *menu_background = clutter_rectangle_new_with_color(GINT_TO_POINTER(get_light_blue_color()));
	  clutter_actor_set_size(menu_background, 800, 96);
	  clutter_actor_set_position(menu_background, 0, 0);
	  clutter_actor_show(menu_background);
	  clutter_container_add_actor(CLUTTER_CONTAINER(menuBottom), menu_background);

	  //NOTE: textures are just placeholders
	  ClutterActor *button_playstop = clutter_texture_new_from_file("/opt/jammo/contour_star.png", NULL); 
	  clutter_actor_set_position(button_playstop,25,20);
	  clutter_actor_set_size(button_playstop, 50, 50);
	  clutter_container_add_actor(CLUTTER_CONTAINER(menuBottom), button_playstop);
	  clutter_actor_show(button_playstop);
	  clutter_actor_set_reactive(button_playstop, TRUE);
	  g_signal_connect (button_playstop, "button-press-event", G_CALLBACK (play_note),NULL);

	  button_trash = clutter_texture_new_from_file("/opt/jammo/contour_star.png", NULL); 
	  clutter_actor_set_position(button_trash, 125, 20);
	  clutter_actor_set_size(button_trash, 50, 50);
	  clutter_container_add_actor(CLUTTER_CONTAINER(menuBottom), button_trash);
	  clutter_actor_show(button_trash);
	  clutter_actor_set_reactive(button_trash, TRUE);
	  //g_signal_connect (buttonTrash, "button-press-event", G_CALLBACK (menuButton_pressed),NULL);
	  //we wont need signal here, because if we want to delete something -> we drag and drop it..

	  ClutterActor *button_finder = clutter_texture_new_from_file("/opt/jammo/contour_star.png", NULL); 
	  clutter_actor_set_position(button_finder, 225,20);
	  clutter_actor_set_size(button_finder, 50, 50);
	  clutter_container_add_actor(CLUTTER_CONTAINER(menuBottom), button_finder);
	  clutter_actor_show(button_finder);
	  clutter_actor_set_reactive(button_finder, TRUE);
	  g_signal_connect (button_finder, "button-press-event", G_CALLBACK (finder_pressed),NULL);

	  button_pencil = clutter_texture_new_from_file("/opt/jammo/contour_star.png", NULL); 
	  clutter_actor_set_position(button_pencil, 325,20);
	  clutter_actor_set_size(button_pencil, 50, 50);
	  clutter_container_add_actor(CLUTTER_CONTAINER(menuBottom), button_pencil);
	  clutter_actor_show(button_pencil);
	  clutter_actor_set_reactive(button_pencil, TRUE);
	  g_signal_connect (button_pencil, "button-press-event", G_CALLBACK (pencil_pressed),NULL);

	  ClutterActor *button_zoom = clutter_texture_new_from_file("/opt/jammo/contour_star.png", NULL); 
	  clutter_actor_set_position(button_zoom,425,20);
	  clutter_actor_set_size(button_zoom, 50, 50);
	  clutter_container_add_actor(CLUTTER_CONTAINER(menuBottom), button_zoom);
	  clutter_actor_show(button_zoom);
	  clutter_actor_set_reactive(button_zoom, TRUE);
	  g_signal_connect (button_zoom, "button-press-event", G_CALLBACK (zoom_pressed),NULL);

}

void drawGrid(ClutterActor* stage) {

	  //Set up a grid
	  grid = clutter_group_new();
	  clutter_container_add_actor (CLUTTER_CONTAINER (stage), grid);
	  clutter_actor_show (grid);
	  clutter_actor_set_reactive(grid, TRUE);
	  clutter_actor_set_size (grid, GRID_SIZE, GRID_SIZE);
	  clutter_actor_set_position (grid, GRID_STARTPOS_X, GRID_STARTPOS_Y);

	  g_signal_connect (grid, "button-release-event", G_CALLBACK (grid_released), NULL);
	  g_signal_connect (grid, "motion-event", G_CALLBACK (grid_motion), NULL);
	  g_signal_connect (grid, "button-press-event", G_CALLBACK (grid_pressed),NULL);

	  //Background for the note screen.. makes easier to follow notes and stuff behavior..
	  //TODO: change to picture.. example: // grid = config_get_configured_actor(NULL,"house"); //just some image

	  ClutterColor* white  = clutter_color_new(255,255,255,   255);
	  ClutterActor *note_background = clutter_rectangle_new_with_color(white);
	  clutter_actor_set_size(note_background, 700, 384);
	  clutter_actor_set_position(note_background, 100, 0);
	  clutter_actor_set_opacity(note_background,128);
	  clutter_actor_set_reactive(note_background, FALSE);
	  clutter_actor_show(note_background);
	  clutter_container_add_actor(CLUTTER_CONTAINER(stage), note_background);

}


gboolean start_midi_editor(gpointer data) {
	printf("Starting midi editor\n");

	ClutterActor* mainview;
	ClutterActor* view;

	mainview = jammo_get_actor_by_id("main-views-widget"); //No drawing on stage anymore, use this.
	clutter_container_foreach(CLUTTER_CONTAINER(mainview), chum_hide_this_tangle_actor, NULL);
	
	view = tangle_view_new();   //This can be defined in json too.
	clutter_container_add(CLUTTER_CONTAINER(mainview), view, NULL);
	clutter_actor_show(view);
	
	  ClutterActor* stage = view;

	  //drawing note grid
	  drawGrid(stage);

	  //drawing piano roll
	  drawPianoRoll(stage);

	  //drawing menubar (bottom)
	  drawMenuBar(stage);

	  //Hook for new note system
	  midi_note_test();

	  //Setting up note test
	  note_array_test = new_note_collection();

return FALSE;
}
