from org.maemo.hermes.engine.linkedin.service import Service
from org.maemo.hermes.engine.names import canonical
from org.maemo.hermes.engine.friend import Friend
import unittest


class FakeContact():
    id_counter = 0
    def __init__(self, name, addr, id=None):
        self.name = name;
        self.urls = addr
        self.id = id or FakeContact.id_counter
        FakeContact.id_counter = FakeContact.id_counter + 1;
    def get_urls(self):
        return self.urls
    def get_name(self):
        return self.name
    def get_identifiers(self):
        return [canonical(self.name)]
    
class FakeLinkedInApi():
    def get_friends(self):
        return self._friends
    def set_friends(self, data):
        self._friends = data
        
class UrlFriend(Friend):
    def __init__(self, name, url):
        Friend.__init__(self, name)
        self.add_url(url)
    
class TestLinkedInService(unittest.TestCase):
    
    def setUp(self):
        self.linkedInApi = FakeLinkedInApi()
        self.testee = Service('linkedin', self.linkedInApi)
        
        
    def test_that_get_unmatched_friends_works(self):
        self._fake_server_response([Friend("1"), Friend("2")])
        
        self._run_service([])
        
        friends = self.testee.get_unmatched_friends()
        assert len(friends) == 2
        names = "1.2".split(".")
        for f in friends: names.remove(f.get_name())
        assert len(names) == 0


    def test_that_process_contact_returns_None_for_unknown_contact(self):
        contact = FakeContact('Person Person', [])
        self._fake_server_response([])
        
        self.testee.pre_process_contact(contact);
        self.testee.process_friends()
        friend = self.testee.process_contact(contact)
        assert friend is None
        

    def test_that_process_contact_returns_friend_object_if_contact_is_known(self):
        same_name = "A Name"
        contact = FakeContact(same_name, [])
        fake_friend = Friend(same_name)
        self._fake_server_response([fake_friend])
        
        self.testee.pre_process_contact(contact);
        self.testee.process_friends()
        friend = self.testee.process_contact(contact)
        assert isinstance(friend, Friend)

        
    def test_main_flow_one_match_by_url_one_by_name(self):
        known_url = "http://linkedin/id=1"
        contact_by_url = FakeContact("By Url", [known_url], 1)
        friend_by_url = UrlFriend("Unimportant", known_url)
        
        known_name = "By Name"
        contact_by_name = FakeContact(known_name, [], 2)
        friend_by_name = Friend(known_name)
        
        self._fake_server_response([friend_by_name, friend_by_url])
        
        self._run_service([contact_by_name, contact_by_url])
        
        assert len(self.testee.get_unmatched_friends()) == 0
        matchers = self.testee.get_contacts_with_match()
        assert contact_by_name in matchers
        assert contact_by_url in matchers
        assert known_url in matchers[contact_by_url].get_urls()
        assert contact_by_url == matchers[contact_by_url].get_contact()
        assert contact_by_name == matchers[contact_by_name].get_contact()

        
    def _run_service(self, contacts):
        for contact in contacts:
            self.testee.pre_process_contact(contact)
        self.testee.process_friends()
        for contact in contacts:
            self.testee.process_contact(contact)
        
    def _fake_server_response(self, data):
        self.linkedInApi.set_friends(data)
    
    
if __name__ == '__main__':
    unittest.main()
