//    Group SMS
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "groupactionswindow.h"
#include "ui_groupactionswindow.h"
#include <QCheckBox>
#include <QLabel>
#include "contactswindow.h"
#include "editgroupwindow.h"
#include "commonstatic.h"
#include "sendwindow.h"

GroupActionsWindow::GroupActionsWindow(QWidget *parent, Group* group) :
    QMainWindow(parent),
    ui(new Ui::GroupActionsWindow)
{
    ui->setupUi(this);
    m_Group = group;

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5NonComposited, true);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

    setWindowTitle(m_Group->Name);

    m_ContactMenu = new QMenu(this);
    QAction* action = new QAction(tr("Select all"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(SelectAllSlot()));
    m_ContactMenu->addAction(action);
    action = new QAction(tr("Deselect all"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(DeselectAllSlot()));
    m_ContactMenu->addAction(action);
    action = new QAction(tr("Remove contact"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(RemoveOneSlot()));
    m_ContactMenu->addAction(action);

    ui->tblContacts->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(ui->tblContacts, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(ContactMenuSlot(QPoint)));
    connect(ui->tblContacts, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(ContactDoubleClick(QModelIndex)));
    connect(ui->btnAdd, SIGNAL(clicked()), this, SLOT(AddContactSlot()));
    connect(ui->btnRemove, SIGNAL(clicked()), this, SLOT(RemoveSlot()));
    connect(ui->btnSendSMS, SIGNAL(clicked()), this, SLOT(SendSlot()));
    connect(ui->btnEdit, SIGNAL(clicked()), this, SLOT(EditSlot()));

    ui->tblContacts->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);

    m_Loop = NULL;
    m_DataModel = NULL;
    PopulateContactsTable();
    SelectAllContacts();

    ui->frame->layout()->setSizeConstraint(QLayout::SetMinAndMaxSize);
}

GroupActionsWindow::~GroupActionsWindow()
{
    delete ui;
    if (m_DataModel){
        delete m_DataModel;
        m_DataModel = NULL;
    }

    if (m_Loop)
        delete m_Loop;
}


bool sortByName(const Contact* s1 , const Contact* s2)
{
    return QString::compare(s1->Name, s2->Name, Qt::CaseInsensitive) < 0;
}

void GroupActionsWindow::PopulateContactsTable()
{    
    if (m_DataModel){
        delete m_DataModel;
        m_DataModel = NULL;
    }

    foreach(GroupContact* gc, m_Group->ContactsId){
        Contact* c = CommonStatic::GetContact(gc->Number);
        if (c)
            m_Contacts.append(c);
    }
    qSort(m_Contacts.begin() , m_Contacts.end(), sortByName );

    m_DataModel = new ContactsDataModel(this, &m_Contacts);
    ui->tblContacts->setModel(m_DataModel);
    ui->tblContacts->verticalHeader()->setDefaultSectionSize(60);
    ui->tblContacts->verticalHeader()->setDefaultAlignment(Qt::AlignVCenter | Qt::AlignLeft);
    ui->tblContacts->resizeColumnToContents(0);
    ui->tblContacts->resizeColumnToContents(1);
}

void GroupActionsWindow::AddContactSlot()
{
    ContactsWindow* frm = new ContactsWindow(this, m_Group);
    frm->Init();
    if (frm->exec() == QDialog::Accepted){
        bool added = false;
        foreach (Contact* c, frm->GetSelectedContacts()){
            GroupContact* gc = new GroupContact();
            gc->Id = c->Id;
            gc->Number = c->Number;
            m_Group->ContactsId.append(gc);
            added = true;
        }

        if (added){
            m_Group->SaveToDb();
            PopulateContactsTable();
            SelectAllContacts();
            resizeEvent(NULL);
        }
    }
    delete frm;
}

void GroupActionsWindow::RemoveSlot()
{
    QList<QString> toRemove = GetSelectedNumbers();
    if (toRemove.count() > 0){
        if (CommonStatic::ConfirmMessage(this,
                                         tr("Confirm remove"),
                                         QString(tr("Remove %1 contacts from group %2?"))
                                         .arg(toRemove.count())
                                         .arg(m_Group->Name)) == QMessageBox::Yes){
            foreach(QString number, toRemove){
                m_Group->Remove(number);
            }
            m_Group->SaveToDb();
            PopulateContactsTable();
            SelectAllContacts();
        }
    }
}

void GroupActionsWindow::SendSlot()
{
    QList<Contact*> contacts = m_DataModel->getSelected();
    if (contacts.count()){
        SendWindow* frm = new SendWindow(this, &contacts);
        frm->exec();
        delete frm;
    }
}

QList<QString> GroupActionsWindow::GetSelectedNumbers()
{
    QList<QString> res;
    foreach (Contact* c, m_DataModel->getSelected())
        res.append(c->Number);
    return res;
}

void GroupActionsWindow::SelectAllContacts(bool select)
{
    m_DataModel->selectAll(select);
}

void GroupActionsWindow::EditSlot()
{
    EditGroupWindow* frm = new EditGroupWindow(this, m_Group);
    if (frm->exec() == QDialog::Accepted){
        m_Group->SaveToDb();
        setWindowTitle(m_Group->Name);
    }
    delete frm;
}

void GroupActionsWindow::show()
{
    QMainWindow::show();
    resizeEvent(NULL);
    m_Loop = new QEventLoop(this);
    m_Loop->exec();
}

void GroupActionsWindow::closeEvent(QCloseEvent*)
{
    m_Loop->quit();
}

void GroupActionsWindow::resizeEvent (QResizeEvent*)
{
    int w = ui->tblContacts->width();
    w = w - ui->tblContacts->columnWidth(0) - ui->tblContacts->columnWidth(1) - 8;
    ui->tblContacts->setColumnWidth(2, w / 2);
    ui->tblContacts->setColumnWidth(3, w / 2);
}

void GroupActionsWindow::ContactDoubleClick(QModelIndex index)
{
    m_DataModel->toggleSelect(index);
}


void GroupActionsWindow::ContactMenuSlot(QPoint point)
{
    m_ContactMenu->exec(mapToGlobal(point));
}

void GroupActionsWindow::SelectAllSlot()
{
    SelectAllContacts();
}

void GroupActionsWindow::DeselectAllSlot()
{
    SelectAllContacts(false);
}

void GroupActionsWindow::RemoveOneSlot()
{
    if (CommonStatic::ConfirmMessage(this,
                                     tr("Confirm remove"),
                                     QString(tr("Remove 1 contact from group %1?"))
                                     .arg(m_Group->Name)) == QMessageBox::Yes){
        m_Group->Remove(m_DataModel->getContact(ui->tblContacts->currentIndex())->Number);
        m_Group->SaveToDb();
        PopulateContactsTable();
        SelectAllContacts();
    }
}
