/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "searchdlg.h"
#include "ui_searchdlg.h"

#include <QStringBuilder>
#include <QUrl>
#include <QNetworkReply>
#include <QDebug>

#include <QScriptEngine>
#include <QScriptValueIterator>
#include <QtScriptTools>
#include <QScriptEngineDebugger>

#ifdef Q_WS_MAEMO_5
    #include "maemo5wnd.h"
#endif

#include "localsearchitemdelegate.h"

SearchDlg::SearchDlg(QNetworkAccessManager *pNetAccessMngr, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::SearchDlg)
{
    ui->setupUi(this);

    m_selectedIndex = -1;
    m_pNetAccessMngr = pNetAccessMngr;
    m_hasChanged = true;

    // ???
    // m_debugger.attachTo(&m_engine);

    m_pItemDelegate = new LocalSearchItemDelegate(ui->listView);

#ifdef Q_WS_MAEMO_5
    m_pItemDelegate->m_font = QFont("Nokia Sans", 24);
#else
    m_pItemDelegate->m_font = ui->listView->font();
#endif

    ui->listView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    ui->listView->setWordWrap(true);
    ui->listView->setItemDelegate(m_pItemDelegate);

    m_pItemDelegate->m_leftMargin = ui->listView->fontMetrics().width("000");
    // m_pItemDelegate->m_rightMargin = right + 10;
    m_pItemDelegate->m_vertMargin = QFontMetrics(m_pItemDelegate->m_font).height() / 2;
}

SearchDlg::~SearchDlg() {
    delete ui;
}

void SearchDlg::on_searchBtt_clicked() {
    search();
}

void SearchDlg::search() {
    // http://code.google.com/apis/ajaxsearch/documentation/reference.html#_fonje_local
    // http://code.google.com/apis/ajaxsearch/documentation/reference.html#_class_GlocalResult
    // http://qtwiki.org/Parsing_JSON_with_QT_using_standard_QT_library

    QUrl url("http://ajax.googleapis.com/ajax/services/search/local");
    url.addQueryItem("v", "1.0");
    url.addQueryItem("q", ui->queryEdit->text());
    url.addQueryItem("sll", m_latLongStr ); // "lat,long"
    url.addQueryItem("rsz", "8");

    /*
     hl? (hl=fr)
    This optional argument supplies the host language of the application making the request.
    If this argument is not present then the system will choose a value based on the value of the Accept-Language http header.
    If this header is not present, a value of en is assumed.
    */
    // url.addQueryItem("hl", ... );

    /*
    This optional argument specifies which type of listing the user is interested in. Valid values include:

        * blended - request KML, Local Business Listings, and Geocode results
        * kmlonly - request KML and Geocode results
        * localonly - request Local Business Listings and Geocode results

    If this argument is not supplied, the default value of localonly is used.
    */
    url.addQueryItem("mrt","localonly");

    qDebug() << QString(url.toEncoded());

#ifdef Q_WS_MAEMO_5
    Maemo5Wnd::ShowProgressIndicator(this, true);
#endif

    m_pReply = m_pNetAccessMngr->get(QNetworkRequest(url));
    connect(m_pReply, SIGNAL(finished()), this, SLOT(onFinished()));
}

void SearchDlg::onFinished() {
#ifdef Q_WS_MAEMO_5
    Maemo5Wnd::ShowProgressIndicator(this, false);
#endif

    if ( m_pReply->error()!=QNetworkReply::NoError ) {
        qDebug() << "Error: " << m_pReply->errorString();
        return;
    }

    QByteArray array = m_pReply->readAll();
    if ( array.length()<1 ) {
        // ????
        QMessageBox::warning(this, "Local Business Search", "Error processing request");
        return;
    }

    QString str = QString::fromUtf8(array.constData());
    str = "result = (" + str + ");";
    qDebug() << str;

    QScriptValue sc = m_engine.evaluate(str); // In new versions it may need to look like engine.evaluate("(" + QString(result) + ")");
    qint32 responseStatus = sc.property("responseStatus").toInt32();
    switch ( responseStatus ) {
        case 200: // ok
            break;
        default:
            qDebug() << "Invalid response status: " << responseStatus << sc.property("responseDetails").toString();
            return;
    }

    ui->listView->setModel(NULL);
    m_model.clear();

    QScriptValueIterator it(sc.property("responseData").property("results"));
    while (it.hasNext()) {
        it.next();
        // GsearchResultClass
        // html
        Business bus;
        QScriptValue result = it.value();
        bus.setDescription( result.property("titleNoFormatting").toString() );
        bus.setCoordinates(
            result.property("lat").toNumber(),
            result.property("lng").toNumber()
        );
        bus.m_url = result.property("url").toString();

        bus.m_address.m_address = result.property("streetAddress").toString();
        bus.m_address.m_city = result.property("city").toString();
        bus.m_address.m_region = result.property("region").toString();
        bus.m_address.m_country = result.property("country").toString();

        QScriptValueIterator i2(result.property("phoneNumbers"));
        while (i2.hasNext()) {
            i2.next();
            Phone phone;
            phone.m_type = i2.value().property("type").toString();
            phone.m_number = i2.value().property("number").toString();

            bus.m_phoneNumbers << phone;
        }

        if ( bus.description().isEmpty() ) {
            qDebug() << "something wrong here";
            continue;
        }
        m_model << bus;
        // ui->listWidget->addItem(bus.title());
    }

    ui->listView->setModel(&m_model);
}

Business SearchDlg::selectedBusiness() {
    Business* pBusiness = m_model[m_selectedIndex];
    Q_ASSERT(pBusiness);
    return *pBusiness;
}

void SearchDlg::on_listView_clicked(QModelIndex index) {
    // for now show, later will show details...
    m_selectedIndex = index.row();
    accept();
}

void SearchDlg::on_listView_doubleClicked(QModelIndex index) {
    // later will show will trigger accept?
}

void SearchDlg::on_queryEdit_textChanged(QString ) {
    m_hasChanged = true;
    ui->listView->setModel(NULL);
}
