/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "fingerscrolleventfilter.h"

#include <QDebug>
#include <QBasicTimer>

#include "maemo5wnd.h"

// ---------------------------------------------------------------------------------------------------------------
FingerScrollEventFilter::FingerScrollEventFilter(QWebView *pView) :
    QObject(pView)
{
    pView->installEventFilter(this);

    pView->setAttribute(Qt::WA_OpaquePaintEvent, true);
    pView->setAttribute(Qt::WA_NoSystemBackground, true);
    pView->setMouseTracking(true);

    // ...
    m_state = Steady;
    m_threshold = 20;
    m_timeStamp = QTime::currentTime();
}

bool FingerScrollEventFilter::eventFilter(QObject *, QEvent *e) {
    switch (e->type()) {
        case QEvent::MouseButtonPress:
            if ( mousePressEvent(static_cast<QMouseEvent *>(e)) ) {
                m_isMousePressed = true;
                return true;
            }
            return true; // ??? should it allow to send event?

        case QEvent::MouseButtonRelease:
            if ( m_isMousePressed ) {
                mouseReleaseEvent(static_cast<QMouseEvent *>(e));
                m_isMousePressed = false;
            }
            return true;

        case QEvent::MouseMove:
            mouseMoveEvent(static_cast<QMouseEvent *>(e));
            return true;

            /*
        case QEvent::KeyPress:
            return keyPressEvent(static_cast<QKeyEvent*>(e));

        case QEvent::KeyRelease:
            // ignore
            return true;
            */

        default:
            break;
    }

    return false;
}

bool FingerScrollEventFilter::keyPressEvent(QKeyEvent *pEvent) {
    return false;
}

QMainWindow *FingerScrollEventFilter::mainWindow(QWidget *pParent/*=NULL*/) {
    if ( !pParent ) pParent = dynamic_cast<QWidget *>(parent());

    Q_ASSERT(pParent);
    QMainWindow *pMain = dynamic_cast<QMainWindow*>(pParent);
    if ( pMain ) return pMain;

    pParent = pParent->parentWidget();
    if ( !pParent ) return NULL;
    return mainWindow(pParent);
}

void FingerScrollEventFilter::startTicker(int interval) {
    Maemo5Wnd::ShowProgressIndicator(mainWindow(), true);

    if (!m_timer.isActive())
        m_timer.start(interval, this);
}

void FingerScrollEventFilter::stopTicker() {
    Maemo5Wnd::ShowProgressIndicator(mainWindow(), false);

    m_timer.stop();
}

static QPoint deaccelerate(const QPoint &speed, int a = 1, int max = 64)
{
    int x = qBound(-max, speed.x(), max);
    int y = qBound(-max, speed.y(), max);
    x = (x == 0) ? x : (x > 0) ? qMax(0, x - a) : qMin(0, x + a);
    y = (y == 0) ? y : (y > 0) ? qMax(0, y - a) : qMin(0, y + a);
    return QPoint(x, y);
}


void FingerScrollEventFilter::timerEvent(QTimerEvent *event) {
    Q_UNUSED(event);

    if (m_state ==  AutoScroll) {
        m_speed = deaccelerate(m_speed);
        scroll(m_speed);
        if (m_speed == QPoint(0, 0)) {
            m_state = Steady;
            stopTicker();
        }
        return;
    }

    stopTicker();
}

bool FingerScrollEventFilter::mousePressEvent(QMouseEvent *pEvent) {
    if (m_ignoreList.removeAll(pEvent) || pEvent->button() != Qt::LeftButton) {
        m_state = Steady;
        // QWebView::mousePressEvent(pEvent);
        return false;
    }

    Q_ASSERT(pEvent->pos().y()>=0);

    switch (m_state) {

        case Steady:
            //qDebug() << "handleMousePress: Steady - register click";
            pEvent->accept();
            m_state = Pressed;
            m_lastPos = m_pressPos = pEvent->pos();
            break;

        case AutoScroll:
            //qDebug() << "handleMousePress: AutoScroll - stop auto scrolling";
            pEvent->accept();
            m_state = Stop;
            m_speed = QPoint(0, 0);
            m_pressPos = pEvent->pos();
            stopTicker();
            break;

        case ManualScroll: // work around bug
            qDebug() << "handleMousePress: ManualScroll - workaround bug (do nothing)";
            pEvent->accept();
            break;

        default:
            qDebug() << "handleMousePress: error? " << m_state;
            return false;
    }

    return true;
}

bool FingerScrollEventFilter::mouseMoveEvent(QMouseEvent *pEvent) {
    if (!(pEvent->buttons() & Qt::LeftButton)) {
        return false;
    }

    // if (m_ignoreList.removeAll(event)) return;

    QPoint delta;
    switch (m_state) {
        case Pressed:
        case Stop:
            delta = pEvent->pos() - m_pressPos;
            if ( delta.x() > m_threshold || delta.x() < -m_threshold ||
                delta.y() > m_threshold || delta.y() < -m_threshold )
            {
                //qDebug() << "handleMouseMove (Pressed or Stop): start manual scroll";

                m_timeStamp = QTime::currentTime();
                m_state = ManualScroll;
                delta = QPoint(0, 0);
                m_pressPos = pEvent->pos();
                // m_offset = scrollOffset(); // ?!?!?!?!?!? felipe
                pEvent->accept();

            } else {
                //qDebug() << "handleMouseMove (Pressed or Stop): bellow threshold: " << m_state;

            }
            break;

        case ManualScroll: {
                //qDebug() << "handleMouseMove: ManualScroll";
                pEvent->accept();
                delta = pEvent->pos() - m_lastPos;
                // QPoint newpos = m_offset - delta;

                /*
                qDebug() << ">> Press: " << m_pressPos.y()
                        << ", Event: " << pEvent->pos().y()
                        << ", Delta: " << delta.y()
                        << ", Scroll: " << m_offset.y()
                        << " = " << newpos.y();
                */

                scroll(delta);
                m_lastPos = pEvent->pos();

                if (m_timeStamp.elapsed() > 100) {
                    //qDebug() << ">> elapsed()>100?";
                    m_timeStamp = QTime::currentTime();
                    m_speed = delta; // - m_delta; // !?!?!??!?!?
                }
            }
            break;

        default:
            qDebug() << "handleMouseMove: error?" << m_state;
            return false;
    }

    return true;
}

bool FingerScrollEventFilter::mouseReleaseEvent(QMouseEvent *pEvent) {
    if (m_ignoreList.removeAll(pEvent) || pEvent->button() != Qt::LeftButton) {
        return false;
    }

    QPoint delta;
    switch (m_state) {
       case Pressed: {
            qDebug() << "handleMouseRelease: Pressed - fire fake events";

            // ????????????????????????????????????????????
            /*
            pEvent->accept();
            m_state = Steady;
            QMouseEvent *m_pPressEvent = new QMouseEvent(
                QEvent::MouseButtonPress,
                m_pressPos, Qt::LeftButton,
                Qt::LeftButton, Qt::NoModifier
            );
            QMouseEvent *m_pReleaseEvent = new QMouseEvent(*pEvent);
            m_ignoreList << m_pPressEvent;
            m_ignoreList << m_pReleaseEvent;
            QApplication::postEvent(this, m_pPressEvent);
            QApplication::postEvent(this, m_pReleaseEvent);
            */
            break;
        }

        case ManualScroll:
            //qDebug() << "handleMouseRelease: ManualScroll";

            pEvent->accept();
            delta = pEvent->pos() - m_lastPos;
            if (m_timeStamp.elapsed() > 100) {
                qDebug() << ">> elapsed>100: ????";
                m_timeStamp = QTime::currentTime();
                m_speed = delta; //  - m_delta;
                // m_delta = delta;
            }

            // m_offset = scrollOffset();
            m_pressPos = pEvent->pos();

            if (m_speed == QPoint(0, 0)) {
                //qDebug() << ">> keep it steady?";
                m_state = Steady;
            } else {
                qDebug() << ">> start Autoscroll";
                m_speed /= 4;
                m_state = AutoScroll;
                startTicker(20);
            }
            break;

        case Stop:
           // qDebug() << "handleMouseRelease: Stop - just stopped? make it steady";
            pEvent->accept();
            m_state = Steady;
            break;

        default:
            qDebug() << "handleMouseRelease: error? " << m_state;
            return false;
    }

    return true;
}
