/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <rtcom-eventlogger/eventlogger.h>
#include <rtcom-eventlogger-ui/rtcom-log-view.h>
#include <rtcom-eventlogger-ui/rtcom-log-model.h>
#include <rtcom-eventlogger-ui/rtcom-log-columns.h>
#include <rtcom-eventlogger-ui/rtcom-log-search-bar.h>

#include "timeline-view.h"

#define GET_PRIVATE(o) ((TimelineViewPrivate *)       \
                        ((TimelineView *)(o))->priv)

#define STATUS_LIMIT 150

enum
{
    PROP_0,
    PROP_NEED_TOP_BOX
};

/**
 * TimelineView displays status messages in a timeline
 * It is used by main_status, delete_status and contact_status windows
 */
typedef struct _TimelineViewPrivate TimelineViewPrivate;
struct _TimelineViewPrivate
{
  GtkWidget *hbox; /* hbox on the top */
  GtkWidget *panarea; /* panable area for status timeline */

  /* log_view and log_model */
  GtkWidget *log_view;
  RTComLogModel *log_model;
  RTComEl *eventlogger;
  GtkWidget *search_bar; /* eventlogger search bar */
  GtkTreeSelection *selection; /* for deleting statuses */

  RTComElQueryGroupBy group_by;
  OssoABookRoster *aggregator;

  gboolean need_top_box;
};

G_DEFINE_TYPE (TimelineView, timeline_view, GTK_TYPE_ALIGNMENT);


/**************************************/
/* Private functions                  */
/**************************************/

/**
 * init_event_logger:
 *
 * Initialize log_model, log_view and search_bar
 *
 * The window object which inits this TimelineView should add
 * the search_bar by calling timeline_view_search_bar_widget_hook ()
 *
 * It should also set abook_aggregator for log_model
 * by calling timeline_view_set_abook_aggregator ()
 *
 * Besides, timeline_view_set_select_status_cb () should be called to
 * set the callback function for "row-activated" event
 */
static void
init_event_logger(TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  priv->log_model = rtcom_log_model_new();
  rtcom_log_model_set_limit (priv->log_model, STATUS_LIMIT);
  priv->log_view = rtcom_log_view_new();

  priv->search_bar = rtcom_log_search_bar_new();
  rtcom_log_search_bar_set_model(
          RTCOM_LOG_SEARCH_BAR(priv->search_bar),
          priv->log_model);

  rtcom_log_view_set_model(
          RTCOM_LOG_VIEW(priv->log_view),
          rtcom_log_search_bar_get_model_filter(
              RTCOM_LOG_SEARCH_BAR(priv->search_bar)));

  rtcom_log_view_set_highlight_new_events (
          RTCOM_LOG_VIEW(priv->log_view), FALSE);

  /* The view has acquired its own reference to the
   * model, so we can drop ours. That way the model will
   * be freed automatically when the view is destroyed.
   */
  g_debug("Unreffing the model, because the view reffed it.");
  g_object_unref(priv->log_model);

  /* get eventlogger from log_model */
  priv->eventlogger = rtcom_log_model_get_eventlogger (priv->log_model);

  priv->group_by = RTCOM_EL_QUERY_GROUP_BY_NONE;
}

/**************************************/
/* GObject functions                  */
/**************************************/

static GObject*
constructor (GType type,
             guint n_params,
             GObjectConstructParam *params)
{
  GObjectClass *object_class = (GObjectClass*) timeline_view_parent_class;
  GObject *object = object_class->constructor (type, n_params, params);
  TimelineView *view = TIMELINE_VIEW (object);
  TimelineViewPrivate *priv = GET_PRIVATE (object);
  GtkWidget *vbox;

  /* Initialize the event logger as well as log_model and log_view */
  init_event_logger (view);

  vbox = gtk_vbox_new (FALSE, 0);
  priv->panarea = hildon_pannable_area_new ();
  g_object_set (priv->panarea,
                "mov-mode", HILDON_MOVEMENT_MODE_VERT,
                "hscrollbar-policy", GTK_POLICY_NEVER,
                NULL);

  if (priv->need_top_box) {
    /* Only create and add top hbox when needed (e.g. me view) */
    priv->hbox = gtk_hbox_new (FALSE, 0);

    gtk_widget_set_name (priv->log_view,
        "friend-status::timeline-view::timeline-view");
    gtk_rc_parse_string (
        "widget \"*.friend-status::timeline-view::timeline-view\""
        " style \"fremantle-touchlist\"");

    gtk_box_pack_start (GTK_BOX (vbox), priv->hbox, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (vbox), priv->log_view);
    hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (priv->panarea), vbox);
    gtk_container_add (GTK_CONTAINER (view), priv->panarea);
  }
  else {
    gtk_container_add (GTK_CONTAINER (priv->panarea), priv->log_view);
    gtk_container_add (GTK_CONTAINER (vbox), priv->panarea);
    gtk_box_pack_start (GTK_BOX(vbox), priv->search_bar, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER(view), vbox);
  }

  g_object_set (view,
                "top-padding", HILDON_MARGIN_HALF,
                "left-padding", HILDON_MARGIN_DOUBLE,
                "right-padding", HILDON_MARGIN_DOUBLE,
                NULL);

  return object;
}

static void
timeline_view_init (TimelineView *view)
{
  TimelineViewPrivate *priv;

  priv = view->priv = G_TYPE_INSTANCE_GET_PRIVATE(view, TIMELINE_TYPE_VIEW,
                                                  TimelineViewPrivate);
  priv->hbox = NULL;
}

static void
timeline_view_dispose (GObject *obj)
{
  TimelineViewPrivate *priv = GET_PRIVATE (obj);
/* TODO Warning????
  if (priv->log_model) {
    g_object_unref (priv->log_model);
    priv->log_model = NULL;
  }
*/
  if (priv->aggregator) {
    g_object_unref (priv->aggregator);
    priv->aggregator = NULL;
  }

  G_OBJECT_CLASS (timeline_view_parent_class)->dispose (obj);
}

static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
    TimelineViewPrivate *priv = GET_PRIVATE (object);

    switch (prop_id)
    {
        case PROP_NEED_TOP_BOX:
            priv->need_top_box = g_value_get_boolean (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
timeline_view_class_init (TimelineViewClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->constructor = constructor;
  object_class->set_property = set_property;
  object_class->dispose = timeline_view_dispose;

  g_type_class_add_private (klass, sizeof (TimelineViewPrivate));

  g_object_class_install_property (object_class, PROP_NEED_TOP_BOX,
      g_param_spec_boolean ("need-top-box", "Need top hbox",
                            "Need top hbox", FALSE,
                            G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}

/**************************************/
/* Public functions                  */
/**************************************/

GtkWidget *
timeline_view_new (gboolean need_top_box)
{
  return g_object_new (TIMELINE_TYPE_VIEW,
                       "need-top-box", need_top_box,
                       NULL);
}

GtkWidget *timeline_view_get_top_box (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);
  return priv->hbox;
}

GtkWidget *
timeline_view_get_action_area_box (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);
  GtkWidget *box = hildon_tree_view_get_action_area_box (GTK_TREE_VIEW (priv->log_view));

  hildon_tree_view_set_action_area_visible (GTK_TREE_VIEW (priv->log_view), TRUE);
  return box;
}

void
timeline_view_add_search_bar (TimelineView *view,
                              GtkWidget *window)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  rtcom_log_search_bar_widget_hook(
      RTCOM_LOG_SEARCH_BAR(priv->search_bar),
      GTK_WIDGET(window),
      GTK_TREE_VIEW(priv->log_view));
}

void
timeline_view_hide_search_bar (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  gtk_widget_hide (priv->search_bar);
}

void
timeline_view_set_abook_aggregator (TimelineView *view,
                                    OssoABookRoster *aggregator)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  priv->aggregator = g_object_ref (aggregator);
  rtcom_log_model_set_abook_aggregator (priv->log_model,
      OSSO_ABOOK_AGGREGATOR (priv->aggregator));
}

RTComEl *
timeline_view_get_eventlogger (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  return priv->eventlogger;
}

void
timeline_view_set_select_status_cb (TimelineView *view,
                                    TimelineViewSelectStatusCallback cb,
                                    gpointer userdata)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  g_signal_connect(priv->log_view, "row-activated",
                   (GCallback) cb, userdata);
}

/**
 * For delete_status_view
 */
void
timeline_view_set_mode_selection_multiple (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  hildon_gtk_tree_view_set_ui_mode (GTK_TREE_VIEW (priv->log_view),
                                    HILDON_UI_MODE_EDIT);
  priv->selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->log_view));
  gtk_tree_selection_set_mode (priv->selection, GTK_SELECTION_MULTIPLE);
}

GtkTreeSelection *
timeline_view_get_selection (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  return priv->selection;
}

void
timeline_view_populate_common_status (TimelineView *view,
                                      const gchar *eventtype)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);
  RTComElQuery *query;

  query = rtcom_el_query_new (priv->eventlogger);
  rtcom_el_query_set_group_by(query, priv->group_by);
/*  rtcom_el_query_set_limit (query, 100); */

  if(!rtcom_el_query_prepare (
              query,
              "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
              "event-type", eventtype, RTCOM_EL_OP_EQUAL,
              NULL)) {
      g_debug("Failed to prepare the query");
      return ;
  }

  rtcom_log_model_populate_query (priv->log_model, query);
  g_object_unref (query);
}

void
timeline_view_populate_all_status (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  const gchar * services[] = {"RTCOM_EL_SERVICE_STATUS", NULL};
  rtcom_log_model_set_group_by (priv->log_model, priv->group_by);
  rtcom_log_model_populate (priv->log_model, services);
}

/**
 * @contact: IM name of a remote contact (e.g. someone@gmail.com)
 *
 * Populate status history of a contact
 */
void
timeline_view_populate_contact_status (TimelineView *view,
                                       const gchar *account,
                                       const gchar *contact)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);
  RTComElQuery *query = rtcom_el_query_new (priv->eventlogger);
  gboolean res;

  if (account) {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "local-uid", account, RTCOM_EL_OP_EQUAL,
                "remote-uid", contact, RTCOM_EL_OP_EQUAL,
                NULL);
  }
  else {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "remote-uid", contact, RTCOM_EL_OP_EQUAL,
                NULL);
  }

  if(!res) {
    g_debug("Failed to get the last status message of %s", contact);
    return ;
  }

  /* Don't need g_object_unref (query),
   * rtcom_log_model will unref the query later */
  rtcom_log_model_populate_query (priv->log_model, query);
}

void
timeline_view_set_group_by (TimelineView *view,
                            RTComElQueryGroupBy group_by)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  /* This is for populating friends status via
   * timeline_view_populate_common_status () */
  priv->group_by = group_by;
  /* This is for populating future added events,
   * so in collapse view a new friend status will update an existing row
   * while in expand view it will add a new row */
  rtcom_log_model_set_group_by (priv->log_model, group_by);
}

/**
 * A hack for showing event-count in the view when grouped by contact
 */
void
timeline_view_row_changed (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);
  GtkTreeIter iter;

  if (gtk_tree_model_get_iter_first (
      GTK_TREE_MODEL (priv->log_model), &iter)) {
      GtkTreePath *path;

      path = gtk_tree_model_get_path (
          GTK_TREE_MODEL (priv->log_model), &iter);

      if (path) {
        g_signal_emit_by_name (
            GTK_TREE_MODEL (priv->log_model),
            "row-changed", path, &iter);
        gtk_tree_path_free (path);
      }
  }
}

void timeline_view_refresh (TimelineView *view)
{
  TimelineViewPrivate *priv = GET_PRIVATE (view);

  rtcom_log_model_refresh (priv->log_model);
}
