from ui.Widget import Widget
from ui.Pixmap import Pixmap, pixmap_for_text, TEMPORARY_PIXMAP
from ui.KineticScroller import KineticScroller
from theme import theme


# width of visible scale portion on screen in KHz
_SCALE_WIDTH = 3000


class FMScale(Widget):
    """
    Widget for a scrollable FM scale with station labels.
    """

    EVENT_FREQUENCY_CHANGED = "event-frequency-changed"
    

    def __init__(self):
    
        # buffer for offscreen drawings
        self.__buffer = None
        
        # whether the buffer contents are valid
        self.__buffer_invalidated = True
    
        # table: freq -> station label pixmap
        self.__stations = {}
        
        # current range of the scale
        self.__range = (87500, 108000)
        
        # offset into the scale (center of the screen) in pixels
        self.__offset = 0
        
        # total width of the scale in pixels
        self.__total_width = 0
        
        # scaling factor for converting screen coordinates to frequency values
        self.__scaling = 1.0
    

        Widget.__init__(self)
        
        # kinetic scrolling
        kscr = KineticScroller(self)
        kscr.connect_clicked(self.__on_click)
        kscr.connect_scrolling_stopped(self.__on_stop_scrolling)



    def connect_frequency_changed(self, cb, *args):
        
        self._connect(self.EVENT_FREQUENCY_CHANGED, cb, *args)


    def __on_click(self, px, py):
    
        w, h = self.get_size()
        v = self.get_value()
        if (px < w / 2):
            self.set_value(v - 100)
        else:
            self.set_value(v + 100)
        
        #v = self.get_value_at(px)
        #self.set_value(v)


    def __on_stop_scrolling(self):

        w, h = self.get_size()
        v = self.get_value_at(w / 2 + 1)
        self.set_value(v)        
        self.emit_event(self.EVENT_FREQUENCY_CHANGED, v)
        
        
    def __prepare_scale(self):
        """
        Prepares the scale by precomputing some values.
        """

        w, h = self.get_size()
        r1, r2 = self.__range

        self.__scaling = w / float(_SCALE_WIDTH)
        self.__total_width = int((r2 - r1) * self.__scaling)

        
        
        
    def set_size(self, w, h):
    
        old_w, old_h = self.get_size()
        Widget.set_size(self, w, h)
        
        if ((w, h) != (old_w, old_h)):
            self.__prepare_scale()
            self.__buffer = Pixmap(None, w, h)
            self.__buffer_invalidated = True
              
        
    def render_this(self):

        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()

        if (self.__buffer_invalidated):
            self.__buffer_invalidated = False
            self.__render(0, w)

        # render overlays
        #TEMPORARY_PIXMAP.copy_buffer(self.__buffer, 0, 0, 0, 0, w, h)        
        #TEMPORARY_PIXMAP.fill_area(w / 2 - 5, 0, 10, h, "#aaaaffa0")
        
        #TEMPORARY_PIXMAP.draw_pixbuf(theme.fmradio_scale_left,
        #                             0, 0, 128, h, True)

        #TEMPORARY_PIXMAP.draw_pixbuf(theme.fmradio_scale_right,
        #                             w - 128, 0, 128, h, True)
        
        #screen.copy_buffer(TEMPORARY_PIXMAP, 0, 0, x, y, w, h)
        screen.copy_buffer(self.__buffer, 0, 0, x, y, w, h)



    def __render(self, x1, x2):
        """
        Renders the given portion of the scale. The given x-coordinates specify
        screen positions.
        """

        if (not self.__buffer): return

        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()    

        r1, r2 = self.__range

        self.__buffer.fill_area(x1, 0, x2 - x1, h, theme.color_fmradio_scale_bg)
        self.__buffer.set_clip_rect(x1, 0, x2 - x1, h)
        for xpos in range(x1 - 50, x2 + 50):
            freq = r1 + int((self.__offset - w / 2 + xpos) / self.__scaling)
            freq -= (freq % 10)

            # don't render outside of the current frequency band
            if (freq < r1 or freq > r2): continue

            h6 = h / 6

            # render ticks only when on visible screen
            if (x1 - 1 <= xpos < x2 + 1):
                # determine type of tick to render
                if (freq % 1000 == 0):
                    self.__buffer.fill_area(xpos - 1, h6,
                                            1, 5 * h6,
                                            theme.color_fmradio_scale_ticks1)
                
                elif (freq % 500 == 0):
                    self.__buffer.fill_area(xpos - 1, 3 * h6,
                                            1, 3 * h6,
                                            theme.color_fmradio_scale_ticks1)

                elif (freq % 100 == 0):
                    self.__buffer.fill_area(xpos - 1, 5 * h6,
                                            1, h6,
                                            theme.color_fmradio_scale_ticks2)

            #end if


            # labels should be rendered outside the visible area as well to
            # avoid cutoffs
            if (freq % 1000 == 0):
                self.__buffer.draw_centered_text(`freq / 1000`,
                                                 theme.font_fmradio_scale,
                                                 xpos - 50, 0, 100, 32,
                                                 theme.color_fmradio_scale_text)

            # render station label
            if (freq % 100 == 0):
                pmap = self.__stations.get(freq)
                if (pmap):
                    p_w, p_h = pmap.get_size()
                    self.__buffer.draw_pixmap(pmap,
                                              xpos - p_w / 2, h - h6 - p_h)
                
        #end for
        self.__buffer.set_clip_rect(None)
                
        
    def set_range(self, a, b):
        """
        Sets the range of the frequency band to display.

        @param a: begin of FM band
        @param b: end of FM band
        """
    
        self.__range = (a, b)
        self.__prepare_scale()


    def set_stations(self, *stations):
        """
        Sets station names for frequencies that will be visible on the scale.
        
        @param stations: variable list of (frequency, name) tuples
        """
    
        self.__stations.clear()
        for freq, name in stations:
            if (not name.strip()):
                name = "%0.1f MHz" % (freq / 1000.0)
            name = " " + name + " "
            pmap = pixmap_for_text(name, theme.font_fmradio_scale_station)
            w, h = pmap.get_size()
            pmap.fill_area(0, 0, w, h, theme.color_fmradio_scale_bg)
            pmap.draw_text(name, theme.font_fmradio_scale_station,
                           0, 0, theme.color_fmradio_scale_station)
            
            pmap.rotate(90)
            self.__stations[freq] = pmap
        #end for
        self.__buffer_invalidated = True
        self.render()


    def get_value_at(self, px):
        """
        Returns the frequency value at the given screen position.
        
        @param px: x-position on screen
        @return: frequency value in KHz at that position
        """
    
        w, h = self.get_size()
        r1, r2 = self.__range

        freq = r1 + int((self.__offset - w / 2 + px + 1) / self.__scaling)
        freq -= freq % 100
        
        freq = max(r1, min(r2, freq))
        
        return freq
        
        
        
    def set_value(self, freq):
        """
        Sets the current frequency value.
        
        @param freq: frequency value in KHz
        """
    
        freq -= freq % 100
        self.__move_to(freq)
        
        
    def get_value(self):
        """
        Returns the current frequency value.
        
        @return: frequency value in KHz
        """
    
        w, h = self.get_size()
        return self.get_value_at(w / 2)        
        


    def move(self, dx, dy):
    
        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()

        if (dx < 0 and self.__offset + dx >= 0):
            # ->
            self.__offset += dx
            self.__buffer.move_area(0, 0, w - abs(dx), h, abs(dx), 0)
            self.__render(0, abs(dx))
            self.render()
            return (dx, 0)
            
        elif (dx > 0 and self.__offset + dx < self.__total_width):
            # <-
            self.__offset += dx
            self.__buffer.move_area(dx, 0, w - dx, h, -dx, 0)
            self.__render(w - abs(dx), w)
            self.render()
            return (dx, 0)            

        else:
            return (0, 0)
            
            
            
    def __move_to(self, freq):
    
        def fx(params):
            from_x, to_x = params
            dx = (to_x - from_x) / 3

            if (abs(dx) > 0):
                self.move(dx, 0)

                params[0] = from_x + dx
                params[1] = to_x                
                return True

            else:
                dx = to_x - from_x
                self.move(dx, 0)
                return False

        w, h = self.get_size()
        r1, r2 = self.__range
        new_pos = int((freq - r1) * self.__scaling)
        self.animate(50, fx, [self.__offset, new_pos])

