# coding: utf-8

import uuid
import ConfigParser
import feedcircuit
import os
import locale

feed_template_default = """
<html>
    <head>
        <meta http-equiv="Content-Type" content="text/html;charset=utf-8" >
        %(style)s
    </head>
    <body>
    </body>
</html>
"""
feed_style_default = "<style> body, table { font-size: 15pt } </style>"
custom_style = '<link rel="stylesheet" href="%(filename)s" type="text/css" />'

def get_file_content(filename):
    if filename:
        f = open(filename, "r")
        try:
            return f.read()
        finally:
            f.close()
    return None


class ConfigParserEx(ConfigParser.SafeConfigParser):


    def get_default(self, section, name, default):
        if self.has_option(section, name):
            return self.get(section, name)
        return default


    def getint_default(self, section, name, default):
        if self.has_option(section, name):
            return self.getint(section, name)
        return default


    def getboolean_default(self, section, name, default):
        if self.has_option(section, name):
            return self.getboolean(section, name)
        return default


    def set(self, section, option, value):
        val = value
        if val:
            val = val.replace("%", "%%")
        ConfigParser.SafeConfigParser.set(self, section, option, val)


class FeedConfig:


    mode_ommit = 0
    mode_cache = 1
    mode_inline = 2


    def __init__(self, url, name, mode, icon_filename = "",
        days_to_keep = 7, include = "", exclude = "", message_count = 0,
        content_tag = "", reformatter_url = "", checked = False,
        ignore_images = False, id = None,
        print_version_keywords = feedcircuit.print_version_keywords_default,
        single_page_keywords = feedcircuit.single_page_keywords_default,
        next_page_keywords = feedcircuit.next_page_keywords_default):
        self.id = id
        if not self.id:
            self.id = uuid.uuid1().hex
        self.url = url
        self.name = name
        self.mode = mode
        self.days_to_keep = days_to_keep
        self.include = include
        self.exclude = exclude
        self.icon_filename = icon_filename
        self.message_count = message_count
        self.reformatter_url = reformatter_url
        self.content_tag = content_tag
        self.print_version_keywords = print_version_keywords
        self.single_page_keywords = single_page_keywords
        self.next_page_keywords = next_page_keywords
        self.ignore_images = ignore_images
        self.ord_num = 0
        self.checked = checked


    def get_filename(self):
        return self.id + ".html"


    def get_print_version_keywords_string(self):
        return ",".join(self.print_version_keywords)


    def set_print_version_keywords_string(self, keywords):
        self.print_version_keywords = [str.strip().lower() for str in keywords.split(",")]


    def get_single_page_keywords_string(self):
        return ",".join(self.single_page_keywords)


    def set_single_page_keywords_string(self, keywords):
        self.single_page_keywords = [str.strip().lower() for str in keywords.split(",")]


    def get_next_page_keywords_string(self):
        return ",".join(self.next_page_keywords)


    def set_next_page_keywords_string(self, keywords):
        self.next_page_keywords = [str.strip().lower() for str in keywords.split(",")]


    def get_content_kwargs(self):
        result = {}
        if self.content_tag:
            parts = self.content_tag.split(" ")
            if len(parts):
                result["name"] = parts[0]
                result["attrs"] = {}
                for part in parts[1:]:
                    name_val = part.split("=")
                    if len(name_val) == 2:
                        result["attrs"][name_val[0]] = \
                            name_val[1].replace('"', "").replace("&quot;", '"')
        return result


    def get_urls(self):
        return filter(len, [str.strip() for str in self.url.split(" ")])


    def get_file_path(self, config_path, filename):
        result = os.path.join(config_path, self.id + "." + filename)
        if os.path.exists(result):
            return result
        result = os.path.join(config_path, filename)
        if os.path.exists(result):
            return result
        return None


    def get_templates(self, config_path):
        item = get_file_content(self.get_file_path(config_path, "item.template.html"))
        update = get_file_content(self.get_file_path(config_path, "update.template.html"))
        feed = get_file_content(self.get_file_path(config_path, "feed.template.html"))
        page = get_file_content(self.get_file_path(config_path, "page.template.html"))
        if not feed:
            style_filename = self.get_file_path(config_path, "style.css")
            if style_filename:
                feed = feed_template_default % {"style": custom_style % {"filename": style_filename}}
            else:
                feed = feed_template_default % {"style": feed_style_default}
        if not page and feed:
            page = feed
        return (item, update, feed, page)



class FeedCircuitConfig:
   
    
    def __init__(self, filename):
        self.filename = filename
        parser = ConfigParserEx()
        parser.read([self.filename])
        self.cache_path = parser.get_default("global", "cache path", os.path.dirname(self.filename))
        self.cache_sub_path = parser.get_default("global", "cache sub path", ".cache")
        self.use_new_window = parser.getboolean_default("global", "use new window", False)
        self.show_banner = parser.getboolean_default("global", "show banner", False)
        self.autorefresh_time = parser.get_default("global", "autorefresh time", "")
        self.connection_id = parser.get_default("global", "connection id", "")
        self.autorefresh_checked = parser.getboolean_default("global", "autorefresh checked", False)
        self.is_interval = parser.getboolean_default("global", "is interval", False)
        self.browser = parser.get_default("global", "browser", "")
        self.list = []
        for section in parser.sections():
            if section != "global":
                feed_config = FeedConfig(
                    parser.get(section, "url"),
                    parser.get(section, "name").decode(locale.getpreferredencoding()),
                    parser.getint_default(section, "mode", FeedConfig.mode_ommit),
                    parser.get_default(section, "icon", ""),
                    parser.getint_default(section, "days", 7),
                    parser.get_default(section, "include", ""),
                    parser.get_default(section, "exclude", ""),
                    parser.getint_default(section, "message count", 0),
                    parser.get_default(section, "content tag", "").decode(locale.getpreferredencoding()),
                    parser.get_default(section, "reformatter url", ""),
                    parser.getboolean_default(section, "checked", False),
                    parser.getboolean_default(section, "ignore images", False),
                    section)
                feed_config.ord_num = parser.getint_default(section, "order number", 0)
                if parser.has_option(section, "print version keywords"):
                    feed_config.set_print_version_keywords_string(
                        parser.get(section, "print version keywords").decode(locale.getpreferredencoding()))
                if parser.has_option(section, "single page keywords"):
                    feed_config.set_single_page_keywords_string(
                        parser.get(section, "single page keywords").decode(locale.getpreferredencoding()))
                if parser.has_option(section, "next page keywords"):
                    feed_config.set_next_page_keywords_string(
                        parser.get(section, "next page keywords").decode(locale.getpreferredencoding()))
                self.list.append(feed_config)
        self.list.sort(lambda f1, f2: f1.ord_num.__cmp__(f2.ord_num))


    def save(self):
        f = open(self.filename, "w")
        try:
            parser = ConfigParserEx()
            parser.add_section("global")
            parser.set("global", "cache path", self.cache_path)
            parser.set("global", "cache sub path", self.cache_sub_path)
            parser.set("global", "use new window", str(self.use_new_window))
            parser.set("global", "show banner", str(self.show_banner))
            parser.set("global", "autorefresh time", self.autorefresh_time)
            parser.set("global", "connection id", self.connection_id)
            parser.set("global", "autorefresh checked", str(self.autorefresh_checked))
            parser.set("global", "is interval", str(self.is_interval))
            parser.set("global", "browser", self.browser)
            i = 0
            for feed_config in self.list:
                parser.add_section(feed_config.id)
                parser.set(feed_config.id, "url", feed_config.url)
                parser.set(feed_config.id, "name", feed_config.name.encode(locale.getpreferredencoding()))
                parser.set(feed_config.id, "mode", str(feed_config.mode))
                parser.set(feed_config.id, "days", str(feed_config.days_to_keep))
                parser.set(feed_config.id, "include", feed_config.include)
                parser.set(feed_config.id, "exclude", feed_config.exclude)
                parser.set(feed_config.id, "icon", feed_config.icon_filename)
                parser.set(feed_config.id, "message count", str(feed_config.message_count))
                parser.set(feed_config.id, "content tag", feed_config.content_tag.encode(locale.getpreferredencoding()))
                parser.set(feed_config.id, "reformatter url", feed_config.reformatter_url)
                parser.set(feed_config.id, "checked", str(feed_config.checked))
                parser.set(feed_config.id, "ignore images", str(feed_config.ignore_images))
                parser.set(feed_config.id, "order number", str(i))
                parser.set(feed_config.id, "print version keywords", feed_config.get_print_version_keywords_string().encode(locale.getpreferredencoding()))
                parser.set(feed_config.id, "single page keywords", feed_config.get_single_page_keywords_string().encode(locale.getpreferredencoding()))
                parser.set(feed_config.id, "next page keywords", feed_config.get_next_page_keywords_string().encode(locale.getpreferredencoding()))
                i += 1
            parser.write(f)
        finally:
            f.close()


