/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#ifndef AAPTINTERFACE_H
#define AAPTINTERFACE_H

#include <QtGui>
#include <QtNetwork>
#include "blacklistselect.h"


const QString KFapmanDir = "/root/.fapman";
const QString KLogFile = "/root/.fapman/lastlog.txt";
const QString KBlacklistFile = "/root/.fapman/black.list";
const QString KOwnRepoFile = "/root/.fapman/repos.list";
const QString KOwnRepoNamesFile = "/root/.fapman/repos.names";

const QString KDefaultUserOpenSaveDir = "/home/user/MyDocs";

#ifdef Q_WS_MAEMO_5
	const QString KHamRepoListFile = "/etc/apt/sources.list.d/hildon-application-manager.list";
#else
	const QString KHamRepoListFile = "/etc/apt/sources.list";
#endif

const QString KLastUpdateFile = "/opt/fapman-cache/lastupdate";
const QString KDateCacheFile = "/opt/fapman-cache/dates.cache";
const QString KAptSourceList = "/opt/fapman-cache/sources.list";
const QString KAptListDir = "/opt/fapman-cache/lists";
const QString KAptArchivePartialDir = "/var/cache/apt/archives/partial";
const QString KAptPreferencesFile = "/etc/apt/preferences";

const QString KDpkgStatusFile = "/var/lib/dpkg/status";
const QString KDpkgInfoDir = "/var/lib/dpkg/info";

const int KListExpireTime = 3600;

const int KDataReadBufferSize = 100000;

class MainWindow;
class FileReadThread;
class DateFetchThread;
class dimmer;
class Package;
class Repository;
class Settings;


class AAptInterface : public QObject
{
	Q_OBJECT

public:
	enum interfaceMode { ModeNone, ModeReadPackages, ModeFetchDates, ModeAptGetUpdate,
						 ModeAptGetSimulate, ModeAptGetInstall, ModeAptGetClean };
	enum multiLine { MultiLineNone, MultiLineDesc, MultiLineIcon, MultiLineUpgradeDesc };

	explicit AAptInterface(QObject* parent);
	virtual ~AAptInterface();
	void setSettings(Settings* s_) { iSettings = s_; }

	void addQueuedOperation(interfaceMode mode_);
	bool run(dimmer* uiDimmer);
	void setProcessPackages(QStringList pkgs) { iProcessPackages=pkgs; iProcessPackagesOrig=pkgs; }
	QStringList processPackages() const { return iProcessPackages; }
	QStringList processPackageVersions() const { return iProcessPackageVersions; }

	bool running() const;
	bool cancel();

	int numSelectedPackages() const { return iNumSelectedPackages; }
	void setNumSelectedPackages(int p_) { iNumSelectedPackages=p_; }

	const QHash<QString, Package*>* packagesAvailable() { return &iPackagesAvailable; }
	const QHash<QString, Package*>* packagesInstalled() { return &iPackagesInstalled; }
	QList<Repository*>* repositories() { return  &iRepositories; }

	QByteArray readLogFile();
	void readRepositoryInfo();
	bool writeRepositories();
	bool needRepoRefresh();
	void removeFromBlacklist(Package* pkg, BlacklistSelect::blackList oldstate);
	void writeBlacklist();

	bool dateCacheExists();

	void setNeedRefresh(int repos, int lists, int dpkg, int dates);
	void setSkipListAndDates() { iSkipRefreshListAndDates=true; }
	bool needListOrDateRefresh();

	QDateTime lastListUpdate() { return iLastListUpdate; }
	QDateTime lastDpkgUpdate() { return iLastDpkgUpdate; }

private:
	AAptInterface(const AAptInterface& old);
	AAptInterface operator= (const AAptInterface& old);

	void runNext();
	void cleanAfterRunEach();
	void cleanAfterRunAll();
	void cleanAfterError();

	bool startAptGetUpdate();
	bool startAptGetSimulate();
	bool startAptGetInstall();
	bool startAptGetClean();

	QString setQProcessErrorMessage(QProcess::ProcessError error);
	QString finishProcessCommonErrorMessages(QByteArray& output);

	void communicateStatusToUi(bool success, QString title="", QString msg="");
	void logToFile( QString data, bool logtime=true );
	void logToFile( QByteArray data, bool logtime=true );

	void startPkgListRead();
	void startFetchDates();

	Package* ReadNextPackage(QFile& f, quint64& currentreaddata);
	bool processPackageDataLine(Package*& pkg, QByteArray& line);

	void readBlacklist();
	void writeDateCache();
	void readDateCache();
	void readPinnedPackages();

private slots:
	void errorAptGetUpdate(QProcess::ProcessError error);
	void errorAptGetSimulate(QProcess::ProcessError error);
	void errorAptGetInstall(QProcess::ProcessError error);
	void errorAptGetClean(QProcess::ProcessError error);

	void finishedAptGetUpdate(int exitCode, QProcess::ExitStatus exitStatus);
	void finishedAptGetSimulate(int exitCode, QProcess::ExitStatus exitStatus);
	void finishedAptGetInstall(int exitCode, QProcess::ExitStatus exitStatus);
	void finishedAptGetClean(int exitCode, QProcess::ExitStatus exitStatus);

	void uiUpdaterAptGetUpdate();
	void uiUpdaterAptGetInstall();
	void progressCheckTimerCallback();

	void dateFetchNetworkReply(QNetworkReply* reply);

private:
	QHash<QString, Package*> iPackagesAvailable;
	QHash<QString, Package*> iPackagesInstalled;
	int iNumSelectedPackages;

	QStringList iProcessPackages;
	QStringList iProcessPackagesOrig;
	QStringList iProcessPackageVersions;
	QList<interfaceMode> iOperationsQueue;
	QList<Repository*> iRepositories;
	QStringList iQueueMessages;

	MainWindow* iMainWindow;
	Settings* iSettings;
	interfaceMode iMode;
	QList<interfaceMode> iModeLog;
	dimmer* iUiDimmer;
	bool iCanCancel;
	bool iTerminated;
	bool iErrorDone;
	bool iNeedRepoRefresh;
	bool iNeedListRefresh;
	bool iNeedDpkgRefresh;
	bool iNeedDateRefresh;
	bool iSkipRefreshListAndDates;
	QDateTime iLastListUpdate;
	QDateTime iLastDpkgUpdate;

	QProcess* iProcAptGetUpdate;
	QProcess* iProcAptGetSimulate;
	QProcess* iProcAptGetInstall;
	QProcess* iProcAptGetClean;
	QByteArray iProcAptGetUpdateOutput;
	QByteArray iProcAptGetInstallOutput;

	int iAptGetDownloadCount;
	int iAptGetInstallCount;
	int iAptGetRemoveCount;
	int iAptGetInstallTotal;
	int iAptGetRemoveTotal;
	qint64 iAptGetCurrentFileDownloadSize;
	qint64 iAptGetCurrentFileTotalSize;
	QString iAptGetCurrentDownloadFileName;
	QTimer* iProgressCheckTimer;
	int iSpeedKbps;
	int iSpeedKbpsPrev;
	bool iUpdateSpeed;

	int iCatalogCounter;
	int iCatalogsTotal;

	QStringList iBlacklist;

	multiLine iMultiLine;
	char* iDataReadBuffer;

	int iDateRequestsWaiting;
	int iDateRequestsSent;
	int iDateRequestsReceived;
	int iDateRequestErrors;
	QNetworkReply::NetworkError iNetworkError;
};


#endif // AAPTINTERFACE_H
