/* $Id: eyes.c 2397 2007-01-17 17:46:35Z nick $
 * 
 * Copyright (c) Benedikt Meurer <benedikt.meurer@unix-ag.uni-siegen.de>>
 * Copyright (c) Danny Milosavljevic <danny_milo@gmx.net>
 * Copyright (c) Dave Camp
 * Copyright (c) Davyd Madeley  <davyd@madeley.id.au>
 * Copyright (c) Nick Schermer <nick@xfce.org>
 * Copyright (c) Mikko Vartiainen <mvartiainen@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MATH_H
#include <math.h>
#endif

#include <sys/types.h>
#include <dirent.h>
#include <string.h>

#include <libhildondesktop/libhildondesktop.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>

#include "eyes.h"
#include "themes.h"
#include "accelerometer.h"

/* for xml: */
#define EYES_ROOT      "Eyes"
#define DEFAULTTHEME   "Default"
#define UPDATE_TIMEOUT 100

#define EYES_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,	\
								   EYES_TYPE_HOME_PLUGIN, \
								   EyesPluginContent))

HD_DEFINE_PLUGIN_MODULE (EyesPlugin, eyes_plugin,      HD_TYPE_HOME_PLUGIN_ITEM)

static void eyes_check_display(DBusGProxy *object, const char *status, EyesPluginContent *eyes);

/*****************************
 *** Eyes Plugin Functions ***
 *****************************/
static void
calculate_pupil_xy (EyesPluginContent *eyes_applet,
		    gint x, gint y,
		    gint *pupil_x, gint *pupil_y, GtkWidget* widget)
{
  double sina;
  double cosa;
  double h;
  double temp;
  double nx, ny;

  gfloat xalign, yalign;
  gint width, height;

  width = GTK_WIDGET(widget)->allocation.width;
  height = GTK_WIDGET(widget)->allocation.height;
  gtk_misc_get_alignment(GTK_MISC(widget),  &xalign, &yalign);

  nx = width*x/1000.0;
  ny = height*y/1000.0;

  h = hypot (nx, ny);

  sina = nx / h;
  cosa = ny / h;

  temp = hypot ((eyes_applet->eye_width / 2) * sina, (eyes_applet->eye_height / 2) * cosa);
  temp -= hypot ((eyes_applet->pupil_width / 2) * sina, (eyes_applet->pupil_height / 2) * cosa);
  temp -= hypot ((eyes_applet->wall_thickness / 2) * sina, (eyes_applet->wall_thickness / 2) * cosa);

  if ( hypot(temp*sina, temp*cosa) > hypot( nx , ny ) )
    {
      *pupil_x = nx + eyes_applet->eye_width / 2;
      *pupil_y = ny + eyes_applet->eye_height / 2;
    }
  else 
    {
      *pupil_x = temp * sina + (eyes_applet->eye_width / 2);
      *pupil_y = temp * cosa + (eyes_applet->eye_height / 2);
    }
}

static void
draw_eye (EyesPluginContent *eyes,
          gint    eye_num,
          gint    pupil_x,
          gint    pupil_y)
{
  GdkPixbuf *pixbuf;
  GdkRectangle rect, r1, r2;

  pixbuf = gdk_pixbuf_copy (eyes->eye_image);
  r1.x = pupil_x - eyes->pupil_width / 2;
  r1.y = pupil_y - eyes->pupil_height / 2;
  r1.width = eyes->pupil_width;
  r1.height = eyes->pupil_height;
  r2.x = 0;
  r2.y = 0;
  r2.width = eyes->eye_width;
  r2.height = eyes->eye_height;
  if (gdk_rectangle_intersect (&r1, &r2, &rect))
    {
      gdk_pixbuf_composite (eyes->pupil_image, pixbuf,
			    rect.x,
			    rect.y,
			    rect.width,
			    rect.height,
			    pupil_x - eyes->pupil_width / 2,
			    pupil_y - eyes->pupil_height / 2, 1.0, 1.0,
			    GDK_INTERP_BILINEAR,
			    255);
      gtk_image_set_from_pixbuf (GTK_IMAGE (eyes->eyes[eye_num]),
				 pixbuf);
    }
  g_object_unref (G_OBJECT (pixbuf));
}

static gint
timer_cb(EyesPluginContent *eyes)
{
  gint x, y, z;
  gint pupil_x, pupil_y;
  gint i;

  for (i = 0; i < eyes->num_eyes; i++)
    {
      if (GTK_WIDGET_REALIZED(eyes->eyes[i]))
        {
	  accel_read(&x, &y, &z);
	  calculate_pupil_xy (eyes, -x, -y, &pupil_x, &pupil_y, eyes->eyes[i]);

	  if ((pupil_x != eyes->pointer_last_x[i]) || (pupil_y != eyes->pointer_last_y[i]))
            {

	      draw_eye (eyes, i, pupil_x, pupil_y);

	      eyes->pointer_last_x[i] = pupil_x;
	      eyes->pointer_last_y[i] = pupil_y;
            }
        }
    }

  return TRUE;
}

static void
properties_load(EyesPluginContent *eyes)
{
  gchar *path;

  if (eyes->active_theme)
    path = g_build_filename(THEMESDIR, eyes->active_theme, NULL);
  else
    path = g_build_filename(THEMESDIR, DEFAULTTHEME, NULL);

  load_theme(eyes, path);

  g_free(path);
}



static void
setup_eyes(EyesPluginContent *eyes)
{
  g_warning ("setup_eyes");
  int i;

  if (eyes->hbox != NULL)
    {
      gtk_widget_destroy(eyes->hbox);
      eyes->hbox = NULL;
    }

  eyes->hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(eyes->align), GTK_WIDGET(eyes->hbox));

  eyes->eyes = g_new0 (GtkWidget *, eyes->num_eyes);
  eyes->pointer_last_x = g_new0 (gint, eyes->num_eyes);
  eyes->pointer_last_y = g_new0 (gint, eyes->num_eyes);

  for (i = 0; i < eyes->num_eyes; i++)
    {
      eyes->eyes[i] = gtk_image_new ();

      gtk_widget_set_size_request(GTK_WIDGET(eyes->eyes[i]),
				  eyes->eye_width,
				  eyes->eye_height);

      gtk_widget_show(eyes->eyes[i]);

      gtk_box_pack_start(GTK_BOX(eyes->hbox), eyes->eyes[i],
			 FALSE, FALSE, 0);

      if ((eyes->num_eyes != 1) && (i == 0))
	gtk_misc_set_alignment (GTK_MISC (eyes->eyes[i]), 1.0, 0.5);
      else if ((eyes->num_eyes != 1) && (i == eyes->num_eyes - 1))
	gtk_misc_set_alignment (GTK_MISC (eyes->eyes[i]), 0.0, 0.5);
      else
	gtk_misc_set_alignment (GTK_MISC (eyes->eyes[i]), 0.5, 0.5);

      eyes->pointer_last_x[i] = G_MAXINT;
      eyes->pointer_last_y[i] = G_MAXINT;

      draw_eye (eyes, i,
		eyes->eye_width / 2,
		eyes->eye_height / 2);
    }

  gtk_widget_show(eyes->hbox);
}



static gboolean
eyes_applet_fill(EyesPluginContent *eyes)
{
  gtk_widget_show_all(GTK_WIDGET(eyes->align));

  if (eyes->timeout_id == 0)
    {
      eyes->timeout_id = g_timeout_add (UPDATE_TIMEOUT,
					(GtkFunction)timer_cb, eyes);
    }

  return TRUE;
}

/******************************
 *** Panel Plugin Functions ***
 ******************************/
static void
eyes_free_data(EyesPluginContent      *eyes)
{
  g_warning("eyes_free_data");
  g_return_if_fail(eyes != NULL);

  if (eyes->timeout_id != 0)
    g_source_remove (eyes->timeout_id);

  g_free (eyes->eyes);
  g_free (eyes->pointer_last_x);
  g_free (eyes->pointer_last_y);

  if (eyes->active_theme != NULL)
    g_free (eyes->active_theme);

  if (eyes->eye_image != NULL)
    g_object_unref (G_OBJECT (eyes->eye_image));

  if (eyes->pupil_image != NULL)
    g_object_unref (G_OBJECT (eyes->pupil_image));

  if (eyes->theme_dir != NULL)
    g_free(eyes->theme_dir);

  if (eyes->theme_name != NULL)
    g_free(eyes->theme_name);

  if (eyes->eye_filename != NULL)
    g_free(eyes->eye_filename);

  if (eyes->pupil_filename != NULL)
    g_free(eyes->pupil_filename);

  if (eyes->proxy != NULL)
    dbus_g_proxy_disconnect_signal (eyes->proxy, MCE_DISPLAY_SIG, G_CALLBACK(eyes_check_display), eyes);
  
  g_free(eyes);
}

static void
eyes_read_rc_file (EyesPluginContent      *eyes)
{
  g_warning ("eyes_read_rc");
  if (eyes->active_theme == NULL)
    eyes->active_theme = g_strdup (DEFAULTTHEME);
}

static EyesPluginContent *
eyes_plugin_new ()
{
  g_warning ("eyes_plugin_new");
  EyesPluginContent *eyes;

  eyes = g_new0(EyesPluginContent, 1);

  eyes->align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);

  gtk_widget_show(GTK_WIDGET(eyes->align));

  eyes_read_rc_file (eyes);

  properties_load(eyes);
  setup_eyes(eyes);
  eyes_applet_fill(eyes);

  eyes->visible = TRUE;

  return eyes;
}

static void
eyes_check_desktop (GObject *gobject, GParamSpec *pspec, EyesPluginContent *eyes)

{
  g_warning ("eyes_check_desktop");
  gchar *name = pspec->name;
  gboolean status;
  g_object_get (gobject, name, &status, NULL);
  if (status) {
    eyes_applet_fill(eyes);
    eyes->visible = TRUE;
  } else if (eyes->timeout_id != 0) {
    g_source_remove (eyes->timeout_id);
    eyes->timeout_id = 0;
    eyes->visible = FALSE;
  }
}

static void
eyes_check_display(DBusGProxy *object, const char *status, EyesPluginContent *eyes)
{
  if (strncmp(status, MCE_DISPLAY_ON_STRING,
	      strlen(MCE_DISPLAY_ON_STRING) + 1) == 0
      && eyes-> visible)
    {
      eyes_applet_fill(eyes);
    }
  else if (strncmp(status, MCE_DISPLAY_OFF_STRING,
		   strlen(MCE_DISPLAY_OFF_STRING) + 1) == 0)
    {
      if (eyes->timeout_id != 0)
	{
	  g_source_remove (eyes->timeout_id);
	  eyes->timeout_id = 0;
	}
    }
  
}

static void
eyes_plugin_dispose (GObject *object)
{
  g_warning ("eyes_plugin_dispose");

  G_OBJECT_CLASS (eyes_plugin_parent_class)->dispose (object);
}

static void eyes_plugin_finalize (GObject *object)
{
  g_warning ("eyes_plugin_finalize");
  EyesPlugin *self = EYES_HOME_PLUGIN (object);

  eyes_free_data(self->priv);

  G_OBJECT_CLASS (eyes_plugin_parent_class)->finalize (object);
}

static void
eyes_plugin_realize (GtkWidget *widget)
{
  g_warning ("eyes_plugin_realize");
  GdkScreen *screen = gtk_widget_get_screen (widget);
  gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));
  gtk_widget_set_app_paintable (widget, TRUE);

  GTK_WIDGET_CLASS (eyes_plugin_parent_class)->realize (widget);
}

static gboolean
eyes_plugin_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
  g_warning ("eyes_plugin_expose_event");
  cairo_t *cr;

  cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
  gdk_cairo_region (cr, event->region);
  cairo_clip (cr);

  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (cr, 0.0, 0.0, 0.0, 0.0);
  cairo_paint (cr);
  
  cairo_destroy (cr);

  return GTK_WIDGET_CLASS (eyes_plugin_parent_class)->expose_event (widget, event);
}

static void
eyes_plugin_init (EyesPlugin *desktop_plugin)
{
  g_warning ("eyes_plugin_init");
  EyesPluginContent *eyes;

  eyes = eyes_plugin_new (desktop_plugin);
  desktop_plugin->priv = eyes;

  g_signal_connect (desktop_plugin, "notify::is-on-current-desktop",
		    G_CALLBACK (eyes_check_desktop), eyes);

  eyes->dbus_conn = NULL;
  eyes->proxy = NULL;

  eyes->dbus_conn = hd_home_plugin_item_get_dbus_g_connection ( &desktop_plugin->hitem, DBUS_BUS_SYSTEM, NULL );
  if (eyes->dbus_conn != NULL)
    {
      eyes->proxy = dbus_g_proxy_new_for_name(eyes->dbus_conn, MCE_SERVICE, MCE_SIGNAL_PATH, MCE_SIGNAL_IF);
      dbus_g_proxy_add_signal (eyes->proxy, MCE_DISPLAY_SIG, G_TYPE_STRING, G_TYPE_INVALID );
      dbus_g_proxy_connect_signal(eyes->proxy, MCE_DISPLAY_SIG, G_CALLBACK(eyes_check_display),eyes,NULL);
    }
  
  gtk_container_add (GTK_CONTAINER (desktop_plugin), eyes->align);
} 

static void
eyes_plugin_class_init (EyesPluginClass *klass) {
  g_warning ("eyes_plugin_class_init");
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = eyes_plugin_dispose;
  object_class->finalize = eyes_plugin_finalize;

  widget_class->realize = eyes_plugin_realize;
  widget_class->expose_event = eyes_plugin_expose_event;
  g_type_class_add_private (klass, sizeof (EyesPluginContent));
} 

static void
eyes_plugin_class_finalize (EyesPluginClass *class) {} 

