/*
 *  desktop switcher widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>

#include <X11/extensions/XTest.h>
#include <gconf/gconf-client.h>	
#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include "desktop-switcher.h"

#define DESKTOP_SWITCHER_HOME_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), DESKTOP_SWITCHER_TYPE_HOME_PLUGIN, DesktopSwitcherHomePluginPrivate))

HD_DEFINE_PLUGIN_MODULE(DesktopSwitcherHomePlugin, desktop_switcher_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

#define DEFAULT_WIDGET_WIDTH 96
#define DEFAULT_WIDGET_HEIGHT 96
#define SWITCH_TO_LEFT -2
#define SWITCH_TO_RIGHT -1
#define DESKTOP_SWITCHER_HOME_PLUGIN_SETTINGS_FILE "config"
#define DESKTOP_SWITCHER_HOME_PLUGIN_SETTINGS_FOLDER ".desktop_switcher"
#define DESKTOP_SWITCHER_CONFIG_KEY_WIDTH "width"
#define DESKTOP_SWITCHER_CONFIG_KEY_HEIGHT "height"
#define DESKTOP_SWITCHER_CONFIG_KEY_ALPHA "alpha"
#define DESKTOP_SWITCHER_CONFIG_KEY_SWITCH_TO "switch_to"
#define DESKTOP_SWITCHER_CONFIG_KEY_ICON_PATH "icon_path"
#define DESKTOP_SWITCHER_CONFIG_KEY_SCROLL_TYPE "scroll_type"

#define DEFAULT_LEFT_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_left.png"
#define DEFAULT_RIGHT_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_right.png"
#define DEFAULT_09_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_09.png"
#define DEFAULT_08_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_08.png"
#define DEFAULT_07_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_07.png"
#define DEFAULT_06_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_06.png"
#define DEFAULT_05_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_05.png"
#define DEFAULT_04_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_04.png"
#define DEFAULT_03_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_03.png"
#define DEFAULT_02_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_02.png"
#define DEFAULT_01_ICON_PATH "/usr/share/pixmaps/desktop-switcher/switch_desktop_01.png"
#define ACTIVE_VIEWS_GCONF_PATH "/apps/osso/hildon-desktop/views/active"
#define CURRENT_VIEW_GCONF_PATH "/apps/osso/hildon-desktop/views/current"

struct _DesktopSwitcherHomePluginPrivate
{
  gchar* iD;
  guint width;
  guint height;
  guint alpha;
  gint switch_to;
  gboolean smooth;
  gchar* icon_path;
  GdkPixbuf* background_pixbuf;
  guint icon_width;
  guint icon_height;
  gboolean pressed;
};

typedef struct _swipe_data_t
{
  GdkWindow* window;
  guint32 time;
} swipe_data_t;

static gboolean
fake_swipe(gpointer data)
{
  swipe_data_t* swipe_data = (swipe_data_t*)data;
  Display* dpy = GDK_WINDOW_XDISPLAY(swipe_data->window);
  gdk_display_warp_pointer(gdk_display_get_default(), 
			   gdk_display_get_default_screen(gdk_display_get_default()), 
			   1, 320);
  XTestFakeButtonEvent(dpy, 1, 1, 0);
  XTestFakeMotionEvent(dpy, gdk_x11_get_default_screen(), 800, 320, 0);
  XTestFakeButtonEvent(dpy, 1, 0, 0);
  
  g_free(swipe_data);
  return FALSE;
 }

static int
_current_view()
{
  GConfClient* client = NULL;
  GError* error = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    return 0;
  }
  int current_view = gconf_client_get_int(client, CURRENT_VIEW_GCONF_PATH, &error);
  if(error)
  {
    g_warning("can not get current view (%s)\n", error->message);
    g_error_free(error);
    error = NULL;
    current_view = 0;
  }
  g_object_unref(client);
  return current_view;
}

static GSList*
_active_views()
{
  GConfClient* client = NULL;
  GError* error = NULL;
  client = gconf_client_get_default();
  if(!GCONF_IS_CLIENT(client))
  {
    return NULL;
  }

  GSList* active_views = gconf_client_get_list(client, ACTIVE_VIEWS_GCONF_PATH, GCONF_VALUE_INT, &error);
  if(error)
  {
    g_warning("can not get list of active views (%s)\n", error->message);
    g_error_free(error);
    error = NULL;
    active_views = NULL;
  }
  g_object_unref(client);
  return active_views;
}

static int
_switch_hop(DesktopSwitcherHomePlugin* desktop_plugin)
{
  int switch_to_desktop = desktop_plugin->priv->switch_to;
  int current_view_index = 0;
  int switch_to_desktop_index = 0;
  int active_views_count = 0;
  GSList* active_views = _active_views();
  int current_view = _current_view();
  if(active_views)
  {
    int index = 0;
    GSList* active_view = active_views;
    active_views_count = g_slist_length(active_views);
    while(active_view)
    {
      int view = GPOINTER_TO_INT(active_view->data);
      if(view == current_view)
	current_view_index = index;
      if(view == switch_to_desktop)
	switch_to_desktop_index = index;
      ++index;
      active_view = active_view->next;
    }
    
    int hop = switch_to_desktop_index - current_view_index;
    if(hop>0)
    {
      if(hop > (int)(active_views_count/2))
      {
	hop-=active_views_count;
      }
    }
    if(hop<0)
    {
      if(hop < (int)(-active_views_count/2))
      {
	hop+=active_views_count;
      }
    }
    return hop;
  }
  return 0;
}

static void
_switch_desktop_left(Display* dpy, int times, gboolean smooth)
{
  int i;
  if(!smooth)
  {
    for(i = 0;i<times;++i)
    {
      gdk_display_warp_pointer(gdk_display_get_default(), 
			       gdk_display_get_default_screen(gdk_display_get_default()), 
			       0, 410);
      XTestFakeButtonEvent(dpy, 1, 1, 0);
      XTestFakeMotionEvent(dpy, gdk_x11_get_default_screen(), 800, 410, 0);
      XTestFakeButtonEvent(dpy, 1, 0, 0);
      XSync(dpy, 0);
    }
  }
  else
  {
    int delay = 15;
    for(i=0;i<times;++i)
    {
      gdk_display_warp_pointer(gdk_display_get_default(), 
			       gdk_display_get_default_screen(gdk_display_get_default()), 
			       0, 420);
      XTestFakeButtonEvent(dpy, 1, 1, 0);
      int t;
      for(t=0;t<16;++t)
      {
	XTestFakeMotionEvent(dpy, gdk_x11_get_default_screen(), t*50, 420, delay);
      }
      XTestFakeButtonEvent(dpy, 1, 0, delay);
      XSync(dpy,0);
    }
  }
}

static void
_switch_desktop_right(Display* dpy, int times, gboolean smooth)
{
  int i;
  if(!smooth)
  {
    for(i = 0;i<times;++i)
    {
      gdk_display_warp_pointer(gdk_display_get_default(), 
			       gdk_display_get_default_screen(gdk_display_get_default()), 
			       800, 420);
      XTestFakeButtonEvent(dpy, 1, 1, i);
      XTestFakeMotionEvent(dpy, gdk_x11_get_default_screen(), 0, 420, i);
      XTestFakeButtonEvent(dpy, 1, 0, i);
      XSync(dpy, 0);
    }
  }
  else
  {
    int delay = 20;
    for(i=0;i<times;++i)
    {
      gdk_display_warp_pointer(gdk_display_get_default(), 
			       gdk_display_get_default_screen(gdk_display_get_default()), 
			       800, 420);
      XTestFakeButtonEvent(dpy, 1, 1, 0);
      int t;
      for(t=0;t<16;++t)
      {
	XTestFakeMotionEvent(dpy, gdk_x11_get_default_screen(), 800-t*50, 420, delay);
      }
      XTestFakeButtonEvent(dpy, 1, 0, delay);
      XSync(dpy, 0);
    }
  }
}

static gboolean
_on_button_release(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  DesktopSwitcherHomePlugin* desktop_plugin = DESKTOP_SWITCHER_HOME_PLUGIN(user_data);
  Display* dpy = GDK_WINDOW_XDISPLAY(event->window);
  if(desktop_plugin->priv->switch_to == SWITCH_TO_LEFT)
  {
    _switch_desktop_left(dpy, 1, desktop_plugin->priv->smooth);
  }
  else if(desktop_plugin->priv->switch_to == SWITCH_TO_RIGHT)
  {
    _switch_desktop_right(dpy, 1, desktop_plugin->priv->smooth);
  }
  else
  {
    int hops = _switch_hop(desktop_plugin);
    if(hops<0)
      _switch_desktop_left(dpy, -hops, desktop_plugin->priv->smooth);
    else
      _switch_desktop_right(dpy, hops, desktop_plugin->priv->smooth);
  }
  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  return TRUE;
}

static gboolean
_on_button_press(GtkWidget* widget, GdkEventButton* event, gpointer user_data)
{
  DesktopSwitcherHomePlugin* desktop_plugin = DESKTOP_SWITCHER_HOME_PLUGIN(user_data);
  desktop_plugin->priv->pressed = TRUE;
  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  return TRUE;
}

static void
_switcher_load_icon(DesktopSwitcherHomePlugin* desktop_plugin)
{
  if(desktop_plugin->priv->background_pixbuf)
    g_object_unref(desktop_plugin->priv->background_pixbuf);
  
  desktop_plugin->priv->background_pixbuf = gdk_pixbuf_new_from_file(desktop_plugin->priv->icon_path, NULL);
  if(desktop_plugin->priv->background_pixbuf)
  {

    desktop_plugin->priv->icon_width = gdk_pixbuf_get_width(desktop_plugin->priv->background_pixbuf);
    desktop_plugin->priv->icon_height = gdk_pixbuf_get_height(desktop_plugin->priv->background_pixbuf);
  }
  else
  {
    desktop_plugin->priv->icon_width = 0;
    desktop_plugin->priv->icon_height = 0;
  }
}

static gboolean
desktop_switcher_expose(GtkWidget* widget, GdkEventExpose* event)
{
  DesktopSwitcherHomePlugin* desktop_plugin = DESKTOP_SWITCHER_HOME_PLUGIN(widget);
  if(desktop_plugin->priv->background_pixbuf == NULL)
    _switcher_load_icon(desktop_plugin);
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  if(desktop_plugin->priv->width != desktop_plugin->priv->icon_width ||
     desktop_plugin->priv->height != desktop_plugin->priv->icon_height)
  {
    cairo_scale(cr, 
		(double)desktop_plugin->priv->width/desktop_plugin->priv->icon_width,
		(double)desktop_plugin->priv->height/desktop_plugin->priv->icon_height);
  }
  if(desktop_plugin->priv->pressed)
  {
    cairo_scale(cr, 0.90, 0.90);
    cairo_translate(cr, 
		    (double)desktop_plugin->priv->icon_width*0.05,
		    (double)desktop_plugin->priv->icon_height*0.05);
    desktop_plugin->priv->pressed = FALSE;
  }
  if(desktop_plugin->priv->background_pixbuf)
    gdk_cairo_set_source_pixbuf(cr, desktop_plugin->priv->background_pixbuf, 0, 0);
  cairo_paint_with_alpha(cr, (256-desktop_plugin->priv->alpha)/256.0);
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(desktop_switcher_home_plugin_parent_class)->expose_event(widget, event);
}

static void
_read_settings(DesktopSwitcherHomePlugin* desktop_plugin)
{
  GKeyFile *key_file;
  gchar* file_name;
  gboolean file_exists;

  key_file = g_key_file_new();
  file_name = g_build_filename(g_get_home_dir(), 
			       DESKTOP_SWITCHER_HOME_PLUGIN_SETTINGS_FOLDER,
			       DESKTOP_SWITCHER_HOME_PLUGIN_SETTINGS_FILE, NULL);
  file_exists = g_key_file_load_from_file (key_file, file_name, G_KEY_FILE_KEEP_COMMENTS, NULL);
  if(file_exists)
  {
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_SWITCHER_CONFIG_KEY_WIDTH, 
			  NULL))
    {
      desktop_plugin->priv->width = g_key_file_get_integer(key_file,
							   desktop_plugin->priv->iD,
							   DESKTOP_SWITCHER_CONFIG_KEY_WIDTH, 
							   NULL);
    }
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_SWITCHER_CONFIG_KEY_HEIGHT, 
			  NULL))
    {
      desktop_plugin->priv->height = g_key_file_get_integer(key_file,
							    desktop_plugin->priv->iD,
							    DESKTOP_SWITCHER_CONFIG_KEY_HEIGHT, 
							    NULL);
    }
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_SWITCHER_CONFIG_KEY_ALPHA, 
			  NULL))
    {
      desktop_plugin->priv->alpha = g_key_file_get_integer(key_file,
							   desktop_plugin->priv->iD,
							   DESKTOP_SWITCHER_CONFIG_KEY_ALPHA, 
							   NULL);
    }
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_SWITCHER_CONFIG_KEY_SWITCH_TO, 
			  NULL))
    {
      desktop_plugin->priv->switch_to = g_key_file_get_integer(key_file,
							       desktop_plugin->priv->iD,
							       DESKTOP_SWITCHER_CONFIG_KEY_SWITCH_TO, 
							       NULL);
    }
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_SWITCHER_CONFIG_KEY_SCROLL_TYPE, 
			  NULL))
    {
      if(g_key_file_get_integer(key_file,
				desktop_plugin->priv->iD,
				DESKTOP_SWITCHER_CONFIG_KEY_SCROLL_TYPE,
				NULL) == 0)
	desktop_plugin->priv->smooth = TRUE;
      else
	desktop_plugin->priv->smooth = FALSE;
    }
    if(g_key_file_has_key(key_file, desktop_plugin->priv->iD, 
			  DESKTOP_SWITCHER_CONFIG_KEY_ICON_PATH, 
			  NULL))
    {
      if(desktop_plugin->priv->icon_path)
	g_free(desktop_plugin->priv->icon_path);
      desktop_plugin->priv->icon_path = g_key_file_get_string(key_file,
							      desktop_plugin->priv->iD,
							      DESKTOP_SWITCHER_CONFIG_KEY_ICON_PATH, 
							      NULL);
    }
  }
  g_key_file_free(key_file);
  g_free(file_name);
}

static void
_write_settings(DesktopSwitcherHomePlugin* desktop_plugin)
{
  GKeyFile *key_file;
  gchar* file_name;
  gchar *file_data;
  gsize size;
  gchar* config_file_folder = g_build_filename(g_get_home_dir(), 
					       DESKTOP_SWITCHER_HOME_PLUGIN_SETTINGS_FOLDER, 
					       NULL);

  if(!g_file_test(config_file_folder, (G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)))
  {
    if(g_mkdir(config_file_folder, 0755)!=0)
    {
      g_free(config_file_folder);
      return;
    }
  }
  key_file = g_key_file_new();
  file_name = g_build_filename(config_file_folder, 
			       DESKTOP_SWITCHER_HOME_PLUGIN_SETTINGS_FILE, 
			       NULL);
  g_key_file_load_from_file (key_file, file_name, G_KEY_FILE_KEEP_COMMENTS, NULL);
  
  g_key_file_set_integer(key_file,
			 desktop_plugin->priv->iD,
			 DESKTOP_SWITCHER_CONFIG_KEY_WIDTH, 
			 desktop_plugin->priv->width);
  g_key_file_set_integer(key_file,
			 desktop_plugin->priv->iD,
			 DESKTOP_SWITCHER_CONFIG_KEY_HEIGHT, 
			 desktop_plugin->priv->height);
  g_key_file_set_integer(key_file,
			 desktop_plugin->priv->iD,
			 DESKTOP_SWITCHER_CONFIG_KEY_ALPHA, 
			 desktop_plugin->priv->alpha);
  g_key_file_set_integer(key_file,
			 desktop_plugin->priv->iD,
			 DESKTOP_SWITCHER_CONFIG_KEY_SWITCH_TO, 
			 desktop_plugin->priv->switch_to);
  if(desktop_plugin->priv->smooth)
    g_key_file_set_integer(key_file,
			   desktop_plugin->priv->iD,
			   DESKTOP_SWITCHER_CONFIG_KEY_SCROLL_TYPE, 
			   0);
  else
    g_key_file_set_integer(key_file,
			   desktop_plugin->priv->iD,
			   DESKTOP_SWITCHER_CONFIG_KEY_SCROLL_TYPE, 
			   1);


  g_key_file_set_string(key_file,
			desktop_plugin->priv->iD,
			DESKTOP_SWITCHER_CONFIG_KEY_ICON_PATH, 
			desktop_plugin->priv->icon_path);
  file_data = g_key_file_to_data (key_file, &size, NULL);
  g_file_set_contents(file_name, file_data, size, NULL);
  
  g_key_file_free (key_file); 
  g_free(file_name); 
  g_free(config_file_folder); 
  g_free(file_data); 
}

static void
desktop_switcher_realize(GtkWidget* widget)
{
  DesktopSwitcherHomePlugin* desktop_plugin = DESKTOP_SWITCHER_HOME_PLUGIN(widget);
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  desktop_plugin->priv->iD = hd_home_plugin_item_get_applet_id(HD_HOME_PLUGIN_ITEM(widget));
  _read_settings(desktop_plugin);
  gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin), 
			      desktop_plugin->priv->width,
			      desktop_plugin->priv->height);
  gtk_window_resize(GTK_WINDOW(desktop_plugin), 
		    desktop_plugin->priv->width,
		    desktop_plugin->priv->height);

  GTK_WIDGET_CLASS(desktop_switcher_home_plugin_parent_class)->realize(widget);
}

static void
desktop_switcher_finalize(GObject* object)
{
  DesktopSwitcherHomePlugin* desktop_plugin = DESKTOP_SWITCHER_HOME_PLUGIN(object);
  g_free(desktop_plugin->priv->icon_path);
  if(desktop_plugin->priv->background_pixbuf != NULL)
    g_object_unref(desktop_plugin->priv->background_pixbuf);
  g_free(desktop_plugin->priv->iD);
  G_OBJECT_CLASS(desktop_switcher_home_plugin_parent_class)->finalize(object);
}

static GtkListStore*
_create_desktop_list(DesktopSwitcherHomePlugin* desktop_plugin, int* row_for_this_desktop)
{
  GtkListStore* list_store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
  GtkTreeIter iter;
  int index = 0;
  if(desktop_plugin->priv->switch_to == SWITCH_TO_RIGHT)
    *row_for_this_desktop = 1;
  else
    *row_for_this_desktop = 0;

  gtk_list_store_append(list_store, &iter);
  gtk_list_store_set(list_store, &iter,
		     0, "left", 
		     1, SWITCH_TO_LEFT,
		     -1);
  index++;
  gtk_list_store_append(list_store, &iter);
  gtk_list_store_set(list_store, &iter,
		     0, "right", 
		     1, SWITCH_TO_RIGHT,
		     -1);
  index++;

  GSList* active_views = _active_views();
  GSList* active_view = active_views;
  while(active_view)
  {
    int view_number = GPOINTER_TO_INT(active_view->data);
    gtk_list_store_append(list_store, &iter);
    gchar* view_number_c = g_strdup_printf("%d", view_number); 
    gtk_list_store_set(list_store, &iter,
		       0, view_number_c,
		       1, view_number,
		       -1);
    g_free(view_number_c);
    if(view_number == desktop_plugin->priv->switch_to)
      *row_for_this_desktop = index;
    index++;
    active_view = active_view->next;
  }
  if(active_views)
    g_slist_free(active_views);
  return list_store;
}

static void
_increment_slider(GtkButton* button, gpointer user_data)
{
  GtkRange* range = GTK_RANGE(user_data);
  gtk_range_set_value(range, gtk_range_get_value(range) + 1);
}

static void
_decrement_slider(GtkButton* button, gpointer user_data)
{
  GtkRange* range = GTK_RANGE(user_data);
  gtk_range_set_value(range, gtk_range_get_value(range) - 1);
}

static void
_show_settings_dialog(GtkWidget* widget, gpointer data)
{
  DesktopSwitcherHomePlugin* desktop_plugin = DESKTOP_SWITCHER_HOME_PLUGIN(widget);
  GtkWidget* dialog;
  GtkWidget* switch_to_selector_button;
  GtkWidget* fast_scroll_toggle_button;
  GtkWidget* smooth_scroll_toggle_button;
  GtkWidget* width_slider;
  GtkWidget* height_slider;
  GtkWidget* alpha_slider;

  GtkWidget* wlabel = gtk_label_new("Width:");
  GtkWidget* hlabel = gtk_label_new("Height:");
  GtkWidget* alabel = gtk_label_new("Transparency:");

  dialog = gtk_dialog_new_with_buttons("Settings",
				       NULL,
				       0,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_OK,
				       NULL);
  switch_to_selector_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
						       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  GtkWidget* selector = hildon_touch_selector_new_text();
  int row_for_this_desktop = 0;
  GtkListStore* desktop_list = _create_desktop_list(desktop_plugin, &row_for_this_desktop);

  hildon_touch_selector_set_model(HILDON_TOUCH_SELECTOR(selector), 0, GTK_TREE_MODEL(desktop_list));

  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(switch_to_selector_button),
				    HILDON_TOUCH_SELECTOR(selector));

  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(selector), 0, row_for_this_desktop);
  fast_scroll_toggle_button = hildon_gtk_radio_button_new(HILDON_SIZE_FINGER_HEIGHT, NULL);
  smooth_scroll_toggle_button = hildon_gtk_radio_button_new_from_widget(HILDON_SIZE_FINGER_HEIGHT, 
									GTK_RADIO_BUTTON(fast_scroll_toggle_button));
  gtk_button_set_label(GTK_BUTTON(fast_scroll_toggle_button), "fast scroll");
  gtk_button_set_label(GTK_BUTTON(smooth_scroll_toggle_button), "smooth scroll");
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(fast_scroll_toggle_button), FALSE);
  gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(smooth_scroll_toggle_button), FALSE);

  if(desktop_plugin->priv->smooth)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(smooth_scroll_toggle_button), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fast_scroll_toggle_button), TRUE);

  width_slider = hildon_gtk_hscale_new();
  gtk_adjustment_configure(gtk_range_get_adjustment(GTK_RANGE(width_slider)),
			   desktop_plugin->priv->width,
			   32, 400,
			   1, 1, 1);

  gtk_scale_set_value_pos(GTK_SCALE(width_slider), GTK_POS_RIGHT);
  gtk_scale_set_draw_value(GTK_SCALE(width_slider), TRUE);
  gtk_scale_set_digits(GTK_SCALE(width_slider), 0);

  height_slider = hildon_gtk_hscale_new();
  gtk_adjustment_configure(gtk_range_get_adjustment(GTK_RANGE(height_slider)),
			   desktop_plugin->priv->height,
			   32, 420,
			   1, 1, 1);
  gtk_scale_set_value_pos(GTK_SCALE(height_slider), GTK_POS_RIGHT);
  gtk_scale_set_draw_value(GTK_SCALE(height_slider), TRUE);
  gtk_scale_set_digits(GTK_SCALE(height_slider), 0);

  alpha_slider = hildon_gtk_hscale_new();
  gtk_adjustment_configure(gtk_range_get_adjustment(GTK_RANGE(alpha_slider)),
			   desktop_plugin->priv->alpha,
			   0, 256,
			   1, 1, 1);
  gtk_scale_set_value_pos(GTK_SCALE(alpha_slider), GTK_POS_RIGHT);
  gtk_scale_set_draw_value(GTK_SCALE(alpha_slider), TRUE);
  gtk_scale_set_digits(GTK_SCALE(alpha_slider), 0);

  GtkWidget* decrement_width_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(decrement_width_button), "-");
  GtkWidget* increment_width_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(increment_width_button), "+");
  GtkWidget* decrement_height_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(decrement_height_button), "-");
  GtkWidget* increment_height_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(increment_height_button), "+");
  GtkWidget* decrement_alpha_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(decrement_alpha_button), "-");
  GtkWidget* increment_alpha_button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(increment_alpha_button), "+");

  g_signal_connect(decrement_width_button, "clicked", G_CALLBACK(_decrement_slider), width_slider);
  g_signal_connect(increment_width_button, "clicked", G_CALLBACK(_increment_slider), width_slider);
  g_signal_connect(decrement_height_button, "clicked", G_CALLBACK(_decrement_slider), height_slider);
  g_signal_connect(increment_height_button, "clicked", G_CALLBACK(_increment_slider), height_slider);
  g_signal_connect(decrement_alpha_button, "clicked", G_CALLBACK(_decrement_slider), alpha_slider);
  g_signal_connect(increment_alpha_button, "clicked", G_CALLBACK(_increment_slider), alpha_slider);

  
  GtkWidget* scroll_button_box = gtk_hbox_new(TRUE, 3);
  gtk_box_pack_start(GTK_BOX(scroll_button_box), fast_scroll_toggle_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(scroll_button_box), smooth_scroll_toggle_button, TRUE, TRUE, 0);

  GtkWidget* size_slider_button_box_width = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(size_slider_button_box_width), decrement_width_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(size_slider_button_box_width), width_slider, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(size_slider_button_box_width), increment_width_button, FALSE, FALSE, 0);
  
  GtkWidget* size_slider_button_box_height = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(size_slider_button_box_height), decrement_height_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(size_slider_button_box_height), height_slider, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(size_slider_button_box_height), increment_height_button, FALSE, FALSE, 0);
  
  GtkWidget* alpha_slider_button_box = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(alpha_slider_button_box), decrement_alpha_button, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(alpha_slider_button_box), alpha_slider, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(alpha_slider_button_box), increment_alpha_button, FALSE, FALSE, 0);

  GtkWidget* slider_label_box = gtk_vbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(slider_label_box), wlabel, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(slider_label_box), hlabel, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(slider_label_box), alabel, TRUE, TRUE, 0);

  GtkWidget* slider_size_box = gtk_vbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(slider_size_box), size_slider_button_box_width, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(slider_size_box), size_slider_button_box_height, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(slider_size_box), alpha_slider_button_box, TRUE, TRUE, 0);

  GtkWidget* slider_box = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(slider_box), slider_label_box, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(slider_box), slider_size_box, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), switch_to_selector_button, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), scroll_button_box, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), slider_box, TRUE, TRUE, 0);

  gtk_widget_show_all(dialog);

  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
  {
    GtkTreeIter iter;
    if(hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(selector),
					  0, &iter))
    {
      int selected_row = 0;
      gtk_tree_model_get(GTK_TREE_MODEL(desktop_list), &iter,
			 1, &selected_row,
			 -1);
      g_free(desktop_plugin->priv->icon_path);
      if(selected_row == SWITCH_TO_LEFT)
      {
	desktop_plugin->priv->switch_to = SWITCH_TO_LEFT;
	desktop_plugin->priv->icon_path = g_strdup(DEFAULT_LEFT_ICON_PATH);
      } 
      else if(selected_row == SWITCH_TO_RIGHT)
      {
	desktop_plugin->priv->switch_to = SWITCH_TO_RIGHT;
	desktop_plugin->priv->icon_path = g_strdup(DEFAULT_RIGHT_ICON_PATH);
      }
      else
      {
	desktop_plugin->priv->switch_to = selected_row;
	switch(desktop_plugin->priv->switch_to)
	{
	case 9:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_09_ICON_PATH);
	  break;
	case 8:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_08_ICON_PATH);
	  break;
	case 7:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_07_ICON_PATH);
	  break;
	case 6:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_06_ICON_PATH);
	  break;
	case 5:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_05_ICON_PATH);
	  break;
	case 4:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_04_ICON_PATH);
	  break;
	case 3:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_03_ICON_PATH);
	  break;
	case 2:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_02_ICON_PATH);
	  break;
	default:
	  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_01_ICON_PATH);
	}
      }
      _switcher_load_icon(desktop_plugin);
      if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(smooth_scroll_toggle_button)))
	desktop_plugin->priv->smooth = TRUE;
      else
	desktop_plugin->priv->smooth = FALSE;
    }

    int new_width = gtk_range_get_value(GTK_RANGE(width_slider));
    int new_height = gtk_range_get_value(GTK_RANGE(height_slider));
    gboolean resize = new_width!=desktop_plugin->priv->width ||
      new_height!=desktop_plugin->priv->height;

    desktop_plugin->priv->width = new_width;
    desktop_plugin->priv->height = new_height;
    desktop_plugin->priv->alpha= gtk_range_get_value(GTK_RANGE(alpha_slider));
    if(resize)
    {
      gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin), 
				  desktop_plugin->priv->width,
				  desktop_plugin->priv->height);
				  
      gtk_window_resize(GTK_WINDOW(desktop_plugin), 
			desktop_plugin->priv->width,
			desktop_plugin->priv->height);
    }


    _write_settings(desktop_plugin);
  }
  g_object_unref(desktop_list);
  gtk_widget_destroy (dialog);
  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
}

static void
_on_current_desktop(GtkWidget* widget,
		    gpointer data)
{
  DesktopSwitcherHomePlugin* desktop_plugin = DESKTOP_SWITCHER_HOME_PLUGIN(widget);
  gboolean on;
  g_object_get(widget, "is-on-current-desktop", &on, NULL);
  if(on)
  {
    desktop_plugin->priv->pressed = FALSE;
    gtk_widget_queue_draw(widget);
  }
}

static void
desktop_switcher_home_plugin_init(DesktopSwitcherHomePlugin* desktop_plugin)
{
  desktop_plugin->priv = DESKTOP_SWITCHER_HOME_PLUGIN_GET_PRIVATE(desktop_plugin);
  desktop_plugin->priv->width = DEFAULT_WIDGET_WIDTH;
  desktop_plugin->priv->height = DEFAULT_WIDGET_HEIGHT;
  desktop_plugin->priv->alpha = 0;
  desktop_plugin->priv->switch_to = SWITCH_TO_LEFT;
  desktop_plugin->priv->smooth = TRUE;
  desktop_plugin->priv->icon_path = g_strdup(DEFAULT_LEFT_ICON_PATH);
  desktop_plugin->priv->background_pixbuf = NULL;
  desktop_plugin->priv->icon_width = 0;
  desktop_plugin->priv->icon_height = 0;
  desktop_plugin->priv->pressed = FALSE;

  GtkWidget* contents = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  g_signal_connect(GTK_CONTAINER(contents),
		   "button-press-event",
		   G_CALLBACK(_on_button_press),
		   desktop_plugin);
  g_signal_connect(GTK_CONTAINER(contents),
		   "button-release-event",
		   G_CALLBACK(_on_button_release),
		   desktop_plugin);
  hd_home_plugin_item_set_settings(HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(_show_settings_dialog), NULL);
  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(_on_current_desktop), NULL);

  gtk_widget_show_all(contents);
  gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin),
			      DEFAULT_WIDGET_WIDTH, DEFAULT_WIDGET_HEIGHT);
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
}

static void
desktop_switcher_home_plugin_class_init(DesktopSwitcherHomePluginClass* klass)
{
  GtkWidgetClass* widget_class = GTK_WIDGET_CLASS(klass);
  g_type_class_add_private(klass, sizeof(DesktopSwitcherHomePluginPrivate));
  widget_class->realize = desktop_switcher_realize;
  widget_class->expose_event = desktop_switcher_expose;
  G_OBJECT_CLASS(klass)->finalize = desktop_switcher_finalize;
}

static void
desktop_switcher_home_plugin_class_finalize(DesktopSwitcherHomePluginClass* klass)
{
}
