#include "utils.h"
#include <QDateTime>
#include <QFileInfo>
#include <QFile>
#ifndef QML_USER_INTERFACE
#include <QPainter>
#include <QRect>
#include <QImage>
#endif

QString Utils::durationFromSecs(int secs) {
    QTime time(0, 0);
    QString format = secs >= 3600 ? "hh:mm:ss" : "mm:ss";

    return time.addSecs(secs).toString(format);
}

QString Utils::durationFromMSecs(int msecs) {
    QTime time(0, 0);
    QString format = msecs >= 3600000 ? "hh:mm:ss" : "mm:ss";

    return time.addMSecs(msecs).toString(format);
}

QString Utils::fileSizeFromPath(const QString &filePath) {
    QFileInfo file(filePath);
    return Utils::fileSizeFromBytes(file.size());
}

QString Utils::fileSizeFromBytes(double bytes) {
    double kb = 1024;
    double mb = kb * 1024;
    double gb = mb * 1024;

    QString size;

    if (bytes > gb) {
        size = QString::number(bytes / gb, 'f', 2) + "GB";
    }
    else if (bytes > mb) {
        size = QString::number(bytes / mb, 'f', 2) + "MB";
    }
    else if (bytes > kb){
        size = QString::number(bytes / kb, 'f', 2) + "KB";
    }
    else {
        size = QString::number(bytes) + "B";
    }

    return size;
}

QString Utils::dateFromSecs(qint64 secs, bool showTime) {
    return Utils::dateFromMSecs(secs * 1000, showTime);
}

QString Utils::dateFromMSecs(qint64 msecs, bool showTime) {
    QString date;

    if (showTime) {
        date = QDateTime::fromMSecsSinceEpoch(msecs).toString("dd/MM/yyyy | HH:mm");
    }
    else {
        date = QDateTime::fromMSecsSinceEpoch(msecs).toString("dd/MM/yyyy");
    }

    return date;
}

QString Utils::base64Id(const QString &artist, const QString &title) {
    return QString(artist.toUtf8().toBase64() + "-" + title.toUtf8().toBase64());
}

QString Utils::localDateTimeFromString(const QString &dateTimeString, Qt::DateFormat format) {
    QDateTime dt = QDateTime::fromString(dateTimeString, format);

    if (!dt.isValid()) {
        dt = QDateTime::currentDateTimeUtc();
    }

    return dt.toLocalTime().toString("dd/MM/yyyy | HH:mm");
}

QString Utils::serviceString(Services::VideoService service) {
    switch (service) {
    case Services::YouTube:
        return QString("YouTube");
    case Services::Dailymotion:
        return QString("Dailymotion");
    case Services::Vimeo:
        return QString("Vimeo");
    default:
        return QString();
    }
}

QString Utils::unescape(const QString &s) {
    int unescapes = 0;
    QByteArray us = s.toUtf8();

    while ((us.contains('%')) && (unescapes < 10)) {
        us = QByteArray::fromPercentEncoding(us);
        unescapes++;
    }

    return QString(us);
}

void Utils::log(const QString &filePath, const QByteArray &message) {
    QFile lf(filePath);

    if (lf.open(QIODevice::Append)) {
        lf.write(QDateTime::currentDateTime().toString().toUtf8() + ": " + message + "\n");
    }

    lf.close();
}

QString Utils::versionNumberString() {
    return QString("1.9.1");
}

#ifndef QML_USER_INTERFACE
void Utils::drawBorderImage(QPainter *painter, const QRect &rect, const QImage &image, int top, int right, int left, int bottom) {
    QRect imageRect = image.rect();

    // Top-left
    painter->drawImage(QRect(rect.left(), rect.top(), left, top),
                       image,
                       QRect(imageRect.left(), imageRect.top(), left, top));

    // Top-right
    painter->drawImage(QRect(rect.right() - right, rect.top(), right, top),
                       image,
                       QRect(imageRect.right() - right, imageRect.top(), right, top));

    // Bottom-left
    painter->drawImage(QRect(rect.left(), rect.bottom() - bottom, left, bottom),
                       image,
                       QRect(imageRect.left(), imageRect.bottom() - bottom, left, bottom));

    // Bottom-right
    painter->drawImage(QRect(rect.right() - right, rect.bottom() - bottom, bottom, right),
                       image,
                       QRect(imageRect.right() - right, imageRect.bottom() - bottom, right, bottom));

    // Top-middle
    painter->drawImage(QRect(rect.left() + left, rect.top(), rect.width() - (left + right), top),
                       image,
                       QRect(imageRect.left() + left, imageRect.top(), imageRect.width() - (left + right), top));

    // Bottom-middle
    painter->drawImage(QRect(rect.left() + left, rect.bottom() - bottom, rect.width() - (left + right), bottom),
                       image,
                       QRect(imageRect.left() + left, imageRect.bottom() - bottom, imageRect.width() - (left + right), bottom));

    // Left-middle
    painter->drawImage(QRect(rect.left(), rect.top() + top, left, rect.height() - (top + bottom)),
                       image,
                       QRect(imageRect.left(), imageRect.top() + top, left, imageRect.height() - (top + bottom)));

    // Right-middle
    painter->drawImage(QRect(rect.right() - right, rect.top() + top, right, rect.height() - (top + bottom)),
                       image,
                       QRect(imageRect.right() - right, imageRect.top() + top, right, imageRect.height() - (top + bottom)));

    // Center
    painter->drawImage(QRect(rect.left() + left, rect.top() + top, rect.width() - (left + right), rect.height() - (top + bottom)),
                       image,
                       QRect(imageRect.left() + left, imageRect.top() + top, imageRect.width() - (left + right), imageRect.height() - (top + bottom)));
}
#endif
