# -*- coding: utf-8 -*-
# Copyright (C) 2009 by Daniel Martin Yerga <dyerga@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# CasualServices: Search and share your favourite places
"""Module to search on Google Local

Exported Classes:

GLocal -- Class to access to search functions

"""

import httplib2
import simplejson

#URLs used for searches
SEARCHURL="http://ajax.googleapis.com/ajax/services/search/local?v=1.0&q=%s&rsz=large&start=%s"
SEARCHLOC="http://ajax.googleapis.com/ajax/services/search/local?v=1.0&q=%s+%s&rsz=large&start=%s"
SEARCHURLGEO ="http://ajax.googleapis.com/ajax/services/search/local?v=1.0&q=%s&sll=%s,%s&rsz=large&start=%s"

class GLocal:
    """Class to access to Google Local searches.

    Public functions:
    search -- Return a dictionary with the results of the search on Google Local.

    searchgeo -- Return a dictionary with the results of the search on Google Local.
    The obtained results are near to a GPS position.

    searchlocated -- Return a dictionary with the results of the search on Google Local.
    The obtained results are near the specified city.

    change_page -- Translate number of page 1, 2, 3 to Google Local.

    """
    def __init__(self):
        """Define some variables to use in some functions.

        """
        pass

    def search(self, text, page):
        """Return a dictionary with the results of the search on Google Local.

        @type   text: string
        @param  text: The text to be searched on Google Local.
        @type   page: string
        @param  page: The number of the page of results.

        @rtype: dictionary
        @returns: A dictionary with the results of the search on Google Local.

        """
        h = httplib2.Http()

        page = self.change_page(page)

        request, content = h.request(SEARCHURL % (text, page))

        content = simplejson.loads(content)
        points = content['responseData']['results']
        #print points, '\n\n'

        return points

    def searchlocated(self, text, location, page):
        """Return a dictionary with the results of the search on Google Local.

        @type   text: string
        @param  text: The text to be searched on Google Local.
        @type   location: string
        @param  location: Location of the service (ex. Madrid)
        @type   page: string
        @param  page: The number of the page of results.

        @rtype: dictionary
        @returns: A dictionary with the results of the search on Google Local.

        """
        h = httplib2.Http()

        page = self.change_page(page)
        request, content = h.request(SEARCHLOC % (text, location, page))

        content = simplejson.loads(content)
        points = content['responseData']['results']
        #print points, '\n\n'

        return points

    def searchgeo(self, text, page, lat, lon, radius):
        """Return a dictionary with the results of the search on Google Local.
        The obtained results are near to a GPS position.

        @type   text: string
        @param  text: The text to be searched on Google Local.
        @type   page: string
        @param  page: The number of the page of results.
        @type   lat: string
        @param  lat: Geographic latitude.
        @type   lon: string
        @param  lon: Geographic longitude.
        @type   radius: string
        @param  radius: Radius where the services should be localized (in Km).

        @rtype: dictionary
        @returns: A dictionary with the results of the search on Google Local.
        The obtained results are near to a GPS position.

        """
        h = httplib2.Http()

        page = self.change_page(page)
        request, content = h.request(SEARCHURLGEO % (text, lat, lon, page))
        content = simplejson.loads(content)
        points = content['responseData']['results']
        #print points, '\n\n'

        return points

    def change_page(self, page):
        """Translate number of page 1, 2, 3 to Google Local format: 10, 20, etc.

        @type   page: string
        @param  page: The number of the page of results.

        @rtype: text
        @returns: Text with the page numbering on Google Local

        """
        if page == '2':
            page = '10'
        elif page == '3':
            page = '20'
        elif page == '4':
            page = '30'
        elif page == '5':
            page = '40'
        elif page == '6':
            page = '50'
        elif page == '7':
            page = '60'
        elif page == '8':
            page = '70'
        elif page == '9':
            page = '80'
        elif page == '10':
            page = '90'

        return page
