#include "processingconfiguration.h"
#include "ui_processingconfiguration.h"
#include "enternamedialog.h"
#include "ClassicPrintProcessing.h"
#include "ClassicPrintSettings.h"
#include <QDir>

ProcessingConfiguration::ProcessingConfiguration(ClassicPrint* cp, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::ProcessingConfiguration)
{
    ui->setupUi(this);

	// Take a local copy of the processing settings
	m_processing = new ClassicPrintProcessing(cp);
	if (cp->getCurrentProcessing()) {
		*m_processing = *cp->getCurrentProcessing();
	}

    // Put the icons on the buttons
    QIcon icon_delete(QIcon(ClassicPrintSettings::config_dir() + "/icons/camera_delete.png"));
    icon_delete.addFile(ClassicPrintSettings::config_dir() + "/icons/camera_delete_dimmed.png", QSize(), QIcon::Disabled);
    ui->btn_delete->setIcon(icon_delete);
    QIcon icon_save(QIcon(ClassicPrintSettings::config_dir() + "/icons/notes_save.png"));
    icon_save.addFile(ClassicPrintSettings::config_dir() + "/icons/notes_save_dimmed.png", QSize(), QIcon::Disabled);
    ui->btn_save->setIcon(icon_save);

    m_cp = cp;

    m_updating = true;

    // Setup the controls to reflect the current settings
	ui->slider_colourisation->setValue(m_processing->colourisationPercent());
	ui->slider_contrast->setValue(m_processing->contrast());
	ui->slider_frame_size->setValue((int)(m_processing->frameSizePercent() * 100.0 / 7.0));

    ui->btn_save->setEnabled(false);
    // Populate the combobox with the available processing list
    int index = 0;
    ClassicPrintProcessing* processing;
    QString name;
    ui->combo_name->clear();
    while (m_cp->getProcessing(index++, name, &processing)) {
        ui->combo_name->addItem(name);
    }
    // If there is more than one left then enable the delete button
    if (ui->combo_name->count() > 1) {
        ui->btn_delete->setEnabled(true);
    }
    else {
        ui->btn_delete->setEnabled(false);
    }

	// Populate the process combobox
	index = 0;
	while (m_cp->getColourProfileIndex(index++, name)) {
		ui->combo_process->addItem(name);
	}

	// If the current item has been modified then ensure it has an entry
	int item = ui->combo_name->findText(m_processing->name());
	if (item >= 0) {
		ui->combo_name->setCurrentIndex(item);
	}
	else {
		ui->combo_name->addItem(m_processing->name());
		item = ui->combo_name->findText(m_processing->name());
		if (item >= 0) {
			ui->combo_name->setCurrentIndex(item);
			ui->btn_save->setEnabled(true);
		}
	}
	// Select the correct process
	item = ui->combo_process->findText(m_processing->colourisation());
	if (item >= 0) {
		ui->combo_process->setCurrentIndex(item);
	}

	// Add the light leak entries in the combo box
	ui->combo_light_leak->addItem("None");
	ui->combo_light_leak->addItem(RANDOM_LEAK, RANDOM_LEAK);
	QDir leaks(ClassicPrintSettings::config_dir() + "/light_leak");
	QStringList filter;
	filter << "*.jpg" << "*.png";
	QStringList files = leaks.entryList(filter);
	foreach (QString file, files) {
		QString name(file);
		// Remove the file extension
		int dot = file.lastIndexOf('.');
		if (dot != -1) {
			name = file.mid(0, dot);
		}
		// Use the filename as the combo text
		ui->combo_light_leak->addItem(name, file);
	}
	// Select the current entry
	int light = ui->combo_light_leak->findData(m_processing->lightLeak());
	if (light >= 0) {
		ui->combo_light_leak->setCurrentIndex(light);
	}
	else {
		ui->combo_light_leak->setCurrentIndex(0);
	}

    m_updating = false;
}

ProcessingConfiguration::~ProcessingConfiguration()
{
    delete ui;
	delete m_processing;
}

void ProcessingConfiguration::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void ProcessingConfiguration::on_combo_name_activated(QString name)
{
	QString					processing_name;
	ClassicPrintProcessing*	processing;
	int						index = 0;

    m_updating = true;
    ui->btn_save->setEnabled(false);

	// Copy the named settings to the local settings
	while (m_cp->getProcessing(index++, processing_name, &processing)) {
		if (processing_name == name) {
			*m_processing = *processing;
			break;
		}
	}

    // Reset the controls
	ui->slider_colourisation->setValue(m_processing->colourisationPercent());
	ui->slider_contrast->setValue(m_processing->contrast());
    // Remove the custom field from the combobox
    int item = ui->combo_name->findText("[Custom]");
    if (item >= 0) {
        ui->combo_name->removeItem(item);
    }
    // If there is more than one item left then enable the delete button
    if (ui->combo_name->count() > 1) {
        ui->btn_delete->setEnabled(true);
    }
    else {
        ui->btn_delete->setEnabled(false);
    }
    m_updating = false;
}

void ProcessingConfiguration::on_btn_save_clicked()
{
    EnterNameDialog enter_name(this);
    enter_name.setName("");
	if (enter_name.exec() != QDialog::Accepted) {
		return;
	}
	if (enter_name.name().length() > 0) {
        // Add the new processing
		m_processing->setName(enter_name.name());
		ClassicPrintProcessing* processing = new ClassicPrintProcessing(m_cp);
		*processing = *m_processing;
        m_cp->addProcessing(processing);
        // Repopulate the combobox
        m_updating = true;
        int index = 0;
        QString name;
        ui->combo_name->clear();
        while (m_cp->getProcessing(index++, name, &processing)) {
            ui->combo_name->addItem(name);
        }
        // If there is more than one left then enable the delete button
        if (ui->combo_name->count() > 1) {
            ui->btn_delete->setEnabled(true);
        }
        else {
            ui->btn_delete->setEnabled(false);
        }
        // If the current item has been modified then ensure it has an entry
		int item = ui->combo_name->findText(m_processing->name());
        if (item >= 0) {
            ui->combo_name->setCurrentIndex(item);
        }
        ui->btn_save->setEnabled(false);
        m_updating = false;
    }

}

void ProcessingConfiguration::on_btn_delete_clicked()
{
    int item = ui->combo_name->currentIndex();
    if (item >= 0) {
        QString name;
        ClassicPrintProcessing* processing;

        m_cp->deleteProcessing(ui->combo_name->currentText());
        ui->combo_name->removeItem(item);
        // This should always work because the button would only be
        // enabled if there was more than one item
        m_cp->getProcessing(0, name, &processing);
        item = ui->combo_name->findText(name);
        if (item != -1) {
            ui->combo_name->setCurrentIndex(item);
        }
    }

}

void ProcessingConfiguration::on_slider_colourisation_valueChanged(int value)
{
    if (!m_updating) {
		m_processing->setColourisationPercent(value);
        setCustom();
    }
}

void ProcessingConfiguration::on_slider_contrast_valueChanged(int value)
{
    if (!m_updating) {
		m_processing->setContrast(value);
        setCustom();
    }
}

void ProcessingConfiguration::setCustom() {
    // See if the custom value is already in the combobox
    if (ui->combo_name->findText("[Custom]") < 0) {
        // No. Add it in
        ui->combo_name->addItem("[Custom]");
        // Make sure it is selected
        ui->combo_name->setCurrentIndex(ui->combo_name->findText("[Custom]"));
        // Make sure the current item has the custom name
		m_processing->setName("[Custom]");

        ui->btn_delete->setEnabled(false);
        ui->btn_save->setEnabled(true);
    }
}


void ProcessingConfiguration::on_combo_process_activated(QString value)
{
	if (!m_updating) {
		m_processing->setColourisation(value);
		setCustom();
	}
}

void ProcessingConfiguration::on_combo_light_leak_activated(int index)
{
	if (!m_updating) {
		m_processing->setLightLeak(ui->combo_light_leak->itemData(index).toString());
		setCustom();
	}
}

void ProcessingConfiguration::on_slider_frame_size_sliderMoved(int position)
{
	if (!m_updating) {
		m_processing->setFrameSizePercent((double)position * 7.0 / 100.0);
		setCustom();
	}
}

void ProcessingConfiguration::on_buttonBox_accepted()
{
	// Apply our local processing settings to the current settings
	if (m_cp->getCurrentProcessing()) {
		*m_cp->getCurrentProcessing() = *m_processing;
	}
	accept();
}
