/*
 *  Camera Launcher for Maemo.
 *  Copyright (C) 2010 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gconf/gconf-client.h>

#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtk.h>
#include <hildon/hildon.h>

#include "cl-utils.h"

DesktopFileInfo *
get_desktop_file_info (const gchar *desktop_file)
{
	DesktopFileInfo *dfinfo = NULL;
	GKeyFile *key_file;

	if (desktop_file) {
		key_file = g_key_file_new ();
		if (g_key_file_load_from_file (key_file, desktop_file, 0, NULL)) {
			gchar *type = NULL;

			/* exclude non application desktop files */
			if (g_key_file_has_key (key_file, "Desktop Entry", "Type", NULL)) {
				type = g_key_file_get_string (key_file, "Desktop Entry", "Type", NULL);
				if (type) {
					if (strcmp (type, "Application")) {
						g_free (type);
						return NULL;
					}
					g_free (type);
				}
			}

			dfinfo = g_new0 (DesktopFileInfo, 1);

			/* get application icon */
			if (g_key_file_has_key (key_file, "Desktop Entry", "Icon", NULL)) {
				dfinfo->icon = g_key_file_get_string (key_file, "Desktop Entry", "Icon", NULL);
			} else {
				dfinfo->icon = NULL;
			}

			/* get application name */
			if (g_key_file_has_key (key_file, "Desktop Entry", "X-Text-Domain", NULL)) {
				textdomain (g_key_file_get_string (key_file, "Desktop Entry", "X-Text-Domain", NULL));
				dfinfo->name = gettext (g_key_file_get_string (key_file, "Desktop Entry", "Name", NULL));
			} else if (g_key_file_has_key (key_file, "Desktop Entry", "Name", NULL)) {
				textdomain ("maemo-af-desktop");
				dfinfo->name = gettext (g_key_file_get_string (key_file, "Desktop Entry", "Name", NULL));
			} else {
				dfinfo->name = NULL;
			}

			/* get application osso_service */
			if (g_key_file_has_key (key_file, "Desktop Entry", "X-Osso-Service", NULL)) {
				dfinfo->osso_service = g_key_file_get_string (key_file, "Desktop Entry", "X-Osso-Service", NULL);
			} else {
				dfinfo->osso_service = NULL;
			}

			/* get application exec */
			if (g_key_file_has_key (key_file, "Desktop Entry", "Exec", NULL)) {
				dfinfo->exec = g_key_file_get_string (key_file, "Desktop Entry", "Exec", NULL);
			} else {
				dfinfo->exec = NULL;
			}
		}

		if (key_file)
			g_key_file_free (key_file);
	}
	return dfinfo;
}

gboolean
get_application_list (GtkListStore *store)
{
	gboolean is_empty = TRUE;
	GtkIconTheme *icon_theme;
	GtkTreeIter iter;
	GDir *dir;

	icon_theme = gtk_icon_theme_get_default ();

	dir = g_dir_open (CL_LAUNCHER_DESKTOP_DATADIR, 0, NULL);
	if (dir) {
		const gchar *filename;

		while ((filename = g_dir_read_name (dir)) != NULL) {
			if (g_str_has_suffix (filename, ".desktop")) {
				DesktopFileInfo *dfinfo = NULL;
				GtkIconInfo *icon_info;
				GdkPixbuf *icon_pixbuf = NULL;
				gchar *desktop_file = NULL;

				desktop_file = g_strdup_printf ("%s/%s", CL_LAUNCHER_DESKTOP_DATADIR, filename);
				if (desktop_file) {
					dfinfo = get_desktop_file_info (desktop_file);
					if (dfinfo) {
						if (!dfinfo->icon) {
							dfinfo->icon = g_strdup ("tasklaunch_default_application");
						}

						icon_info = gtk_icon_theme_lookup_icon (icon_theme, dfinfo->icon, 48,
											GTK_ICON_LOOKUP_NO_SVG);
						if (icon_info) {
							const gchar *fname = gtk_icon_info_get_filename (icon_info);
							icon_pixbuf = gdk_pixbuf_new_from_file_at_size (fname, 48, 48, 0);
							gtk_icon_info_free (icon_info);
						}

						/* fill application store */
						gtk_list_store_append (store, &iter);
						gtk_list_store_set (store, &iter,
								    SELECTOR_COLUMN_NAME, dfinfo->name,
								    SELECTOR_COLUMN_ICON, icon_pixbuf,
								    SELECTOR_COLUMN_FILENAME, filename,
								    SELECTOR_COLUMN_OSSO_SERVICE, dfinfo->osso_service,
								    SELECTOR_COLUMN_EXEC, dfinfo->exec,
								    -1);

						if (is_empty == TRUE)
							is_empty = FALSE;

						if (icon_pixbuf)
							g_object_unref (icon_pixbuf);

						g_free (dfinfo);
					}
					g_free (desktop_file);
				}
			}
		}
		g_dir_close (dir);
	}

	return is_empty;
}

gboolean
get_application_list_from_list (GtkListStore *store, const GSList *list)
{
	const GSList *application;
	gboolean is_empty = TRUE;
	GtkIconTheme *icon_theme;
	GtkTreeIter iter;

	icon_theme = gtk_icon_theme_get_default ();

	for (application = list; application != NULL; application = application->next) {
		const gchar *filename;

		filename = gconf_value_get_string (application->data);

		if (g_str_has_suffix (filename, ".desktop")) {
			DesktopFileInfo *dfinfo = NULL;
			GtkIconInfo *icon_info;
			GdkPixbuf *icon_pixbuf = NULL;
			gchar *desktop_file = NULL;

			desktop_file = g_strdup_printf ("%s/%s", CL_LAUNCHER_DESKTOP_DATADIR, filename);
			if (desktop_file) {
				dfinfo = get_desktop_file_info (desktop_file);
				if (dfinfo) {
					if (!dfinfo->icon) {
						dfinfo->icon = g_strdup ("tasklaunch_default_application");
					}

					icon_info = gtk_icon_theme_lookup_icon (icon_theme, dfinfo->icon, 48,
										GTK_ICON_LOOKUP_NO_SVG);
					if (icon_info) {
						const gchar *fname = gtk_icon_info_get_filename (icon_info);
						icon_pixbuf = gdk_pixbuf_new_from_file_at_size (fname, 48, 48, 0);
						gtk_icon_info_free (icon_info);
					}

					/* fill application store */
					gtk_list_store_append (store, &iter);
					gtk_list_store_set (store, &iter,
							    SELECTOR_COLUMN_NAME, dfinfo->name,
							    SELECTOR_COLUMN_ICON, icon_pixbuf,
							    SELECTOR_COLUMN_FILENAME, filename,
							    SELECTOR_COLUMN_OSSO_SERVICE, dfinfo->osso_service,
							    SELECTOR_COLUMN_EXEC, dfinfo->exec,
							    -1);

					if (is_empty == TRUE)
						is_empty = FALSE;

					if (icon_pixbuf)
						g_object_unref (icon_pixbuf);

					g_free (dfinfo);
				}
				g_free (desktop_file);
			}
		}
	}

	return is_empty;
}
