/**
 *  Viito -- an image dictionary for sign language
 *
 *  Copyright (C) 2010  Aki Niemi <aki.niemi@iki.fi>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QtGui>
#include <QDomDocument>

#include "vmainwindow.h"
#include "vimagewidget.h"
#include "vsearchwidget.h"
#include "vdatamodel.h"

static QDomDocument readXmlDocument(const QString &fileName)
{
    QFile file(fileName);
    QDomDocument doc;

    QString error;
    int errorLine;
    int errorColumn;

    if (!file.open(QIODevice::ReadOnly)) {
	qDebug() << "Read failed:" << file.error();
	return doc;
    }

    if (!doc.setContent(&file, &error, &errorLine, &errorColumn)) {
	qDebug() << "Error parsing XML document:" << error
		 << "at line:" << errorLine
		 << "and column:" << errorColumn;
    }

    file.close();
    return doc;
}

VMainWindow::VMainWindow()
    : QMainWindow()
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
}

VMainWindow::VMainWindow(const QString &file)
    : QMainWindow()
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);

    createModels(readXmlDocument(file));

    QListView *mainView = new QListView(this);
    mainView->setModel(mDataModel);
    setCentralWidget(mainView);

    mImageView = new QListView(mainView);
    mImageView->setWindowFlags(mImageView->windowFlags() | Qt::Window);
    mImageView->setAttribute(Qt::WA_Maemo5StackedWindow);
    mImageView->setAttribute(Qt::WA_Maemo5AutoOrientation, true);

    mImageWidget = new VImageWidget(mImageView);

    mSearchWidget = new VSearchWidget(this);
    mSearchWidget->setModel(mSearchModel);

    QAction *aboutAction = new QAction(tr("About"), this);
    menuBar()->addAction(aboutAction);
    QAction *searchAction = new QAction(tr("Search"), this);
    menuBar()->addAction(searchAction);

    connect(aboutAction, SIGNAL(triggered()), this, SLOT(about()));
    connect(searchAction, SIGNAL(triggered()), this, SLOT(search()));

    connect(mainView, SIGNAL(clicked(const QModelIndex&)),
	    this, SLOT(topicClicked(const QModelIndex&)));
    connect(mImageView, SIGNAL(clicked(const QModelIndex&)),
	    this, SLOT(imageClicked(const QModelIndex&)));
    connect(mImageWidget, SIGNAL(next()),
	    this, SLOT(nextImage()));
    connect(mImageWidget, SIGNAL(previous()),
	    this, SLOT(previousImage()));
}

void VMainWindow::createModels(const QDomDocument &doc)
{
    mDataModel = new QStandardItemModel(this);
    mSearchModel = new QStandardItemModel(this);

    QDomNodeList topics = doc.elementsByTagName("topic");

    for (int i = 0; i < topics.count(); i++) {

	QDomNode node = topics.at(i);

	QString name = node.attributes().namedItem("name").nodeValue();
	if (name.isEmpty())
	    continue;

	QStandardItemModel *imageModel = new QStandardItemModel;
	VTopicItem *topic = new VTopicItem(name, imageModel);

	QDomNodeList images = node.childNodes();

	for (int j = 0; j < images.count(); j++) {

	    QDomNamedNodeMap imgattrs = images.at(j).attributes();

	    QString name = imgattrs.namedItem("name").nodeValue();
	    QString author = imgattrs.namedItem("author").nodeValue();
	    QString path = imgattrs.namedItem("file").nodeValue();

	    VImageItem *image = new VImageItem(name, author, path);

	    imageModel->appendRow(image);
	    mSearchModel->appendRow(image->clone());
	}
	imageModel->sort(0);
	mDataModel->appendRow(topic);
    }
    mDataModel->sort(0);
}

void VMainWindow::topicClicked(const QModelIndex &index)
{
    //FIXME: Ugly
    QStandardItemModel *model = (QStandardItemModel *)index.data(VTopicItem::ModelRole).value<void *>();

    mImageView->setModel(model);
    mImageView->setWindowTitle(index.data().toString());
    mImageView->show();
}

void VMainWindow::imageClicked(const QModelIndex &index)
{
    setImage(index);
    mImageWidget->show();
}

void VMainWindow::setImage(const QModelIndex &index, int offset)
{
    QModelIndex next;

    if (offset != 0) {
	next = index.sibling(index.row() + offset, 0);
	if (!next.isValid())
	    return;

	mImageView->scrollTo(next);
	mImageView->setCurrentIndex(next);
    } else {
	next = index;
    }

    mImageWidget->setWindowTitle(next.data().toString());
    mImageWidget->setImage(next.data(VImageItem::PathRole).toString(),
			   QString("\251 %1").arg(next.data(VImageItem::AuthorRole).toString()));
}

void VMainWindow::nextImage()
{
    setImage(mImageView->currentIndex(), 1);
}

void VMainWindow::previousImage()
{
    setImage(mImageView->currentIndex(), -1);
}

void VMainWindow::about()
{
    QMessageBox messageBox;
    messageBox.addButton(tr("Done"), QMessageBox::AcceptRole);
    messageBox.setTextFormat(Qt::RichText);

    messageBox.setWindowTitle(tr("About Viito"));
    messageBox.setText(tr("Viito is a free and open source sign language "
			  "image dictionary application.<br/><br/>Copyright"
			  " &#169; 2010 Aki Niemi<br/><br/>Source for image "
			  "data:<br/><a href=\"http://www.papunet.fi\">"
			  "Papunetin kuvapankki</a>, Elina Vanninen and "
			  "Kalevi Puistolinna."));

    messageBox.exec();
}

void VMainWindow::search()
{
    mSearchWidget->show();
}
