//  © Copyright 2010 Reuben D'Netto (rdnetto+uremote@gmail.com)
//  This file is part of uRemote v1.0
//
//  uRemote is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  uRemote is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with uRemote.  If not, see <http://www.gnu.org/licenses/>.


#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "torrentdata.h"
#include "torrentitem.h"
#include "torrentdialog.h"
#include "addtorrentdialog.h"
#include "settingsdialog.h"
#include "filterdialog.h"
#include "aboutdialog.h"
#include <QMessageBox>
#include <QtAlgorithms>

MainWindow::MainWindow(QWidget *parent)
	: QMainWindow(parent), ui(new Ui::MainWindow){
	ui->setupUi(this);
	ui->scrollArea->setProperty("FingerScrollable", true);
	ui->menuFremantle->setTitle("");

	Settings = new QSettings("Reuben D'Netto", "uRemote", this);

	UpdateTimer = new QTimer(this);
	connection = new BtConnection(this);
	SuppressMessages = false;

	connect(ui->actionUpdate, SIGNAL(triggered()),
			connection, SLOT(GetTorrents()));
	connect(ui->actionFilter, SIGNAL(triggered()),
			this, SLOT(ShowFilterDialog()));
	connect(ui->actionSettings, SIGNAL(triggered()),
			this, SLOT(ShowSettingsDialog()));
	connect(ui->actionAdd, SIGNAL(triggered()),
			this, SLOT(ShowAddTorrentDialog()));
	connect(ui->actionAbout, SIGNAL(triggered()),
			this, SLOT(ShowAbout()));
	connect(UpdateTimer, SIGNAL(timeout()),
			connection, SLOT(GetTorrents()));
	connect(connection, SIGNAL(QueryComplete()),
			this, SLOT(UpdateView()));
	connect(connection, SIGNAL(Error(QString)),
			this, SLOT(OnError(QString)));


	if(Settings->allKeys().count() == 0){
		ShowSettingsDialog();
	}else{
		connection->GetTorrents(false);
		if(AutoUpdatesEnabled()) UpdateTimer->start(RefreshRate());
	} //end if
}

MainWindow::~MainWindow(){
	delete ui;
	delete connection;
}

void MainWindow::UpdateView(){
	QList<TorrentData> newTorrents = Filter(connection->Torrents());

	//add new items if necessary
	while(Torrents.length() < newTorrents.length()){
		TorrentItem *widget = new TorrentItem(TorrentData::Null(), ui->scrollAreaWidgetContents);
		//manually specified because it expands indefinitely otherwise
		widget->setMaximumWidth(ui->scrollAreaWidgetContents->width() - 50);
		connect(widget, SIGNAL(clicked(TorrentData, int)), this, SLOT(ShowTorrentDialog(TorrentData)));
		//inserted before spacer
		ui->scrollAreaLayout->insertWidget(ui->scrollAreaWidgetContents->children().count() - 2, widget);
		Torrents.append(widget);
	} //end while

	//remove extra items
	while(Torrents.length() > newTorrents.length()){
		Torrents.last()->hide();
		Torrents.takeLast()->deleteLater();
	} //end while

	//update data
	for(int i = 0; i < Torrents.length(); i += 1)
		Torrents[i]->setData(newTorrents[i]);
}

void MainWindow::ShowFilterDialog(){
	QList<QString> labels = connection->Labels();
	FilterDialog* filter = new FilterDialog(labels, this);
	filter->select(Filter(), FilterIsState(), SortMode());

	connect(filter, SIGNAL(closed(QString, bool, TSortMode)),
			this, SLOT(UpdateFilter(QString, bool, TSortMode)));

	filter->show();
	ui->scrollArea->ensureVisible(0, 0); //scroll to top
}

void MainWindow::UpdateFilter(QString label, bool isState, TSortMode sortMode){
	setFilter(label);
	setFilterIsState(isState);
	setSortMode(sortMode);
	UpdateView();
}

void MainWindow::ShowSettingsDialog(){
	SettingsDialog* dialog = new SettingsDialog(this);
	dialog->show();
}

void MainWindow::OnError(QString error){
	if(SuppressMessages) return;

	SuppressMessages = true;
	setAutoUpdatesEnabled(false);
	QMessageBox::warning(this, "Error", "There was an error connecting to the server: " + error + (AutoUpdatesEnabled() ? "\nAutomatic Updates are now disabled." : false), QMessageBox::Ok, QMessageBox::NoButton);
	SuppressMessages = false;
}

void MainWindow::ShowTorrentDialog(TorrentData torrent){
	TorrentDialog *dialog = new TorrentDialog(this, connection, torrent);
	dialog->show();
}

void MainWindow::ShowAddTorrentDialog(){
	AddTorrentDialog *dialog = new AddTorrentDialog(this, connection);
	dialog->show();
}

void MainWindow::ShowAbout(){
	AboutDialog *dialog = new AboutDialog(this);
	dialog->show();
}

QList<TorrentData> MainWindow::Filter(QList<TorrentData> torrents){
	QList<TorrentData> result;

	//Filter
	foreach(TorrentData torrent, torrents){
		if(FilterIsState()){
			if(Filter() == "Downloading" && torrent.Progress == 1) continue;
			if(Filter() == "Completed"   && torrent.Progress != 1) continue;
			if(Filter() == "Active" && !(torrent.Status & TS_Started)) continue;
			if(Filter() == "Inactive" && (torrent.Status & TS_Started)) continue;
			if(Filter() == "NoLabel" && !torrent.Label.isEmpty()) continue;
		}else{
			if(torrent.Label != Filter()) continue;
		} //end if

		result.append(torrent);
	} //end foreach

	//Sort
	//prefilter by name (acts as a secondary filter when not the primary filter)
	qStableSort(result.begin(), result.end(), TorrentData::CompareByName);

	switch(SortMode()){
	case TSM_Name:
		//already sorted
	break;
	case TSM_Status:
		qStableSort(result.begin(), result.end(), TorrentData::CompareByStatus);
	break;
	case TSM_Eta:
		qStableSort(result.begin(), result.end(), TorrentData::CompareByEta);
	break;
	case TSM_Label:
		qStableSort(result.begin(), result.end(), TorrentData::CompareByLabel);
	break;
	default:
		qDebug(("Error: invalid sort mode: " + QString::number(SortMode())).toLocal8Bit());
	break;
	} //end switch

	return result;
}
