//  © Copyright 2010 Reuben D'Netto (rdnetto+uremote@gmail.com)
//  This file is part of uRemote v1.0
//
//  uRemote is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  uRemote is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with uRemote.  If not, see <http://www.gnu.org/licenses/>.


#include "filesdialog.h"
#include "ui_filesdialog.h"
#include <QLabel>
#include <QMessageBox>

FilesDialog::FilesDialog(QWidget *parent, BtConnection* con, TorrentData tor) :
	QMainWindow(parent), m_ui(new Ui::FilesDialog), Data(tor){
	Connection = con;
	m_ui->setupUi(this);
	m_ui->PriorityCmb->addItems(QString("Skip|Low|Normal|High").split('|'));

	connect(Connection, SIGNAL(Updating()), this, SLOT(Poll()));
	connect(Connection, SIGNAL(QueryComplete()), this, SLOT(UpdateData()));
	connect(m_ui->FilesTbl, SIGNAL(itemSelectionChanged()), this, SLOT(LoadStatus()));
	connect(m_ui->PriorityCmb, SIGNAL(currentIndexChanged(QString)), this, SLOT(SetStatus(QString)));
	connect(m_ui->ApplyBtn, SIGNAL(clicked()), this, SLOT(ApplyChanges()));
	connect(m_ui->RevertBtn, SIGNAL(clicked()), this, SLOT(RevertChanges()));

	Reload();
}

FilesDialog::~FilesDialog(){
	delete m_ui;
}

void FilesDialog::changeEvent(QEvent *e){
	QMainWindow::changeEvent(e);
	switch (e->type()) {
	case QEvent::LanguageChange:
		m_ui->retranslateUi(this);
		break;
	default:
		break;
	}
}

void FilesDialog::UpdateData(){
	static bool removed = false;
	if(removed) return;

	int index = Connection->GetIndex(Data.Hash);
	if(index == -1){
		//torrent removed
		//leave notification to parent dialog
		removed = true; //prevent repeated messages
		this->hide();
		delete this;
		//this->deleteLater();
	}else{
		qDebug("Torrent updated");
		Data = Connection->Torrents()[index];
	} //end if

	Reload();
}

void FilesDialog::Reload(){
	if(m_ui->FilesTbl->rowCount() == 0){
		//load data from scratch
		m_ui->FilesTbl->setRowCount(Data.Files.length());
		m_ui->FilesTbl->setColumnCount(4);
		m_ui->FilesTbl->setHorizontalHeaderLabels(QString("Filename|Size|%|Priority").split('|'));
		m_ui->FilesTbl->horizontalHeader()->setResizeMode(0, QHeaderView::Stretch);
		m_ui->FilesTbl->horizontalHeader()->setResizeMode(1, QHeaderView::ResizeToContents);
		m_ui->FilesTbl->horizontalHeader()->setResizeMode(2, QHeaderView::ResizeToContents);
		m_ui->FilesTbl->horizontalHeader()->setResizeMode(3, QHeaderView::ResizeToContents);

		for(int i = 0; i < Data.Files.length(); i += 1){
			m_ui->FilesTbl->setItem(i, 0, new QTableWidgetItem(Data.Files[i].Filename));
			m_ui->FilesTbl->setItem(i, 1, new QTableWidgetItem(Data.Files[i].SizeStr()));
			m_ui->FilesTbl->setItem(i, 2, new QTableWidgetItem(Data.Files[i].Progress()));
			m_ui->FilesTbl->setItem(i, 3, new QTableWidgetItem(Data.Files[i].PriorityStr()));
		} //end for

	}else{
		//update existing entries
		for(int i = 0; i < Data.Files.length(); i += 1){
			m_ui->FilesTbl->item(i, 2)->setText(Data.Files[i].Progress());
			if(! Changes.contains(i)) m_ui->FilesTbl->item(i, 3)->setText(Data.Files[i].PriorityStr());
		} //end for
	} //end if
}

void FilesDialog::LoadStatus(){
	if(m_ui->FilesTbl->selectedItems().length() != 0)
		m_ui->PriorityCmb->setCurrentIndex(Data.Files[m_ui->FilesTbl->selectedItems()[0]->row()].Priority);
}
void FilesDialog::SetStatus(QString){
	foreach(QTableWidgetItem *item, m_ui->FilesTbl->selectedItems()){
		//don't apply change if same priority is selected - filters out event firing on selection
		if(Data.Files[item->row()].Priority != m_ui->PriorityCmb->currentIndex()){
			//format changed files
			QFont font = item->font();
			font.setItalic(true);
			item->setFont(font);

			if(item->column() == 0) Changes[item->row()] = static_cast<TFPriority>(m_ui->PriorityCmb->currentIndex());
			if(item->column() == 3) item->setText(TorrentFile::PriorityStr(Changes[item->row()]));
		} //end if
	} //end if
}
void FilesDialog::ApplyChanges(){
	//group changes based on priority
	QList< QList<int> > files;
	for(int i = 0; i < 4; i += 1) files.append(QList<int>());

	foreach(int file, Changes.keys()){
		files[Changes[file]].append(file);
	} //end foreach

	//apply changes
	for(int prio = 0; prio < 4; prio += 1){
		if(files[prio].count() == 0) continue;

		Connection->SetFilePriority(Data.Hash, static_cast<TFPriority>(prio), files[prio]);
	} //end for

	//remove formatting
	foreach(int row, Changes.keys()){
		for(int col = 0; col < 4; col += 1){
			QTableWidgetItem *item = m_ui->FilesTbl->item(row, col);
			QFont font = item->font();
			font.setItalic(false);
			item->setFont(font);
		} //end for
	} //end foreach

	Changes.clear();
}
void FilesDialog::RevertChanges(){
	//remove formatting
	foreach(int row, Changes.keys()){
		for(int col = 0; col < 4; col += 1){
			QTableWidgetItem *item = m_ui->FilesTbl->item(row, col);
			QFont font = item->font();
			font.setItalic(false);
			item->setFont(font);
		} //end for
	} //end foreach

	Changes.clear();
	Reload(); //restore current values
}
