#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QMessageBox>
#include <QProcess>
#include <QTimer>
#include <QDebug>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    plugin(PLUGINPATH)
{
    qDebug() << "TweakFlashVer v0.4.0";

    settings = new QSettings("tweakflashver", "main");
    ui->setupUi(this);

    qDebug() << "Loading... ";
    loadFlashVers();
    qDebug() << "Loaded";
    if (ui->comboBox->count() == 0) // no version loaded from config
    {
        qDebug() << "Initializing settings";
        settings->beginGroup("global");
        settings->setValue("pluginpath", PLUGINPATH);
        settings->setValue("pluginsize", PLUGINSIZE);
        settings->endGroup();

#ifdef Q_WS_MAEMO_5
    #if QT_VERSION >= 0x040700
        QString inifile(":/ini/fremantle_pr13.ini");
    #else
        QString inifile(":/ini/fremantle_pr12.ini");
    #endif
#else
        QString inifile(":/ini/diablo.ini");
#endif
        QSettings rcsettings(inifile, QSettings::IniFormat);
        qDebug () << inifile << rcsettings.allKeys();
        foreach (QString group, rcsettings.childGroups())
        {
            rcsettings.beginGroup(group);
            settings->beginGroup(group);
            foreach (QString key, rcsettings.allKeys())
            {
                settings->setValue(key, rcsettings.value(key).toString());
                qDebug() << key << rcsettings.value(key).toString();
            }
            settings->endGroup();
            rcsettings.endGroup();
        }
        settings->sync();
        qDebug() << "Default settings saved";
        loadFlashVers();
        qDebug() << "Settings re-loaded";
    }

    qDebug() << "Checking plugin file";
    settings->beginGroup("global");
    if (!QFile::exists(settings->value("pluginpath", PLUGINPATH).toString()))
    {
        QMessageBox::critical(this, tr("Error"), tr("Flash plugin file not found"));
        qApp->quit();
    }
    else if (plugin.size() != settings->value("pluginsize", PLUGINSIZE).toUInt())
    {
        QMessageBox::critical(this, tr("Error"), tr("Flash plugin file size mismatch (%0 vs %1)").arg(plugin.size()).arg(settings->value("pluginsize", PLUGINSIZE).toUInt()));
        qApp->quit();
    }
    settings->endGroup();
    qDebug() << "Checks out OK";

    connect(ui->pushButton_2, SIGNAL(clicked()), this, SLOT(changeVer()));
    connect(ui->checkBox, SIGNAL(clicked(bool)), this, SLOT(setManualMode(bool)));
    connect(ui->comboBox, SIGNAL(currentIndexChanged(QString)), this, SLOT(updateFields()));

    readFlashVer();
    setManualMode(ui->checkBox->isChecked());
}

void MainWindow::on_actionWhy_triggered()
{
    QMessageBox::information(this, tr("Why you shouldn't pick the highest available version"), tr("The recommended is Linux 10.0 as that's the closest version to the one actually in the firmware (least chance of the site serving us with content we don't know how to interpret) and the one that usually is enough to pass 'requires Flash 10' checks. In reality, you will want to set the LOWEST version that will enable a site."));
}

void MainWindow::readFlashVer()
{
    if ( plugin.open(QIODevice::ReadOnly) && plugin.seek(OFFSET2)  )
    {
        ui->lineEdit->setText(plugin.read(PART2LEN));
    }
    plugin.close();
}


void MainWindow::setManualMode(bool b)
{
    ui->comboBox->setEnabled(!b);
    ui->lineEdit_2->setEnabled(b);
    ui->lineEdit_3->setEnabled(b);
    ui->lineEdit_4->setEnabled(b);
    if (!b)
        updateFields();
}

void MainWindow::updateFields()
{
    if (ui->comboBox->count() == 0)
        return;

    QString str = settings->value(ui->comboBox->currentText()+"/part2", "UNK   0,0,0,0").toString().right(PART2LEN-4); // -4 skips the platform part
    QStringList fields = str.split(",");
    ui->lineEdit_2->setText(fields[0]);
    ui->lineEdit_3->setText(fields[1]);
    ui->lineEdit_4->setText(fields[2]);
}

void MainWindow::changeVer()
{
    if (!QFile::exists(PLUGINBACKPUPPATH)) // create backup
        if (!QFile::copy(PLUGINPATH, PLUGINBACKPUPPATH))
        {
            QMessageBox::critical(this, tr("Error"), tr("Unable to create backup !"));
            return;
        }
    QString part1 = settings->value(ui->comboBox->currentText()+"/part1").toString();
    QString part2 = settings->value(ui->comboBox->currentText()+"/part2").toString();

    if (ui->checkBox->isChecked()) // manual
    {
        if (ui->lineEdit_2->text().length() + ui->lineEdit_3->text().length() + ui->lineEdit_4->text().length() != DIGITS ||
            ui->lineEdit_2->text().isEmpty() ||
            ui->lineEdit_3->text().isEmpty() ||
            ui->lineEdit_4->text().isEmpty() )
        {
            QMessageBox::warning(this, tr("Incorrect version string"), tr("The version string MUST contain %0 numbers in total, and every field has to have at least a single digit. That's the way it is. Sorry.").arg(DIGITS));
            return;
        }
        part1 = QString("%0.%1 r%2").arg(ui->lineEdit_2->text()).arg(ui->lineEdit_3->text()).arg(ui->lineEdit_4->text());
        part2 = QString("LNX %0,%1,%2,0").arg(ui->lineEdit_2->text()).arg(ui->lineEdit_3->text()).arg(ui->lineEdit_4->text());
    }

    if (part1.length() != PART1LEN || part2.length() != PART2LEN)
    {
        QMessageBox::critical(this, tr("Error"), tr("Version string length wrong, would break binary !"));
        return;
    }
    if (
        plugin.open(QIODevice::ReadWrite) &&
        plugin.seek(OFFSET1) &&
        plugin.write(part1.toAscii()) &&
        plugin.seek(OFFSET2) &&
        plugin.write(part2.toAscii()))
    {
        QMessageBox::information(this, tr("Success"), tr("New version info successfully written, restarting browser daemon."));
        QStringList param;
        param << "tablet-browser-daemon";
        QProcess::execute("/usr/sbin/stop", param);
        QProcess::execute("/usr/sbin/start", param);
    }
    else
    {
        QMessageBox::critical(this, tr("Error"), tr("Uh-oh. Error while opening/writing Flash plugin binary. I hope I didn't break it. Fear not, even if I did, I made a backup to %0, just copy it back to %1 as root").arg(PLUGINBACKPUPPATH).arg(PLUGINPATH));
    }

    plugin.close();
    readFlashVer();
}


void MainWindow::enableAutoRotate()
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif

}

void MainWindow::loadFlashVers()
{
    if (settings) {
        foreach (QString group, settings->childGroups())
        {
            if (group != "global")
                ui->comboBox->addItem(group);
        }
        updateFields();
    }
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::on_actionAbout_Qt_triggered()
{
    qApp->aboutQt();
}

void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::about(this, tr("Tweak Flash Plugin Version"), tr("This applications allows you to painlessly change the version of the Flash plugin reported to sites from a GUI. This will, however, NOT make Flash faster or sites genuinely requiring Flash 10 work. It is not a Flash 10.1 upgrade replacement. What it does is getting you around pesky Flash version checks. Enjoy. <BR><BR>Copyright (C) 2010 by Attila Csipa, licensed under the GNU GPL v2"));
}

void MainWindow::on_actionRestore_plugin_to_factory_state_triggered()
{
    QString ppath;
    if (QFile::exists(PLUGINBACKPUPPATH))
        ppath = PLUGINBACKPUPPATH;
    else if (QFile::exists(PLUGINBACKPUPPATH_LEGACY))
        ppath = PLUGINBACKPUPPATH_LEGACY;
    else {
            QMessageBox::critical(this, tr("Error"), tr("We're doomed ! Unable to find backup ! If you have removed it, please restore the file to %0").arg( PLUGINBACKPUPPATH));
            return;
    }

    if (!QFile(ppath).exists() || QFile(ppath).size() != PLUGINSIZE)
    {
            QMessageBox::critical(this, tr("Error"), tr("We're doomed ! Unable to find backup ! If you have removed it, please restore the file to %0").arg( PLUGINBACKPUPPATH));
            return;
    }

    if (!(QFile::remove(PLUGINPATH) && QFile::copy(ppath, PLUGINPATH)))
    {
        QMessageBox::critical(this, tr("Error"), tr("We're doomed ! Unable to restore backup !"));
        return;
    }
    readFlashVer();
    QMessageBox::information(this, tr("Success"), tr("Flash plugin restored to original state"));

}
