/* The MIT License:

Copyright (c) 2009 Ivan Gagis

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE. */

/* 
 * File:   FSFile.hpp
 * Author: ivan
 *
 * Created on February 28, 2009, 3:25 PM
 */

#pragma once

#include <stdio.h>

#ifdef __linux__
#include <dirent.h>
#include <sys/stat.h>
#endif

#include <ting/debug.hpp>
#include "File.hpp"

namespace file{

class FSFile : public file::File{
	FILE *handle;
public:
	FSFile(const std::string& pathName = std::string()) :
			File(pathName),
			handle(0)
	{}

	~FSFile(){
		this->Close();
	}

	//override
	virtual void Open(EMode mode){
		if(this->IsOpened())
			throw File::Exc("file already opened");

		const char* modeStr;
		switch(mode){
			case File::WRITE:
				modeStr="r+b";
				break;
			case File::CREATE:
				modeStr="w+b";
				break;
			case File::READ:
				modeStr="rb";
				break;
			default:
				throw File::Exc("unknown mode");
				break;
		}
		this->handle = fopen(this->Path().c_str(), modeStr);
		if(!this->handle)
			throw File::Exc("fopen() failed");

		//set open mode
		if(mode == CREATE)
			this->ioMode = WRITE;
		else
			this->ioMode = mode;

		this->isOpened = true;//set "opened" flag
	}



	//override
	virtual void Close(){
		if(!this->IsOpened())
			return;

		ASSERT(this->handle)

		fclose(this->handle);
        this->handle = 0;

		this->isOpened = false;
	}



	//override
	virtual unsigned Read(
			ting::Buffer<ting::u8>& buf,
			unsigned numBytesToRead = 0,
			unsigned offset = 0
		)
	{
		if(!this->IsOpened())
			throw File::Exc("file is not opened, cannot read");

		unsigned actualNumBytesToRead =
				numBytesToRead == 0 ? buf.SizeInBytes() : numBytesToRead;

		if(actualNumBytesToRead > buf.SizeInBytes() - offset)
			throw File::Exc("attempt to read more bytes than output buffer size");

		ASSERT(actualNumBytesToRead + offset <= buf.SizeInBytes())
		ASSERT(this->handle)
        unsigned numBytesRead = fread(&buf[offset], 1, actualNumBytesToRead, this->handle);
        if(numBytesRead != actualNumBytesToRead){//something happened
            if(!feof(this->handle))
				throw File::Exc("fread() error");//if it is not an EndOfFile then it is error
        }
        return numBytesRead;
	}



	//override
	//returns number of bytes actually written
	virtual unsigned Write(
			const ting::Buffer<ting::u8>& buf,
			unsigned numBytesToWrite = 0,
			unsigned offset = 0
		)
	{
		if(!this->IsOpened())
			throw File::Exc("file is not opened, cannot write");

		if(this->ioMode != WRITE)
			throw File::Exc("file is opened, but not in WRITE mode");

		unsigned actualNumBytesToWrite =
				numBytesToWrite == 0 ? buf.SizeInBytes() : numBytesToWrite;

		if(actualNumBytesToWrite > buf.SizeInBytes() - offset)
			throw File::Exc("attempt to write more bytes than passed buffer contains");

		ASSERT(actualNumBytesToWrite + offset <= buf.SizeInBytes())
		ASSERT(this->handle)
		unsigned bytesWritten = fwrite(&buf[offset], 1, actualNumBytesToWrite, this->handle);
		if(bytesWritten != actualNumBytesToWrite)//something bad has happened
			throw File::Exc("fwrite error");

		return bytesWritten;
	}



	//override
	virtual void SeekFwd(unsigned numBytesToSeek){
		if(!this->IsOpened())
			throw File::Exc("file is not opened, cannot seek");

		ASSERT(this->handle)
		if( fseek(this->handle, numBytesToSeek, SEEK_CUR) != 0 )
			throw File::Exc("fseek() failed");
	}



	//override
	virtual bool Exists()const{
		if(this->Path().size() == 0)
			return false;

		if(this->Path()[this->Path().size() - 1] == '/'){
#if defined(__linux__)
			DIR *pdir = opendir(this->Path().c_str());
			if(!pdir)
				return false;
			else{
				closedir(pdir);
				return true;
			}
#else
			throw File::Exc("Checking for directory existance is not supported");
#endif
		}else
			return this->File::Exists();
	}



	//override
	virtual void MkDir(){
		if(this->IsOpened())
			throw File::Exc("illegal state");

		if(this->Path().size() == 0 || this->Path()[this->Path().size() - 1] != '/')
			throw File::Exc("invalid directory name");

#if defined(__linux__)
//		TRACE(<< "creating directory = " << this->Path() << std::endl)
		umask(0);//clear umask for proper permissions of newly created directory
		if(mkdir(this->Path().c_str(), 0777) != 0)
			throw File::Exc("mkdir() failed");
#else
		throw File::Exc("creating directory is not supported");
#endif
	}
};

}//~namepspace
