//      timer.c
//
//      Copyright 2010 Michael Cronenworth <mike@cchtml.com>
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
//      MA 02110-1301, USA.

#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <gtk/gtk.h>
#include <libosso.h>

#include "stopish.h"

static long int timerStartTime = 0;
static long int timerSave = 0;

//Preferences
static int timerPrecision = TIMER_PRECISION_MINUTE;


char *stopish_get_time_string( void )
{
    char *timeBuffer;
    long int currentTime;
    int h, m, s, ss;

    // get current time
    currentTime = stopish_current_time(  );

    // calculate time format
    ss = ( currentTime - timerStartTime ) % 10;
    s = ( currentTime - timerStartTime ) / 10;
    m = s / 60;
    s = s % 60;
    h = m / 60;
    m = m % 60;

    if ( timerPrecision == TIMER_PRECISION_MINUTE && m > 60 ) {
        // rollover once we hit one hour
        stopish_set_time_start( stopish_current_time(  ) );
        h = m = s = ss = 0;
    }
    else if ( timerPrecision == TIMER_PRECISION_HOUR && h > 24 ) {
        // rollover once we hit one day
        stopish_set_time_start( stopish_current_time(  ) );
        h = m = s = ss = 0;
    }

    if ( stopish_get_type(  ) == STOPISH_TYPE_COUNTDOWN ) {
        h = -h;
        m = -m;
        s = -s;
        ss = -ss;
    }
    // countdown check
    if ( stopish_get_type(  ) == STOPISH_TYPE_COUNTDOWN &&
         ( h == 0 && m == 0 && s == 0 && ss == 0 ) )
        return NULL;

    timeBuffer = malloc( 64 );
    if ( timerPrecision == TIMER_PRECISION_MINUTE )
        sprintf( timeBuffer, "%.02d:%.02d.%.1d",
                 ( unsigned int ) m, ( unsigned int ) s,
                 ( unsigned int ) ss );
    else
        sprintf( timeBuffer, "%.02d:%.02d:%.02d.%.1d",
                 ( unsigned int ) h, ( unsigned int ) m,
                 ( unsigned int ) s, ( unsigned int ) ss );

    return timeBuffer;
}


void stopish_set_time_start( long int time )
{
    // reset timer to user-inputted time
    timerStartTime = time;
    timerSave = 0;
}


void stopish_timer_resume( void )
{
    timerStartTime = stopish_current_time(  );
    timerStartTime -= timerSave;
}


void stopish_timer_save( void )
{
    // save time counter
    timerSave = stopish_current_time(  ) - timerStartTime;
}


long int stopish_current_time( void )
{
    struct timeval tv;
    int s, us;

    gettimeofday( &tv, NULL );
    s = tv.tv_sec % 100000;
    us = tv.tv_usec / 100000;

    return ( s * 10 + us );
}


void stopish_timer_set_precision( int setting )
{
    timerPrecision = setting;
}


int stopish_timer_get_precision( void )
{
    return timerPrecision;
}
