// This file is part of the SpeedCrunch project
// Copyright (C) 2005-2006 Johan Thelin <e8johan@gmail.com>
// Copyright (C) 2007-2008 Helder Correia <helder.pereira.correia@gmail.com>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.


#include "keypad.hxx"

#include <QApplication>
#include <QGridLayout>
#include <QLocale>
#if QT_VERSION >= 0x040400 && defined(Q_WS_MAC) && !defined(QT_NO_STYLE_MAC)
#include <QMacStyle>
#endif
#include <QPushButton>
#include <QMessageBox>
#include <QStyle>
#include <QStyleOptionButton>


struct Keypad::Private
{
    // parent
    Keypad * p;

    // attributes
    char radixChar;
    bool portrait;

    // widgets
    QPushButton * key0;
    QPushButton * key1;
    QPushButton * key2;
    QPushButton * key3;
    QPushButton * key4;
    QPushButton * key5;
    QPushButton * key6;
    QPushButton * key7;
    QPushButton * key8;
    QPushButton * key9;
    QPushButton * keyAcos;
    QPushButton * keyAdd;
    QPushButton * keyAns;
    QPushButton * keyAsin;
    QPushButton * keyAtan;
    QPushButton * keyC;
    QPushButton * keyCos;
    QPushButton * keyDiv;
    QPushButton * keyDot;
    QPushButton * keyEE;
    QPushButton * keyEq;
    QPushButton * keyExp;
    QPushButton * keyFact;
    QPushButton * keyLn;
    QPushButton * keyLPar;
    QPushButton * keyMul;
    QPushButton * keyPerc;
    QPushButton * keyPi;
    QPushButton * keyRaise;
    QPushButton * keyRPar;
    QPushButton * keySin;
    QPushButton * keySqrt;
    QPushButton * keySub;
    QPushButton * keyTan;
    QPushButton * keyX;
    QPushButton * keyXEq;
    QPushButton * keyCloseKeypad;

    // methods
    void createButtons();
    void disableButtonFocus();
    void layoutButtons();
    void setButtonTooltips();
    void setUpButtonPressedSignal();
    void sizeButtons();
};


// public

Keypad::Keypad( char radixChar, QWidget * parent, bool portrait )
  : QWidget( parent ), d( new Keypad::Private )
{
  d->p = this;
  d->portrait = portrait;
//  QMessageBox msgBox;
//  if (portrait) msgBox.setText("portrait");
//  msgBox.exec();
  if ( radixChar == 'C' )
    d->radixChar = QLocale().decimalPoint().toAscii();
  else
    d->radixChar = radixChar;

  d->createButtons();
  d->sizeButtons();
  d->layoutButtons();
  d->setButtonTooltips();
  d->setUpButtonPressedSignal();
  d->disableButtonFocus();
  setLayoutDirection( Qt::LeftToRight );
}


Keypad::~Keypad()
{
  delete d;
}


char Keypad::radixChar() const
{
  return d->radixChar;
}


// public slots

void Keypad::setRadixChar( char c )
{
  if ( c == 'C' )
    c = QLocale().decimalPoint().toAscii();
  if ( d->radixChar != c )
  {
    d->radixChar = c;
    d->keyDot->setText( QString( c ) );
  }
}


void Keypad::retranslateText()
{
  d->setButtonTooltips();;
}


// protected slots

void Keypad::key0Pressed()     { emit buttonPressed( Keypad::Key0         ); };
void Keypad::key1Pressed()     { emit buttonPressed( Keypad::Key1         ); };
void Keypad::key2Pressed()     { emit buttonPressed( Keypad::Key2         ); };
void Keypad::key3Pressed()     { emit buttonPressed( Keypad::Key3         ); };
void Keypad::key4Pressed()     { emit buttonPressed( Keypad::Key4         ); };
void Keypad::key5Pressed()     { emit buttonPressed( Keypad::Key5         ); };
void Keypad::key6Pressed()     { emit buttonPressed( Keypad::Key6         ); };
void Keypad::key7Pressed()     { emit buttonPressed( Keypad::Key7         ); };
void Keypad::key8Pressed()     { emit buttonPressed( Keypad::Key8         ); };
void Keypad::key9Pressed()     { emit buttonPressed( Keypad::Key9         ); };
void Keypad::keyAcosPressed()  { emit buttonPressed( Keypad::KeyAcos      ); };
void Keypad::keyAddPressed()   { emit buttonPressed( Keypad::KeyPlus      ); };
void Keypad::keyAnsPressed()   { emit buttonPressed( Keypad::KeyAns       ); };
void Keypad::keyAsinPressed()  { emit buttonPressed( Keypad::KeyAsin      ); };
void Keypad::keyAtanPressed()  { emit buttonPressed( Keypad::KeyAtan      ); };
void Keypad::keyCosPressed()   { emit buttonPressed( Keypad::KeyCos       ); };
void Keypad::keyCPressed()     { emit buttonPressed( Keypad::KeyClear     ); };
void Keypad::keyDivPressed()   { emit buttonPressed( Keypad::KeyDivide    ); };
void Keypad::keyDotPressed()   { emit buttonPressed( Keypad::KeyRadixChar ); };
void Keypad::keyEEPressed()    { emit buttonPressed( Keypad::KeyEE        ); };
void Keypad::keyEqPressed()    { emit buttonPressed( Keypad::KeyEquals    ); };
void Keypad::keyExpPressed()   { emit buttonPressed( Keypad::KeyExp       ); };
void Keypad::keyFactPressed()  { emit buttonPressed( Keypad::KeyFactorial ); };
void Keypad::keyLnPressed()    { emit buttonPressed( Keypad::KeyLn        ); };
void Keypad::keyLParPressed()  { emit buttonPressed( Keypad::KeyLeftPar   ); };
void Keypad::keyMulPressed()   { emit buttonPressed( Keypad::KeyTimes     ); };
void Keypad::keyPercPressed()  { emit buttonPressed( Keypad::KeyPercent   ); };
void Keypad::keyPiPressed()    { emit buttonPressed( Keypad::KeyPi        ); };
void Keypad::keyRaisePressed() { emit buttonPressed( Keypad::KeyRaise     ); };
void Keypad::keyRParPressed()  { emit buttonPressed( Keypad::KeyRightPar  ); };
void Keypad::keySinPressed()   { emit buttonPressed( Keypad::KeySin       ); };
void Keypad::keySqrtPressed()  { emit buttonPressed( Keypad::KeySqrt      ); };
void Keypad::keySubPressed()   { emit buttonPressed( Keypad::KeyMinus     ); };
void Keypad::keyTanPressed()   { emit buttonPressed( Keypad::KeyTan       ); };
void Keypad::keyXPressed()     { emit buttonPressed( Keypad::KeyX         ); };
void Keypad::keyXEqPressed()   { emit buttonPressed( Keypad::KeyXEquals   ); };
void Keypad::keyCloseKeypadPressed() {

}

// private

void Keypad::Private::createButtons()
{
  key0      = new QPushButton( "0",    p );
  key1      = new QPushButton( "1",    p );
  key2      = new QPushButton( "2",    p );
  key3      = new QPushButton( "3",    p );
  key4      = new QPushButton( "4",    p );
  key5      = new QPushButton( "5",    p );
  key6      = new QPushButton( "6",    p );
  key7      = new QPushButton( "7",    p );
  key8      = new QPushButton( "8",    p );
  key9      = new QPushButton( "9",    p );
  keyAcos   = new QPushButton( "acos", p );
  keyAdd    = new QPushButton( "+",    p );
  keyAns    = new QPushButton( "ans",  p );
  keyAsin   = new QPushButton( "asin", p );
  keyAtan   = new QPushButton( "atan", p );
  keyC      = new QPushButton( "C",    p );
  keyCos    = new QPushButton( "cos",  p );
  keyDiv    = new QPushButton( "/",    p );
  keyEE     = new QPushButton( "E",    p );
  keyEq     = new QPushButton( "=",    p );
  keyExp    = new QPushButton( "exp",  p );
  keyFact   = new QPushButton( "!",    p );
  keyLn     = new QPushButton( "ln",   p );
  keyLPar   = new QPushButton( "(",    p );
  keyMul    = new QPushButton( "*",    p );
  keyPerc   = new QPushButton( "%",    p );
  keyRaise  = new QPushButton( "^",    p );
  keyRPar   = new QPushButton( ")",    p );
  keySin    = new QPushButton( "sin",  p );
  keySub    = new QPushButton( "-",    p );
  keyTan    = new QPushButton( "tan",  p );
  keyXEq    = new QPushButton( "x=",   p );
  keyX      = new QPushButton( "x",    p );

  keyDot    = new QPushButton( QString( radixChar ),     p );
  keyPi     = new QPushButton( QString::fromUtf8( "π" ), p );
  keySqrt   = new QPushButton( QString::fromUtf8( "√" ), p );
  keyCloseKeypad = new QPushButton( "close",    p );

  QFont f = key0->font();
  f.setBold( true );
  f.setPointSize( f.pointSize() + 3 );
  key0->setFont( f );
  key1->setFont( f );
  key2->setFont( f );
  key3->setFont( f );
  key4->setFont( f );
  key5->setFont( f );
  key6->setFont( f );
  key7->setFont( f );
  key8->setFont( f );
  key9->setFont( f );
  keyEq->setFont( f );
  keyMul->setFont( f );
  keyDiv->setFont( f );
  keyAdd->setFont( f );
  keySub->setFont( f );
  keyDot->setFont( f );

  keyCloseKeypad->hide();
  keyPi->hide();
  keyX->hide();
  keyAcos->hide();
 if (portrait) keyAns->show(); else keyAns->hide();
  keyAsin->hide();
  keyAtan->hide();
  keyCos->hide();
 if (portrait) keyC->show(); else keyC->hide();
  keyEE->hide();
 // keyEq->hide();
 // keyExp->hide();
 if (portrait) keyFact->hide();
  keyLn->hide();
  keyLPar->hide();
 // keyMul->hide();
 if (portrait) keyPerc->hide();
 // keyRaise->hide();
  keyRPar->hide();
  keySin->hide();
if (portrait)  keySqrt->hide();
// keySub->hide();
  keyTan->hide();
  keyXEq->hide();
}


void Keypad::Private::disableButtonFocus()
{
  key0->setFocusPolicy    ( Qt::NoFocus );
  key1->setFocusPolicy    ( Qt::NoFocus );
  key2->setFocusPolicy    ( Qt::NoFocus );
  key3->setFocusPolicy    ( Qt::NoFocus );
  key4->setFocusPolicy    ( Qt::NoFocus );
  key5->setFocusPolicy    ( Qt::NoFocus );
  key6->setFocusPolicy    ( Qt::NoFocus );
  key7->setFocusPolicy    ( Qt::NoFocus );
  key8->setFocusPolicy    ( Qt::NoFocus );
  key9->setFocusPolicy    ( Qt::NoFocus );
  keyAcos->setFocusPolicy ( Qt::NoFocus );
  keyAdd->setFocusPolicy  ( Qt::NoFocus );
  keyAns->setFocusPolicy  ( Qt::NoFocus );
  keyAsin->setFocusPolicy ( Qt::NoFocus );
  keyAtan->setFocusPolicy ( Qt::NoFocus );
  keyCos->setFocusPolicy  ( Qt::NoFocus );
  keyC->setFocusPolicy    ( Qt::NoFocus );
  keyDiv->setFocusPolicy  ( Qt::NoFocus );
  keyDot->setFocusPolicy  ( Qt::NoFocus );
  keyEE->setFocusPolicy   ( Qt::NoFocus );
  keyEq->setFocusPolicy   ( Qt::NoFocus );
  keyExp->setFocusPolicy  ( Qt::NoFocus );
  keyFact->setFocusPolicy ( Qt::NoFocus );
  keyLn->setFocusPolicy   ( Qt::NoFocus );
  keyLPar->setFocusPolicy ( Qt::NoFocus );
  keyMul->setFocusPolicy  ( Qt::NoFocus );
  keyPerc->setFocusPolicy ( Qt::NoFocus );
  keyPi->setFocusPolicy   ( Qt::NoFocus );
  keyRaise->setFocusPolicy( Qt::NoFocus );
  keyRPar->setFocusPolicy ( Qt::NoFocus );
  keySin->setFocusPolicy  ( Qt::NoFocus );
  keySqrt->setFocusPolicy ( Qt::NoFocus );
  keySub->setFocusPolicy  ( Qt::NoFocus );
  keyTan->setFocusPolicy  ( Qt::NoFocus );
  keyXEq->setFocusPolicy  ( Qt::NoFocus );
  keyX->setFocusPolicy    ( Qt::NoFocus );
}


void Keypad::Private::layoutButtons()
{
  int layoutSpacing = 3;

#if QT_VERSION >= 0x040400 && defined(Q_WS_MAC) && !defined(QT_NO_STYLE_MAC)
  // Workaround for a layouting bug in QMacStyle, Qt 4.4.0. Buttons would overlap
//  if ( qobject_cast<QMacStyle *>(p->style()) )
//    layoutSpacing = -1;
#endif

  QGridLayout * layout = new QGridLayout( p );
  layout->setMargin( 3 );
  layout->setSpacing( layoutSpacing );
  if (this->portrait){
      layout->addWidget( key7,      0, 1 );
      layout->addWidget( key8,      0, 2 );
      layout->addWidget( key9,      0, 3 );
      layout->addWidget( keyC,      0, 4 );
      layout->addWidget( keyRaise,  0, 5 );

      layout->addWidget( key4,      1, 1 );
      layout->addWidget( key5,      1, 2 );
      layout->addWidget( key6,      1, 3 );
      layout->addWidget( keyMul,    1, 4 );
      layout->addWidget( keyDiv,    1, 5 );

      layout->addWidget( key1,      2, 1 );
      layout->addWidget( key2,      2, 2 );
      layout->addWidget( key3,      2, 3 );
      layout->addWidget( keyAdd,    2, 4 );
      layout->addWidget( keySub,    2, 5 );

      layout->addWidget( key0,      3, 1 );
      layout->addWidget( keyDot,    3, 2 );
      layout->addWidget( keyExp,    3, 3 );
      layout->addWidget( keyAns,    3, 4 );
      layout->addWidget( keyEq,     3, 5 );
    } else {
      layout->addWidget( key0,      0, 0 );
      layout->addWidget( key1,      0, 1 );
      layout->addWidget( key2,      0, 2 );
      layout->addWidget( key3,      0, 3 );
      layout->addWidget( key4,      0, 4 );
      layout->addWidget( key5,      0, 5 );
      layout->addWidget( key6,      0, 6 );
      layout->addWidget( key7,      0, 7 );
      layout->addWidget( key8,      0, 8 );
      layout->addWidget( key9,      0, 9 );
      layout->addWidget( keyEq,     0, 10 );
    //  layout->addWidget( keyAcos,   2, 8 );
      layout->addWidget( keyAdd,    1, 0 );
    //  layout->addWidget( keyAns,    1, 6 );
    //  layout->addWidget( keyAsin,   1, 8 );
    //  layout->addWidget( keyAtan,   3, 8 );
    //  layout->addWidget( keyC,      0, 4 );
    //  layout->addWidget( keyCos,    2, 7 );
      layout->addWidget( keySub,    1, 1 );
      layout->addWidget( keyMul,    1, 2 );
      layout->addWidget( keyDiv,    1, 3 );
      layout->addWidget( keyDot,    1, 4 );
    //  layout->addWidget( keyEE,     1, 4 );
    //  layout->addWidget( keyLn,     0, 8 );
    //  layout->addWidget( keyLPar,   2, 4 );
    //  layout->addWidget( keyPi,     0, 6 );
      layout->addWidget( keyRaise,  1, 5 );
    //  layout->addWidget( keyRPar,   2, 5 );
    //  layout->addWidget( keySin,    1, 7 );
      layout->addWidget( keySqrt,   1, 6 );
      layout->addWidget( keyExp,    1, 9 );
      layout->addWidget( keyFact,   1, 7 );
      layout->addWidget( keyPerc,   1, 8 );
    //  layout->addWidget( keyTan,    3, 7 );
    //  layout->addWidget( keyX,      2, 6 );
    //  layout->addWidget( keyXEq,    3, 6 );
    //  layout->addWidget( keyCloseKeypad,   1, 9 );
    }
}


void Keypad::Private::setButtonTooltips()
{
  keyAcos->setToolTip( Keypad::tr( "Inverse cosine"      ) );
  keyAns->setToolTip ( Keypad::tr( "The last result"     ) );
  keyAsin->setToolTip( Keypad::tr( "Inverse sine"        ) );
  keyAtan->setToolTip( Keypad::tr( "Inverse tangent"     ) );
  keyC->setToolTip   ( Keypad::tr( "Clear expression"    ) );
  keyCos->setToolTip ( Keypad::tr( "Cosine"              ) );
  keyEE->setToolTip  ( Keypad::tr( "Scientific notation" ) );
  keyExp->setToolTip ( Keypad::tr( "Exponential"         ) );
  keyLn->setToolTip  ( Keypad::tr( "Natural logarithm"   ) );
  keySin->setToolTip ( Keypad::tr( "Sine"                ) );
  keySqrt->setToolTip( Keypad::tr( "Square root"         ) );
  keyTan->setToolTip ( Keypad::tr( "Tangent"             ) );
  keyXEq->setToolTip ( Keypad::tr( "Assign variable x"   ) );
  keyX->setToolTip   ( Keypad::tr( "The variable x"      ) );
}


void Keypad::Private::setUpButtonPressedSignal()
{
  connect( key0,     SIGNAL( clicked() ), p, SLOT( key0Pressed()    ) );
  connect( key1,     SIGNAL( clicked() ), p, SLOT( key1Pressed()    ) );
  connect( key2,     SIGNAL( clicked() ), p, SLOT( key2Pressed()    ) );
  connect( key3,     SIGNAL( clicked() ), p, SLOT( key3Pressed()    ) );
  connect( key4,     SIGNAL( clicked() ), p, SLOT( key4Pressed()    ) );
  connect( key5,     SIGNAL( clicked() ), p, SLOT( key5Pressed()    ) );
  connect( key6,     SIGNAL( clicked() ), p, SLOT( key6Pressed()    ) );
  connect( key7,     SIGNAL( clicked() ), p, SLOT( key7Pressed()    ) );
  connect( key8,     SIGNAL( clicked() ), p, SLOT( key8Pressed()    ) );
  connect( key9,     SIGNAL( clicked() ), p, SLOT( key9Pressed()    ) );
  connect( keyAcos,  SIGNAL( clicked() ), p, SLOT( keyAcosPressed() ) );
  connect( keyAdd,   SIGNAL( clicked() ), p, SLOT( keyAddPressed()  ) );
  connect( keyAns,   SIGNAL( clicked() ), p, SLOT( keyAnsPressed()  ) );
  connect( keyAsin,  SIGNAL( clicked() ), p, SLOT( keyAsinPressed() ) );
  connect( keyAtan,  SIGNAL( clicked() ), p, SLOT( keyAtanPressed() ) );
  connect( keyC,     SIGNAL( clicked() ), p, SLOT( keyCPressed()    ) );
  connect( keyCos,   SIGNAL( clicked() ), p, SLOT( keyCosPressed()  ) );
  connect( keyDiv,   SIGNAL( clicked() ), p, SLOT( keyDivPressed()  ) );
  connect( keyEE,    SIGNAL( clicked() ), p, SLOT( keyEEPressed()   ) );
  connect( keyEq,    SIGNAL( clicked() ), p, SLOT( keyEqPressed()   ) );
  connect( keyExp,   SIGNAL( clicked() ), p, SLOT( keyExpPressed()  ) );
  connect( keyFact,  SIGNAL( clicked() ), p, SLOT( keyFactPressed() ) );
  connect( keyLn,    SIGNAL( clicked() ), p, SLOT( keyLnPressed()   ) );
  connect( keyLPar,  SIGNAL( clicked() ), p, SLOT( keyLParPressed() ) );
  connect( keyMul,   SIGNAL( clicked() ), p, SLOT( keyMulPressed()  ) );
  connect( keyPerc,  SIGNAL( clicked() ), p, SLOT( keyPercPressed() ) );
  connect( keyRaise, SIGNAL( clicked() ), p, SLOT( keyRaisePressed()) );
  connect( keyRPar,  SIGNAL( clicked() ), p, SLOT( keyRParPressed() ) );
  connect( keySin,   SIGNAL( clicked() ), p, SLOT( keySinPressed()  ) );
  connect( keySub,   SIGNAL( clicked() ), p, SLOT( keySubPressed()  ) );
  connect( keyTan,   SIGNAL( clicked() ), p, SLOT( keyTanPressed()  ) );
  connect( keyXEq,   SIGNAL( clicked() ), p, SLOT( keyXEqPressed()  ) );
  connect( keyX,     SIGNAL( clicked() ), p, SLOT( keyXPressed()    ) );
  connect( keyDot,   SIGNAL( clicked() ), p, SLOT( keyDotPressed()  ) );
  connect( keyPi,    SIGNAL( clicked() ), p, SLOT( keyPiPressed()   ) );
  connect( keySqrt,  SIGNAL( clicked() ), p, SLOT( keySqrtPressed() ) );
  connect( keyCloseKeypad,  SIGNAL( clicked() ), p, SLOT( keyCloseKeypadPressed() ) );
}


void Keypad::Private::sizeButtons()
{
  // The same font in all buttons, so just pick one
  QFontMetrics fm = key0->fontMetrics();

  int maxWidth = fm.width( keyAcos->text() );
  int textHeight = qMax( fm.lineSpacing(), 14 );
  //int ls = fm.lineSpacing();

  QStyle::ContentsType type = QStyle::CT_ToolButton;
  QStyleOptionButton opt;
  opt.initFrom(keyAcos);
  QSize size = QSize( 100, 50 );
  QSize minSize = QSize( 40, 50 );
//  QSize size = keyAcos->style()->sizeFromContents( type, &opt,
//    QSize( int(maxWidth), int(textHeight) ).expandedTo( QApplication::globalStrut() ), keyAcos );

#ifdef Q_WS_X11
  // we would like to use the button size as indicated by the widget style,
  // but in some cases, e.g. KDE's Plastik or Oxygen, another few pixels
  // (typically 5) are added as the content margin, thereby making the button
  // incredibly wider than necessary
  // workaround: take only the hinted height, adjust the width ourselves (with our own margin)
//  maxWidth += 6;
//  int hh = size.height();
//  int ww = hh * 162 / 100; // use golden ratio
//  size = QSize(qMax(ww, maxWidth), hh);
#endif

  // limit the size of the buttons
  key0->setMaximumSize     ( size ); key0->setMinimumSize     ( minSize );
  key1->setMaximumSize     ( size ); key1->setMinimumSize     ( minSize );
  key2->setMaximumSize     ( size ); key2->setMinimumSize     ( minSize );
  key3->setMaximumSize     ( size ); key3->setMinimumSize     ( minSize );
  key4->setMaximumSize     ( size ); key4->setMinimumSize     ( minSize );
  key5->setMaximumSize     ( size ); key5->setMinimumSize     ( minSize );
  key6->setMaximumSize     ( size ); key6->setMinimumSize     ( minSize );
  key7->setMaximumSize     ( size ); key7->setMinimumSize     ( minSize );
  key8->setMaximumSize     ( size ); key8->setMinimumSize     ( minSize );
  key9->setMaximumSize     ( size ); key9->setMinimumSize     ( minSize );
  keyAcos->setMaximumSize  ( size ); keyAcos->setMinimumSize  ( minSize );
  keyAdd->setMaximumSize   ( size ); keyAdd->setMinimumSize   ( minSize );
  keyAns->setMaximumSize   ( size ); keyAns->setMinimumSize   ( minSize );
  keyAsin->setMaximumSize  ( size ); keyAsin->setMinimumSize  ( minSize );
  keyAtan->setMaximumSize  ( size ); keyAtan->setMinimumSize  ( minSize );
  keyC->setMaximumSize     ( size ); keyC->setMinimumSize     ( minSize );
  keyCos->setMaximumSize   ( size ); keyCos->setMinimumSize   ( minSize );
  keyDiv->setMaximumSize   ( size ); keyDiv->setMinimumSize   ( minSize );
  keyDot->setMaximumSize   ( size ); keyDot->setMinimumSize   ( minSize );
  keyEE->setMaximumSize    ( size ); keyEE->setMinimumSize    ( minSize );
  keyEq->setMaximumSize    ( size ); keyEq->setMinimumSize    ( minSize );
  keyExp->setMaximumSize   ( size ); keyExp->setMinimumSize   ( minSize );
  keyFact->setMaximumSize  ( size ); keyFact->setMinimumSize  ( minSize );
  keyLn->setMaximumSize    ( size ); keyLn->setMinimumSize    ( minSize );
  keyLPar->setMaximumSize  ( size ); keyLPar->setMinimumSize  ( minSize );
  keyMul->setMaximumSize   ( size ); keyMul->setMinimumSize   ( minSize );
  keyPerc->setMaximumSize  ( size ); keyPerc->setMinimumSize  ( minSize );
  keyPi->setMaximumSize    ( size ); keyPi->setMinimumSize    ( minSize );
  keyRaise->setMaximumSize ( size ); keyRaise->setMinimumSize ( minSize );
  keyRPar->setMaximumSize  ( size ); keyRPar->setMinimumSize  ( minSize );
  keySin->setMaximumSize   ( size ); keySin->setMinimumSize   ( minSize );
  keySqrt->setMaximumSize  ( size ); keySqrt->setMinimumSize  ( minSize );
  keySub->setMaximumSize   ( size ); keySub->setMinimumSize   ( minSize );
  keyTan->setMaximumSize   ( size ); keyTan->setMinimumSize   ( minSize );
  keyXEq->setMaximumSize   ( size ); keyXEq->setMinimumSize   ( minSize );
  keyX->setMaximumSize     ( size ); keyX->setMinimumSize     ( minSize );
}
