/*
 * This file is a part of Queen Beecon Widget
 * queen-beecon-advanced-tools-dialog: Utility for QBW Advanced Tools Dialog Box Management
 *
 * http://talk.maemo.org/showthread.php?t=45388
 *
 * Copyright (c) 2010 No!No!No!Yes! (Alessandro Peralma)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <hildon/hildon.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <string.h>
#include "queen-beecon-logger.h"
#include "queen-beecon.h"
#include "queen-beecon-settings-manager.h"
#include "queen-beecon-advanced-tools-dialog.h"
#include "queen-beecon-exec-async.h"

#include "he-helper.h"
#include "he-about-dialog.h"

extern gchar *qbwExecReason[];

guint checksum(gchar *buffer)
{
	int i;
	guint chksm=0;

	for (i=0;buffer[i]!='\0';i++) if ((guint)buffer[i]>32 && (guint)buffer[i]<127) chksm+=(guint)(buffer[i]^i)<<1;
	return(chksm);
}

void on_command_picker_value_changed(HildonPickerButton * button, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	int cmdSel = hildon_picker_button_get_active (HILDON_PICKER_BUTTON (button));
	gchar *c = g_strdup(self->priv->c_commands[cmdSel]);
	gtk_text_buffer_set_text(self->priv->cmdTestDriveBuffer,c,-1);
	g_free(c);

}

void on_logger_picker_value_changed(HildonPickerButton * button, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	QbwLoggerVerbosity prev_verbosity = qbw_logger(QBW_LOGGER_GET_VERBOSITY, QBW_LOGGER_VNULL, NULL);
	QbwLoggerVerbosity verbosity = hildon_picker_button_get_active (HILDON_PICKER_BUTTON (button));
	if (verbosity == QBW_LOGGER_VMAX) {
		qbw_logger(QBW_LOGGER_RESET_LOGFILE, QBW_LOGGER_VNULL, NULL);
		hildon_picker_button_set_active (HILDON_PICKER_BUTTON (button), prev_verbosity);
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Warning! Logfile Reset & Truncated!", self);
	} else {
		qbw_logger(QBW_LOGGER_SET_VERBOSITY, verbosity, NULL);
	}
}

void aboutWBt_clicked_cb(HildonButton *button, gpointer data)
{
	he_about_dialog_present(NULL,
                        "Queen Beecon Widget" /* auto-detect app name */,
                        "queen-beecon",
                        VERSION,
                        "Super powerful and customizable\nWidget generation engine",
                        "(c) 2010 No!No!No!Yes! Alessandro Peralma (... Hic Sunt Leones!)",
                        "http://wiki.maemo.org/Queen_BeeCon_Widget",
                        "https://garage.maemo.org/tracker/?group_id=1295",
                        "https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=QTSK26NUKNXFC");
}

void testDriveBt_clicked_cb(HildonButton * button, QueenBeecon *self)
{
#define LINE_BUFFER_LENGTH 32768
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	int cmdSel;
	FILE *fp = NULL;
	gchar *line;//AP Big Big command output buffer//AP
	size_t l;//AP
	GtkTextIter start, end;
	line = g_malloc0(LINE_BUFFER_LENGTH);
	gtk_text_buffer_get_bounds (self->priv->cmdTestDriveBuffer, &start, &end);
	gchar *c = g_strdup(gtk_text_buffer_get_text (self->priv->cmdTestDriveBuffer,&start, &end, FALSE));
	if (c[0] == '\0') {
		cmdSel = hildon_picker_button_get_active (HILDON_PICKER_BUTTON (self->priv->commandPicker));
		g_free(c);
		c = g_strdup(self->priv->c_commands[cmdSel]);
		gtk_text_buffer_set_text(self->priv->cmdTestDriveBuffer,c,-1);
	}
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) TEST DRIVE CMD %s %s", self, qbwExecReason[QBW_TEST_DRIVE], c);
	gchar *instanceCmdSubst = queen_beecon_g_command_param_subst(self, c, qbwExecReason[QBW_TEST_DRIVE]);
	fp = popen (instanceCmdSubst, "r");
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) ========> popen(%s) TEST DRIVE returned %p", self, instanceCmdSubst, fp);
	g_free(instanceCmdSubst);
	l=fread (line, 1, LINE_BUFFER_LENGTH, fp);
	line[LINE_BUFFER_LENGTH-1]='\000';//AP
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fread(l=%d,line=%s)", self, l, line);
	gint exitCode=pclose(fp);

	exitCode = (exitCode >> 8) & 0xff;
	gchar *buff = g_strdup_printf("[Exit Code = %d Output Size = %d]\n%s", (char)exitCode, l, line);//If command has output we issue a newline after Exit Code
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) to out buffer(l=%d,buff=%s)", self, strlen(buff), buff);
	gtk_text_buffer_set_text(self->priv->cmdOutputBuffer,buff,strlen(buff));

	gtk_text_buffer_get_bounds (self->priv->cmdOutputBuffer, &start, &end);
	if (g_strcmp0(gtk_text_buffer_get_text (self->priv->cmdOutputBuffer,&start, &end, FALSE), buff)) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) QBW Warning! Expected/Shown Command Output mismatch. Possible charset different from UTF-8!", self);
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Warning! Expected/Shown Command Output mismatch. Possible charset different from UTF-8!", self);
	}
	g_free(buff);
	g_free(line);
	g_free(c);
}

void copyCommandBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	GtkClipboard *clipboard = NULL;
	GtkTextIter start, end;
	gtk_text_buffer_get_bounds (self->priv->cmdTestDriveBuffer, &start, &end);
	gtk_text_buffer_select_range (self->priv->cmdTestDriveBuffer, &start, &end);
	/* do copy */
	clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	gtk_text_buffer_copy_clipboard (GTK_TEXT_BUFFER(self->priv->cmdTestDriveBuffer), clipboard);
}

void pasteCommandBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	GtkClipboard *clipboard = NULL;
	GtkTextIter start, end;
	gtk_text_buffer_get_bounds (self->priv->cmdTestDriveBuffer, &start, &end);
	gtk_text_buffer_select_range (self->priv->cmdTestDriveBuffer, &start, &end);
	  /* do paste */
	clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	gtk_text_buffer_paste_clipboard (GTK_TEXT_BUFFER (self->priv->cmdTestDriveBuffer), clipboard, NULL, TRUE);
}

void updateCommandBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	int cmdSel;
	GtkWidget *confirmDlg;
	gchar *confirmMsg;
	gint retcode;
	GtkTextIter start, end;

	gtk_text_buffer_get_bounds (self->priv->cmdTestDriveBuffer, &start, &end);
	gchar *c = g_strdup(gtk_text_buffer_get_text (self->priv->cmdTestDriveBuffer,&start, &end, FALSE));
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Current Test Drive Command Buffer [%s]", self, c);
	if ( !c || c[0]=='\0' ) {
		hildon_banner_show_informationf (NULL, NULL, "WARNING! Test Drive Buffer empty or not valid.\nUnable to update selected command.");
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Warning! Test Drive buffer empty or not valid. Unable to update selected command.", self);
		g_free(c);
		return;
	}

	cmdSel = hildon_picker_button_get_active (HILDON_PICKER_BUTTON (self->priv->commandPicker));
	if (cmdSel>=0) { //Update command in list
		confirmMsg = g_strdup_printf("Warning! You are about to update currently selected command with Test Drive buffer.\n\ntitle=[%s]\ncommand=[%s]\n\n\nAre you Sure?", self->priv->c_titles[cmdSel], self->priv->c_commands[cmdSel]);
		confirmDlg = hildon_note_new_confirmation (GTK_WINDOW(NULL), confirmMsg);
		retcode = gtk_dialog_run (GTK_DIALOG (confirmDlg));
		gtk_widget_destroy (confirmDlg);
		g_free(confirmMsg);
	} else { //Update command in instance
		confirmMsg = g_strdup_printf("WARNING! You are about to Update current Instance Command with Test Drive buffer.\n\nAre you Sure?");
		confirmDlg = hildon_note_new_confirmation (GTK_WINDOW(NULL), confirmMsg);
		retcode = gtk_dialog_run (GTK_DIALOG (confirmDlg));
		gtk_widget_destroy (confirmDlg);
		g_free(confirmMsg);
	}
	if (retcode != GTK_RESPONSE_OK) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Didn't like to confirm ... got scared!!! Beat it! :)",self);
		return;
	}

	if (cmdSel>=0) { //Update command in list
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Current Title Command to update [%s]", self, self->priv->c_titles[cmdSel]);
		EditCommand(self, g_strdup(self->priv->c_titles[cmdSel]), g_strdup(c), TRUE, cmdSel);
	} else { //Update command in instance
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Current Title Instance Command to update [%s]", self, self->priv->instanceTitle);
		EditCommand(self, g_strdup(self->priv->instanceTitle), g_strdup(c), TRUE, cmdSel);
	}
	queen_beecon_write_settings (self, TRUE); // We write commands to config file
	hildon_banner_show_informationf (NULL, NULL, "Command/Instance successfully updated.");
	g_free(c);
	self->priv->instanceUpdateOK=TRUE;
	gtk_dialog_response(GTK_DIALOG (self->priv->advancedDialog), GTK_RESPONSE_NONE);
}

void importCommandBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	GtkTextIter start, end;
	gboolean fileExists = FALSE;
	GKeyFile *keyFile = NULL;
	gint retcode;

	gtk_text_buffer_get_iter_at_line(self->priv->importExportBuffer, &start, 4);
	gtk_text_buffer_get_end_iter(self->priv->importExportBuffer, &end);
	gchar *c = g_strdup(gtk_text_buffer_get_text (self->priv->importExportBuffer,&start, &end, FALSE));
	if (!c || c[0]=='\0') {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) error with c=%p", self, c);
		hildon_banner_show_informationf (NULL, NULL, "WARNING! Import Buffer is empty!\nPlease paste relevant Command Export Data first.");
		g_free(c);
		return;
	}
	GtkWidget *confirmDlg = hildon_note_new_confirmation (GTK_WINDOW(NULL), "WARNING! You are about to import the Command Data currently in Import Buffer.\n\nAre you Sure?");	retcode = gtk_dialog_run (GTK_DIALOG (confirmDlg));	gtk_widget_destroy (confirmDlg);
	if (retcode != GTK_RESPONSE_OK) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Didn't like to confirm ... got scared!!! Beat it! :)",self);
		g_free(c);
		return;
	}
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) c=%s", self, c);

	guint calcChksum = checksum(c);//very trivial :)
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) import command data checksum=%d", self, calcChksum);
	g_free(c);
	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile %p", self, keyFile);

	gtk_text_buffer_get_iter_at_line(self->priv->importExportBuffer, &start, 0);
	c = g_strdup(gtk_text_buffer_get_text (self->priv->importExportBuffer,&start, &end, FALSE));

	fileExists = g_key_file_load_from_data (keyFile, c, strlen(c), G_KEY_FILE_KEEP_COMMENTS, NULL);
	if (!fileExists) {
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Import from ADB command buffer error g_key_file_load_from_data()", self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Import from ADB command buffer error g_key_file_load_from_data()", self);
		g_key_file_free (keyFile);
		g_free(c);
		return;
	}

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Export/Import buffer ready for KeyFile operations", self);
	// Get all the elements
	gdouble exportVersion = g_key_file_get_double (keyFile, "queen-beecon-header", "version", NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) exportVersion=%f", self, exportVersion);
	if (exportVersion!=QUEEN_BEECON_SETTINGS_VERSION) {
		hildon_banner_show_informationf (NULL, NULL, "WARNING! WRONG Version for Import Data!");
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) WRONG Version!!! Handle new version in next release possibly", self);
		g_key_file_free (keyFile);
		g_free(c);
		return;
	}

	guint stampChksum = g_key_file_get_integer (keyFile, "queen-beecon-header", "checksum", NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) stampChksum=%d", self, stampChksum);
	g_free(c);

	if (!g_key_file_has_group(keyFile, "queen-beecon-exported-command")) {
		hildon_banner_show_informationf (NULL, NULL, "ERROR! Buffer does not contain valid Command Export Data\nImport cancelled!");
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) ERROR! Buffer does not contain valid Command Export Data\nImport cancelled!", self);
		g_key_file_free (keyFile);
		return;
	}
	if (stampChksum==calcChksum||stampChksum==911) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) CHECKSUM is GOOD!!! ready to read and import command data!", self);
		gchar *title = g_strdup(g_key_file_get_string (keyFile, "queen-beecon-exported-command", "title", NULL));
		gchar *command = g_key_file_get_string (keyFile, "queen-beecon-exported-command", "command", NULL);
		g_key_file_free (keyFile);

		gboolean uniqueCommand; gint i;
		do {
			uniqueCommand=TRUE;
			for(i=0;i<self->priv->c_size;i++) {
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Testing unicity command title [%s] vs. [%s]", self, title, self->priv->c_titles[i]);
				if (!g_strcmp0(title, self->priv->c_titles[i])) {
					qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) MATCH!!!", self);
					uniqueCommand=FALSE;
					break;
				}
			}
			if (!uniqueCommand) {
				gchar *auxCmd = g_strconcat(title, "+", NULL);
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Generating new title [%s]", self, auxCmd);
				g_free(title);
				title=auxCmd;
			}
		} while (!uniqueCommand);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) About to call AddCommand title=[%s] command=[%s]", self, title, command);
		AddCommand(self, title, g_strdup(command), FALSE);

		//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
		queen_beecon_write_settings (self, FALSE); // We write commands to config file
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Rebuilding command selector adding new command", self);
		hildon_picker_button_set_active(HILDON_PICKER_BUTTON (self->priv->commandPicker), -1);
		hildon_touch_selector_set_active (hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), 0, -1);
		gtk_list_store_clear(GTK_LIST_STORE(hildon_touch_selector_get_model(hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), 0)));
		for(i=0;i<self->priv->c_size;i++) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Append Command [%d] [%s]", self, i, self->priv->c_titles[i]);
			hildon_touch_selector_append_text (hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), self->priv->c_titles[i]);
		}
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Rebuilding command to picker [%p] set active %d", self, self->priv->commandPicker, self->priv->c_size-1);
		hildon_touch_selector_set_active (hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), 0, self->priv->c_size-1);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Rebuilding command to picker [%p] set active %d", self, self->priv->commandPicker, self->priv->c_size-1);
		hildon_picker_button_set_active (HILDON_PICKER_BUTTON (self->priv->commandPicker), self->priv->c_size-1);

		c = g_strdup(self->priv->c_commands[self->priv->c_size-1]);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Filling command Test Drive Buffer [%p] with new command [%s]", self, self->priv->cmdTestDriveBuffer, c);
		gtk_text_buffer_set_text(self->priv->cmdTestDriveBuffer,c,-1);
		g_free(c);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Command successfully added to Command List and to Test Drive command input.", self);
		hildon_banner_show_informationf (NULL, NULL, "SUCCESS! Command added to Command List and to Test Drive command input as [%s]", title);
		//GtkWidget *info = hildon_note_new_information (NULL, "SUCCESS! Command added to Command List and to Test Drive command input.");gtk_dialog_run (GTK_DIALOG (info));gtk_object_destroy (GTK_OBJECT (info));
	} else {
		hildon_banner_show_informationf (NULL, NULL, "WARNING! Bad Checksum!\nPossibly corrupt Import Command Data!");
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Bad Checksum, possibly corrupt import command data!", self);
	}
}

void exportCommandBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	int cmdSel;
	GtkTextIter start, end;

	gtk_text_buffer_get_bounds (self->priv->cmdTestDriveBuffer, &start, &end);
	gchar *c = g_strdup(gtk_text_buffer_get_text (self->priv->cmdTestDriveBuffer,&start, &end, FALSE));
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) cmdTestDriveBuffer %s", self, c);
	if (!c || c[0]=='\0') {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) error with c=%p", self, c);
		hildon_banner_show_informationf (NULL, NULL, "WARNING! Command Buffer is empty!\nPlease select from List or type New Command first.");
		g_free(c);
		return;
	}
	cmdSel = hildon_picker_button_get_active (HILDON_PICKER_BUTTON (self->priv->commandPicker));
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) cmdSel %d", self, cmdSel);
	gchar *exportData;
	if (cmdSel>=0)
		exportData = g_strdup_printf("title=%s\ncommand=%s", self->priv->c_titles[cmdSel], c);
	else
		exportData = g_strdup_printf("title=%s\ncommand=%s", self->priv->instanceTitle, c);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) exportData (checksum here applies = %d", self, checksum(exportData));
	gchar* headerVersion = g_strdup_printf("[queen-beecon-header]\nversion=%f\nchecksum=%d\n[queen-beecon-exported-command]\n%s", QUEEN_BEECON_SETTINGS_VERSION, checksum(exportData),exportData);//very trivial :)
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) headerVersion %s", self, headerVersion);
	gtk_text_buffer_set_text (self->priv->importExportBuffer, headerVersion, -1);
	g_free(c);
	g_free(exportData);
	g_free(headerVersion);

}

void pasteFromClipboardADBExportImportBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	GtkClipboard *clipboard = NULL;
	GtkTextIter start, end;
	gtk_text_buffer_get_bounds (self->priv->importExportBuffer, &start, &end);
	gtk_text_buffer_select_range (self->priv->importExportBuffer, &start, &end);
	/* do paste */
	clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	gtk_text_buffer_paste_clipboard (GTK_TEXT_BUFFER (self->priv->importExportBuffer), clipboard, NULL, TRUE);
}

void copyToClipboardADBExportImportBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	GtkClipboard *clipboard = NULL;
	GtkTextIter start, end;
	gtk_text_buffer_get_bounds (self->priv->importExportBuffer, &start, &end);
	gtk_text_buffer_select_range (self->priv->importExportBuffer, &start, &end);
	/* do copy */
	clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	gtk_text_buffer_copy_clipboard (GTK_TEXT_BUFFER(self->priv->importExportBuffer), clipboard);
}

void importInstanceBt_clicked_cb ( HildonButton *button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	GtkTextIter start, end;
	gchar *confirmMsg;
	GtkWidget *confirmDlg;
	gint retcode;
	gboolean fileExists;
	GKeyFile *keyFile = NULL;

	gtk_text_buffer_get_iter_at_line(self->priv->importExportBuffer, &start, 4);
	gtk_text_buffer_get_end_iter(self->priv->importExportBuffer, &end);
	gchar *c = g_strdup(gtk_text_buffer_get_text (self->priv->importExportBuffer,&start, &end, FALSE));
	if (!c || c[0]=='\0') {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) error with c=%p", self, c);
		hildon_banner_show_informationf (NULL, NULL, "WARNING! Import Buffer is empty!\nPlease paste relevant Instance Export Data first.");
		g_free(c);
		return;
	}
	confirmMsg = g_strdup_printf("WARNING! You are about to Import the Instance Data currently in Import Buffer.\n\nAre you Sure?");
	confirmDlg = hildon_note_new_confirmation (GTK_WINDOW(NULL), confirmMsg);
	retcode = gtk_dialog_run (GTK_DIALOG (confirmDlg));
	gtk_widget_destroy (confirmDlg);
	g_free(confirmMsg);
	if (retcode != GTK_RESPONSE_OK) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Didn't like to confirm ... got scared!!! Beat it! :)",self);
		g_free(c);
		return;
	}

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) c=%s", self, c);

	guint calcChksum = checksum(c);//very trivial :)
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) import instance data checksum=%d", self, calcChksum);
	g_free(c);

	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile %p", self, keyFile);

	gtk_text_buffer_get_iter_at_line(self->priv->importExportBuffer, &start, 0);
	c = g_strdup(gtk_text_buffer_get_text (self->priv->importExportBuffer,&start, &end, FALSE));

	fileExists = g_key_file_load_from_data (keyFile, c, strlen(c), G_KEY_FILE_KEEP_COMMENTS, NULL);
	if (!fileExists) {
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Import from ADB command buffer error g_key_file_load_from_data()", self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Import from ADB instance buffer error g_key_file_load_from_data()", self);
		g_key_file_free (keyFile);
		g_free(c);
		return;
	}

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Export/Import buffer ready for KeyFile operations", self);
	// Get all the elements
	gdouble exportVersion = g_key_file_get_double (keyFile, "queen-beecon-header", "version", NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) exportVersion=%f", self, exportVersion);
	if (exportVersion!=QUEEN_BEECON_SETTINGS_VERSION) {
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW WRONG Version!!!", self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) WRONG Version!!!", self);
		g_key_file_free (keyFile);
		g_free(c);
		return;
	}

	guint stampChksum = g_key_file_get_integer (keyFile, "queen-beecon-header", "checksum", NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) stampChksum=%d", self, stampChksum);

	if (!g_key_file_has_group(keyFile, "queen-beecon-exported-instance")) {
		hildon_banner_show_informationf (NULL, NULL, "ERROR! Buffer does not contain valid Instance Export Data\nImport cancelled!");
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) ERROR! Buffer does not contain valid Instance Export Data\nImport cancelled!", self);
		g_key_file_free (keyFile);
		g_free(c);
		return;
	}
	if (stampChksum==calcChksum||stampChksum==911) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) CHECKSUM is GOOD!!! ready to read and import instance data!", self);
		//g_mutex_lock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);
		queen_beecon_read_settings ( self, TRUE, TRUE, c);

		gboolean uniqueCommand; gint i;
		do {
			uniqueCommand=TRUE;
			for(i=0;i<self->priv->c_size;i++) {
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Testing unicity command title [%s] vs. [%s]", self, self->priv->instanceTitle, self->priv->c_titles[i]);
				if (!g_strcmp0(self->priv->instanceTitle, self->priv->c_titles[i])) {
					qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) MATCH!!!", self);
					uniqueCommand=FALSE;
					break;
				}
			}
			if (!uniqueCommand) {
				gchar *auxCmd = g_strconcat(self->priv->instanceTitle, "+", NULL);
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) Generating new title [%s]", self, auxCmd);
				g_free(self->priv->instanceTitle);
				self->priv->instanceTitle=auxCmd;
			}
		} while (!uniqueCommand);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) AddCommand title=[%s] command=[%s]", self, self->priv->instanceTitle, self->priv->instanceCmd);
		AddCommand(self, g_strdup(self->priv->instanceTitle), g_strdup(self->priv->instanceCmd), TRUE);

		queen_beecon_write_settings (self, FALSE); // We write current added command to config file
		//g_mutex_unlock(QUEEN_BEECON_GET_CLASS(self)->config_file_mutex);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Rebuilding command selector adding new command", self);
		hildon_picker_button_set_active(HILDON_PICKER_BUTTON (self->priv->commandPicker), -1);
		hildon_touch_selector_set_active (hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), 0, -1);
		gtk_list_store_clear(GTK_LIST_STORE(hildon_touch_selector_get_model(hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), 0)));

		for(i=0;i<self->priv->c_size;i++) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Append Command [%d] [%s]", self, i, self->priv->c_titles[i]);
			hildon_touch_selector_append_text (hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), self->priv->c_titles[i]);
		}
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Rebuilding command to picker [%p] set active %d", self, self->priv->commandPicker, self->priv->c_size-1);
		hildon_touch_selector_set_active (hildon_picker_button_get_selector(HILDON_PICKER_BUTTON (self->priv->commandPicker)), 0, self->priv->c_size-1);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Rebuilding command to picker [%p] set active %d", self, self->priv->commandPicker, self->priv->c_size-1);
		hildon_picker_button_set_active (HILDON_PICKER_BUTTON (self->priv->commandPicker), self->priv->c_size-1);

		hildon_banner_show_informationf (NULL, NULL, "SUCCESS! Instance imported into current Beecon/Snippet!\nCommand added to command list as [%s]", self->priv->instanceTitle);
		//GtkWidget *info = hildon_note_new_information (NULL, "SUCCESS! Instance imported into current Beecon/Snippet!\nCommand added to command list.");gtk_dialog_run (GTK_DIALOG (info));gtk_object_destroy (GTK_OBJECT (info));
		self->priv->instanceImportOK=TRUE;
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Instance Settings loaded successfully", self);
		g_free(c);
		g_key_file_free (keyFile);
		gtk_dialog_response(GTK_DIALOG (self->priv->advancedDialog), GTK_RESPONSE_NONE);
	} else {
		hildon_banner_show_informationf (NULL, NULL, "WARNING! Bad Checksum!\nPossibly corrupt Import Instance Data!");
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Bad Checksum, possibly corrupt import data!", self);
		g_free(c);
		g_key_file_free (keyFile);
	}
}

void exportInstanceBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gchar *filename = NULL;
	gboolean fileExists;
	GKeyFile *keyFile = NULL;
	GError *error=NULL;
	gchar **instance_keys = NULL;
	gsize element_count;
	GtkTextIter start,end;
	int i;

	if(self->priv->widgetID == NULL) {
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Widget instance not initialized, not reading settings..", self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Widget instance not initialized, not reading settings..", self);
		return;
	}

	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile %p", self, keyFile);
	filename = g_strconcat (HOME_DIR, QUEEN_BEECON_SETTINGS_FILE, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) filename %s", self, filename);
	fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileExists %d", self, fileExists);

	gdouble settingsVer = g_key_file_get_double (keyFile, "config", "version", NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) settingsVer %f", self, settingsVer);

	/* Get all the elements */
	instance_keys = g_key_file_get_keys (keyFile, self->priv->widgetID, &element_count, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) instance_keys %p", self, instance_keys);
	if (instance_keys != NULL && element_count != 0) {
		gtk_text_buffer_set_text (self->priv->importExportBuffer, "", -1);
		gtk_text_buffer_get_iter_at_line(self->priv->importExportBuffer, &end, 0);
		for (i = 0; i < element_count; i++) {
			gchar *keyVal = g_strdup_printf("%s=%s\n", instance_keys [i], g_key_file_get_value(keyFile, self->priv->widgetID, instance_keys [i], &error));
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyVal %s", self, keyVal);
			gtk_text_buffer_insert (self->priv->importExportBuffer, &end, keyVal, -1);
			g_free(keyVal);
		}
		//Let's eval buffer checksum
		gtk_text_buffer_get_bounds (self->priv->importExportBuffer, &start, &end);
		gchar *c = g_strdup(gtk_text_buffer_get_text (self->priv->importExportBuffer,&start, &end, FALSE));
		gchar *headerVersion = g_strdup_printf("[queen-beecon-header]\nversion=%f\nchecksum=%d\n[queen-beecon-exported-instance]\n", settingsVer, checksum(c));//very trivial :)
		g_free(c);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) headerVersion %s", self, headerVersion);
		gtk_text_buffer_get_iter_at_line(self->priv->importExportBuffer, &end, 0);
		gtk_text_buffer_insert (self->priv->importExportBuffer, &end, headerVersion, -1);
		g_free(headerVersion);
	}
	g_key_file_free (keyFile);
	g_free (filename);

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Instance Settings exported", self);
}

void importFileBufferBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	GtkWidget *file_chooser = hildon_file_chooser_dialog_new (NULL, GTK_FILE_CHOOSER_ACTION_OPEN);
	//gtk_window_set_title (GTK_WINDOW (file_chooser), "Choose file to import from");
	if (gtk_dialog_run (GTK_DIALOG (file_chooser)) == GTK_RESPONSE_OK) {
		GtkTextIter start, end;
		GError *error = NULL;
		gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_chooser));

		if (!filename) {
			gtk_widget_destroy (file_chooser);
			return;
		}

		gchar *contents = NULL;
		if (!g_file_get_contents (filename, &contents, NULL, &error)) {
			hildon_banner_show_informationf (NULL, NULL, "WARNING! File Import failed: %s", error->message);
			g_free (filename);
			g_clear_error (&error);
			gtk_widget_destroy (file_chooser);
			return;
		}

		gtk_text_buffer_get_bounds (self->priv->importExportBuffer, &start, &end);
		gtk_text_buffer_delete (self->priv->importExportBuffer, &start, &end);
		gtk_text_buffer_insert (self->priv->importExportBuffer, &start, contents, -1);

		g_free (filename);
		g_free (contents);
	}
	gtk_widget_destroy (file_chooser);
}

void exportFileBufferBt_clicked_cb ( HildonButton * button, QueenBeecon *self )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);

	gchar *mydocsdir = g_strdup (g_getenv ("MYDOCSDIR"));
	if (!mydocsdir) mydocsdir = g_build_filename (g_get_home_dir (), "MyDocs", NULL);

	GtkWidget *file_chooser = hildon_file_chooser_dialog_new (NULL, GTK_FILE_CHOOSER_ACTION_SAVE);
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (file_chooser), mydocsdir);
	g_free (mydocsdir);

	if (gtk_dialog_run (GTK_DIALOG (file_chooser)) == GTK_RESPONSE_OK) {
		GtkTextIter start, end;
		GError *error = NULL;
		gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_chooser));

		if (!filename) {
			gtk_widget_destroy (file_chooser);
			return;
		}

		gtk_text_buffer_get_bounds (self->priv->importExportBuffer, &start, &end);

		gchar *contents = gtk_text_buffer_get_text (self->priv->importExportBuffer, &start, &end, FALSE);
		if (!contents) {
			hildon_banner_show_information (NULL, NULL, "Export failed");
			g_free (filename);
			gtk_widget_destroy (file_chooser);
			return;
		}

		if (!g_file_set_contents (filename, contents, -1, &error)) {
			hildon_banner_show_informationf (NULL, NULL, "WARNING! File Export failed: %s", error->message);
			g_clear_error (&error);
		}
		g_free (filename);
		g_free (contents);
	}
	gtk_widget_destroy (file_chooser);
}

gboolean queen_beecon_advanced_dialog ( QueenBeecon *self, gint *cmdSel )
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V1, "(%p) %s", self, G_STRFUNC);
	HildonGtkInputMode input_mode;
	gboolean reReadSettings = FALSE;
	gboolean timeToGoBack = FALSE;
	gchar *title = NULL;
	gint i;

	title = g_strconcat ("Advanced Settings and Tools (... Hic Sunt Leones!)", NULL);
	self->priv->advancedDialog = gtk_dialog_new_with_buttons (title, NULL, 0, NULL);
	g_free(title);

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Building advanced settings dialog %p", self, self->priv->advancedDialog);
	GtkWidget *settingsArea = gtk_dialog_get_content_area (GTK_DIALOG (self->priv->advancedDialog));
	GtkWidget *scroll = hildon_pannable_area_new ();
	g_object_set (scroll, "hscrollbar-policy", GTK_POLICY_NEVER, NULL);
	g_object_set (scroll, "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
	gtk_container_add (GTK_CONTAINER (settingsArea), scroll);
	gtk_window_set_default_size (GTK_WINDOW (self->priv->advancedDialog), -1, 380);

	GtkWidget *content_area = gtk_vbox_new (FALSE, 0);
	hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (scroll), content_area);

	GtkWidget *cmdSelectionLabel = gtk_label_new ("Command Test Drive");
	gtk_box_pack_start (GTK_BOX (content_area), cmdSelectionLabel, FALSE, FALSE, 0);

    GtkWidget *cmdTestDriveLabelHelp = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (cmdTestDriveLabelHelp), "<span font_desc=\"Nokia Sans 13\">Type command you would like to execute and press <u>Run</u> to see RAW Output in box below.</span>");//AP1
	gtk_box_pack_start (GTK_BOX (content_area), cmdTestDriveLabelHelp, TRUE, TRUE, 0);

	//BEGIN Command Selector

	GtkWidget *testDriveBox = gtk_hbox_new (FALSE, 0);

	GtkWidget *aboutW = hildon_button_new_with_text (HILDON_SIZE_AUTO_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
                                          NULL);
	hildon_button_set_image (HILDON_BUTTON (aboutW), gtk_image_new_from_stock (GTK_STOCK_INFO, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (aboutW), 0.5, 0.5, 0, 0);//AP1
	g_signal_connect (aboutW, "clicked", G_CALLBACK (aboutWBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (testDriveBox), aboutW, FALSE, FALSE, 0);

    GtkWidget *loggerSelector = hildon_touch_selector_new_text ();
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (loggerSelector), "Disable");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (loggerSelector), "v0");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (loggerSelector), "v1");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (loggerSelector), "v2");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (loggerSelector), "v3");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (loggerSelector), "v4");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (loggerSelector), "Reset");

    GtkWidget *loggerPicker = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_image (HILDON_BUTTON (loggerPicker), gtk_image_new_from_stock (GTK_STOCK_SELECT_ALL, GTK_ICON_SIZE_MENU));
    hildon_button_set_title (HILDON_BUTTON (loggerPicker), "Log:");
	hildon_button_set_alignment (HILDON_BUTTON (loggerPicker), 0, 0.5, 0, 0);
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (loggerPicker), HILDON_TOUCH_SELECTOR (loggerSelector));
	hildon_picker_button_set_active (HILDON_PICKER_BUTTON (loggerPicker), qbw_logger(QBW_LOGGER_GET_VERBOSITY, QBW_LOGGER_VNULL, NULL));
	gtk_box_pack_start (GTK_BOX (testDriveBox), loggerPicker, FALSE, FALSE, 0);

	GtkWidget *commandPickerSelector = hildon_touch_selector_new_text ();
	for(i=0;i<self->priv->c_size;i++)
	{
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (commandPickerSelector), self->priv->c_titles[i]);
	}
	self->priv->commandPicker = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_image (HILDON_BUTTON (self->priv->commandPicker), gtk_image_new_from_stock (GTK_STOCK_INDEX, GTK_ICON_SIZE_MENU));
	hildon_button_set_title (HILDON_BUTTON (self->priv->commandPicker), "Command:");
	hildon_button_set_alignment (HILDON_BUTTON (self->priv->commandPicker), 0, 0.5, 0, 0);
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (self->priv->commandPicker), HILDON_TOUCH_SELECTOR (commandPickerSelector));
	hildon_picker_button_set_active (HILDON_PICKER_BUTTON (self->priv->commandPicker), *cmdSel);
	gtk_box_pack_start (GTK_BOX (testDriveBox), self->priv->commandPicker, TRUE, TRUE, 0);

	GtkWidget *testDriveBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
                                          "Run");
	hildon_button_set_image (HILDON_BUTTON (testDriveBt), gtk_image_new_from_stock (GTK_STOCK_EXECUTE, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (testDriveBt), 0, 0.5, 0, 0);//AP1
	g_signal_connect (testDriveBt, "clicked", G_CALLBACK (testDriveBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (testDriveBox), testDriveBt, FALSE, FALSE, 0);


	GtkWidget *updateCmdBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
                                          "Save");
	hildon_button_set_image (HILDON_BUTTON (updateCmdBt), gtk_image_new_from_stock (GTK_STOCK_OK, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (updateCmdBt), 0, 0.5, 0, 0);//AP1
	g_signal_connect (updateCmdBt, "clicked", G_CALLBACK (updateCommandBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (testDriveBox), updateCmdBt, FALSE, FALSE, 0);

	GtkWidget *copyCmdBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
                                          NULL);
	hildon_button_set_image (HILDON_BUTTON (copyCmdBt), gtk_image_new_from_stock (GTK_STOCK_COPY, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (copyCmdBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (copyCmdBt), "Command");
	g_signal_connect (copyCmdBt, "clicked", G_CALLBACK (copyCommandBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (testDriveBox), copyCmdBt, FALSE, FALSE, 0);

    GtkWidget *pasteCmdBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
                                          NULL);
	hildon_button_set_image (HILDON_BUTTON (pasteCmdBt), gtk_image_new_from_stock (GTK_STOCK_PASTE, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (pasteCmdBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (pasteCmdBt), "Command");
	g_signal_connect (pasteCmdBt, "clicked", G_CALLBACK (pasteCommandBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (testDriveBox), pasteCmdBt, FALSE, FALSE, 0);

    gtk_box_pack_start (GTK_BOX (content_area), testDriveBox, TRUE, TRUE, 0);

	GtkWidget *cmdTestDrive = hildon_text_view_new ();
	input_mode = hildon_gtk_text_view_get_input_mode(GTK_TEXT_VIEW (cmdTestDrive));
	input_mode &= ~HILDON_GTK_INPUT_MODE_AUTOCAP; input_mode |= HILDON_GTK_INPUT_MODE_MULTILINE | HILDON_GTK_INPUT_MODE_FULL; hildon_gtk_text_view_set_input_mode(GTK_TEXT_VIEW (cmdTestDrive), input_mode);
	gtk_widget_set_size_request (GTK_WIDGET (cmdTestDrive), -1, 150);//AP
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (cmdTestDrive), GTK_WRAP_WORD_CHAR);
	gtk_box_pack_start (GTK_BOX (content_area), cmdTestDrive, TRUE, TRUE, 0);

	GtkWidget *cmdOutput = hildon_text_view_new ();
	input_mode = hildon_gtk_text_view_get_input_mode(GTK_TEXT_VIEW (cmdOutput));
	input_mode &= ~HILDON_GTK_INPUT_MODE_AUTOCAP; hildon_gtk_text_view_set_input_mode(GTK_TEXT_VIEW (cmdOutput), input_mode);
	gtk_widget_set_size_request (GTK_WIDGET (cmdOutput), -1, 150);//AP
	hildon_text_view_set_placeholder(HILDON_TEXT_VIEW (cmdOutput), "Exit Status & Command RAW Output will be displayed HERE");
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (cmdOutput), GTK_WRAP_WORD_CHAR);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(cmdOutput), FALSE);
	gtk_box_pack_start (GTK_BOX (content_area), cmdOutput, FALSE, FALSE, 0);

	GtkWidget *IEBufferLabel = gtk_label_new ("Import/Export Operations");
	gtk_box_pack_start (GTK_BOX (content_area), IEBufferLabel, FALSE, FALSE, 0);

	GtkWidget *cmdImportBox = gtk_hbox_new (TRUE, 0);

	GtkWidget *importCommandBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
                                          "Import Buffer\nto Command");
	hildon_button_set_image (HILDON_BUTTON (importCommandBt), gtk_image_new_from_stock (GTK_STOCK_LEAVE_FULLSCREEN, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (importCommandBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (importCommandBt), "Buffer to Command");
	g_signal_connect (importCommandBt, "clicked", G_CALLBACK (importCommandBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdImportBox), importCommandBt, TRUE, TRUE, 0);

	GtkWidget *pasteFromClipboardBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
										  "Paste Clipboard\nto Buffer");
	hildon_button_set_image (HILDON_BUTTON (pasteFromClipboardBt), gtk_image_new_from_stock (GTK_STOCK_PASTE, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (pasteFromClipboardBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (pasteFromClipboardBt), "Clipboard to Buffer");
	g_signal_connect (pasteFromClipboardBt, "clicked", G_CALLBACK (pasteFromClipboardADBExportImportBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdImportBox), pasteFromClipboardBt, TRUE, TRUE, 0);

    GtkWidget *importFileBufferBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                          NULL,
    									  "Import File\nto Buffer");
	hildon_button_set_image (HILDON_BUTTON (importFileBufferBt), gtk_image_new_from_stock (GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU));
    hildon_button_set_alignment (HILDON_BUTTON (importFileBufferBt), 0, 0.5, 0, 0);//AP1
    //hildon_button_set_value (HILDON_BUTTON (importFileBufferBt), "File to\nBuffer");
    g_signal_connect (importFileBufferBt, "clicked", G_CALLBACK (importFileBufferBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdImportBox), importFileBufferBt, TRUE, TRUE, 0);

	GtkWidget *importInstanceBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
                                          "Import Buffer\nto Beecon");
	hildon_button_set_image (HILDON_BUTTON (importInstanceBt), gtk_image_new_from_stock (GTK_STOCK_LEAVE_FULLSCREEN, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (importInstanceBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (importInstanceBt), "Buffer to Beecon");
	g_signal_connect (importInstanceBt, "clicked", G_CALLBACK (importInstanceBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdImportBox), importInstanceBt, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (content_area), cmdImportBox, TRUE, TRUE, 0);

	GtkWidget *cmdExportBox = gtk_hbox_new (TRUE, 0);

	GtkWidget *exportCommandBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
										  "Export Command\nto Buffer");
	hildon_button_set_image (HILDON_BUTTON (exportCommandBt), gtk_image_new_from_stock (GTK_STOCK_FULLSCREEN, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (exportCommandBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (exportCommandBt), "Command to Buffer");
	g_signal_connect (exportCommandBt, "clicked", G_CALLBACK (exportCommandBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdExportBox), exportCommandBt, TRUE, TRUE, 0);

	GtkWidget *copyToClipboardBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
										  "Copy Buffer\nto Clipboard");
	hildon_button_set_image (HILDON_BUTTON (copyToClipboardBt), gtk_image_new_from_stock (GTK_STOCK_COPY, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (copyToClipboardBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (copyToClipboardBt), "Buffer to Clipboard");
	g_signal_connect (copyToClipboardBt, "clicked", G_CALLBACK (copyToClipboardADBExportImportBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdExportBox), copyToClipboardBt, TRUE, TRUE, 0);

    GtkWidget *exportFileBufferBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
   										  NULL,
   										  "Export Buffer\nto File");
	hildon_button_set_image (HILDON_BUTTON (exportFileBufferBt), gtk_image_new_from_stock (GTK_STOCK_SAVE, GTK_ICON_SIZE_MENU));
    hildon_button_set_alignment (HILDON_BUTTON (exportFileBufferBt), 0, 0.5, 0, 0);//AP1
    //hildon_button_set_value (HILDON_BUTTON (exportFileBufferBt), "Buffer to File");
    g_signal_connect (exportFileBufferBt, "clicked", G_CALLBACK (exportFileBufferBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdExportBox), exportFileBufferBt, TRUE, TRUE, 0);

    GtkWidget *exportInstanceBt = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT |
                                          HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
										  NULL,
										  "Export Beecon\nto Buffer");
	hildon_button_set_image (HILDON_BUTTON (exportInstanceBt), gtk_image_new_from_stock (GTK_STOCK_FULLSCREEN, GTK_ICON_SIZE_MENU));
	hildon_button_set_alignment (HILDON_BUTTON (exportInstanceBt), 0, 0.5, 0, 0);//AP1
	//hildon_button_set_value (HILDON_BUTTON (exportInstanceBt), "Beecon to Buffer");
	g_signal_connect (exportInstanceBt, "clicked", G_CALLBACK (exportInstanceBt_clicked_cb), self);
    gtk_box_pack_start (GTK_BOX (cmdExportBox), exportInstanceBt, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (content_area), cmdExportBox, TRUE, TRUE, 0);

	GtkWidget *IEBufferLabelHelp = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (IEBufferLabelHelp), "<span font_desc=\"Nokia Sans 13\">Buffer for Command or Instance Export/Import ... handle with care!</span>");//AP1
	gtk_label_set_line_wrap (GTK_LABEL (IEBufferLabelHelp), TRUE);
	gtk_misc_set_alignment (GTK_MISC (IEBufferLabelHelp), 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX (content_area), IEBufferLabelHelp, FALSE, FALSE, 0);

	GtkWidget *cmdInstanceText = hildon_text_view_new ();
	input_mode = hildon_gtk_text_view_get_input_mode(GTK_TEXT_VIEW (cmdInstanceText));
	input_mode &= ~HILDON_GTK_INPUT_MODE_AUTOCAP; hildon_gtk_text_view_set_input_mode(GTK_TEXT_VIEW (cmdInstanceText), input_mode);
	gtk_widget_set_size_request (GTK_WIDGET (cmdInstanceText), -1, 200);//AP
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (cmdInstanceText), GTK_WRAP_WORD_CHAR);
	gtk_box_pack_start (GTK_BOX (content_area), cmdInstanceText, FALSE, FALSE, 0);

	//END

	gtk_widget_show_all (self->priv->advancedDialog);

	PangoFontDescription *fd = pango_font_description_from_string("Monospace bold 13");//AP1
	gtk_widget_modify_font(GTK_WIDGET (cmdInstanceText), fd);//AP1
	gtk_widget_modify_font(GTK_WIDGET (cmdTestDrive), fd);//AP1
	gtk_widget_modify_font(GTK_WIDGET (cmdOutput), fd);//AP1

	self->priv->importExportBuffer = hildon_text_view_get_buffer (HILDON_TEXT_VIEW (cmdInstanceText));
	self->priv->cmdTestDriveBuffer = hildon_text_view_get_buffer (HILDON_TEXT_VIEW (cmdTestDrive));
	self->priv->cmdOutputBuffer = hildon_text_view_get_buffer (HILDON_TEXT_VIEW (cmdOutput));

	g_signal_connect (G_OBJECT (loggerPicker), "value-changed",G_CALLBACK (on_logger_picker_value_changed), self);
	g_signal_connect (G_OBJECT (self->priv->commandPicker), "value-changed",G_CALLBACK (on_command_picker_value_changed), self);

	*cmdSel = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (commandPickerSelector), 0);
	if (*cmdSel>=0) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Working on global selector title=[%s] command=[%s]", self, self->priv->c_titles[*cmdSel], self->priv->c_commands[*cmdSel]);
		gchar *c = g_strdup(self->priv->c_commands[*cmdSel]);
		gtk_text_buffer_set_text(self->priv->cmdTestDriveBuffer,c,-1);
		g_free(c);
	} else {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Working on instance title=[%s] command=[%s]", self, self->priv->instanceTitle, self->priv->instanceCmd);
		gchar *c = g_strdup(self->priv->instanceCmd);
		gtk_text_buffer_set_text(self->priv->cmdTestDriveBuffer,c,-1);
		g_free(c);
	}
	self->priv->instanceUpdateOK=FALSE;
	self->priv->instanceImportOK=FALSE;

	queen_beecon_config_file_housekeeping(self); // Housekeeping!!! How HARD!!!! :)

	while (!timeToGoBack) {
		self->priv->currentDialog=self->priv->advancedDialog;
		int dialogRunResponse = gtk_dialog_run (GTK_DIALOG (self->priv->advancedDialog));
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Advanced Dialogue response =[%d]", self, dialogRunResponse);

		switch(dialogRunResponse) {
			case GTK_RESPONSE_ACCEPT:
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) GTK_RESPONSE_ACCEPT...", self);
				break;
			default:
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Time to go back... response %d", self, dialogRunResponse);
				if (self->priv->instanceUpdateOK)
					*cmdSel = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (commandPickerSelector), 0);
				timeToGoBack = TRUE;
				if (self->priv->instanceImportOK) reReadSettings=TRUE;
				break;
		}
	}
	gtk_widget_destroy (self->priv->advancedDialog);

	return(reReadSettings);
}
