#ifndef TRACKCOLLECTION_H
#define TRACKCOLLECTION_H

#include <QObject>
#include <QList>
#include "track.h"

namespace Spotify
{
    /**
     * Interface for collections of tracks such as Playlists, Searches, etc
     */
    class Session;
    class TrackCollection: public QObject
    {
        Q_OBJECT;

        public:
            TrackCollection( QObject* parent = 0 );
            virtual ~TrackCollection();

            /**
             * Call load() on all unloaded tracks.
             */
            virtual void updateTracks();

            /**
             * Get the track at a given index.
             * @param index Index
             * @return Pointer to the track at index.
             */
            virtual Spotify::Track* getTrack( int index );

            /**
             * Insert a list of tracks at a given position.
             * @param tracks A list containing pointers to Spotify::Track's.
             * @param position The start position.
             */
            virtual void insertTracks( QList<Spotify::Track*> tracks, const int position );

            /**
             * Add a track to the list. The track is appended to the end of the list.
             * @return The index of the added track.
             */
            virtual int addTrack( Spotify::Track* track );

            /**
             * Remove tracks from a set of positions. The removed tracks will be deleted.
             * @param positions Indices to remove tracks from.
             * @param num_tracks Number of tracks to remove.
             */
            virtual void removeTracks( const int* positions, int num_tracks );

            /**
             * Move tracks
             */
            virtual void moveTracks( const int* tracks, int num_tracks, int new_position );

            /**
             * Get tracks
             */
            virtual QList< Spotify::Track* > getTracks(bool includeUnavailable = true);

            /**
             * Load meta-data for the track collection.
             */
            virtual bool load() = 0;

            /**
             * Check if the track collection is loaded.
             */
            virtual bool isLoaded() const { return m_isLoaded; }

            /**
             * Get the number of tracks in this track collection.
             */
            virtual int getTrackCount( bool includeUnavailable = true ) const;

            /**
             * Get the total play time of the (loaded and available) tracks in this collection.
             */
            virtual QTime getTotalTime( bool includeUnavailable = true ) const;

            /**
             * Sort tracks by track name
             */
            void sortByTracks(bool ascending);

            /**
             * Sort tracks by artist names
             */
            void sortByArtists(bool ascending);

            /**
             * Sort tracks by album names
             */
            void sortByAlbums(bool ascending);

            /**
             * Sort tracks by popularity (descending)
             */
            void sortByPopularity();

            /**
             * Restore original track collection order
             */
            void restoreOrignalCollection();



        signals:
            void trackAdded( Spotify::Track* track, int position );
            void trackRemoved( int position );
            void trackMoved( int oldPosition, int newPosition );
            void updated();
            void loaded();

        protected:
            QList< Spotify::Track* > m_tracks;
            QList< Spotify::Track* > m_tracksOriginal;
            bool m_isLoaded;

};

} // namespace Spotify

#endif // TRACKCOLLECTION_H
