//lastfmscrobbler.h: Header file for the LastFMScrobbler class (scrobbling support for Last.Fm service using scrobbling / submission API 2.0)

#ifndef LASTFMSCROBBLER_H
#define LASTFMSCROBBLER_H

#include <QObject>
#include <QList>
#include <QByteArray>
#include <QDateTime>
#include <QCryptographicHash>
#include <QNetworkAccessManager>
#include <QUrl>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QDebug>
#include <QDomDocument>
#include <QFile>
#include <QDataStream>

//misc. defines

#define LASTFM_SCROBBLER_URL "http://ws.audioscrobbler.com/2.0/"

//submission API method defines
#define METHOD_AUTH_SESSION         "auth.getMobileSession"
#define METHOD_UPDATE_NOW_PLAYING   "track.updateNowPlaying"
#define METHOD_SCROBBLE_TRACKS      "track.scrobble"

#define SCROBBLES_CACHE_FILENAME    ".scrobbles_cache"

//Last.fm error code defines

#define ERROR_SERVICE_OFFLINE                 11
#define ERROR_SERVICE_TEMPORARILY_UNAVAILABLE 16



//scrobble track instance
class TrackData
{
public:
    QString m_strTrackName;
    QString m_strArtistName;
    QString m_strAlbumName;
    QString m_strAlbumArtist;
    QDateTime m_timestamp;
};

class LastFMScrobbler : public QObject
{
    Q_OBJECT

public:

    LastFMScrobbler(QObject* parent = 0);
    ~LastFMScrobbler();

    void setAPIKey(const QString &strAPIKey); //set API key value

    void setSecret(const QString &strSecret); //set secret value

    void setUserAgent(const QString& strUserAgent); //set user agent

    void setCachePath(const QString& strPath); //set path to directory to which queued scrobbles should be cached

    void setCache(bool enabled); //enabled / disable use of caching (note: caching path should be explicitely set)

    void authenticateSession(const QString& strUserID, const QString& strPassword); //request a new session using the supplied Last.FM user credentials.

    void updateNowPlaying(const QString& strTrackName, const QString& strArtistName, const QString& strAlbumName, const QString& strAlbumArtist); //update the client 'now playing' status

    void scrobbleTrack(); //scrobble the current track

    bool isSessionValid(); //return true in case session valid (i.e., Last.fm session key retrieved)

private:

    QString m_strAPIKey; //API key

    QString m_strSecret; //shared secret

    QString m_strSessionKey; //session key received upon successful authentication

    QString m_strUserAgent; //user agent identifying client

    QString m_strCachePath; //path to cache data

    QList<TrackData> m_ScrobblesList; //list / queue of (unsubmitted) scrobbles

    TrackData m_currTrack; //current track to be scrobbled

    QNetworkAccessManager * m_pNetworkManager; //network access manger instance

    QNetworkReply* m_pAuthReply;

    QNetworkReply* m_pNowPlayingReply;

    QNetworkReply* m_pScrobbleTracksReply;

    bool m_bCachingEnabled; //TRUE in case caching support of scrobbles enabled

    QByteArray getAPISignature(const QMap<QString,QString> &paramMap); //constructs and returns an API method signature

    QNetworkReply* postRequest( QMap<QString, QString> paramMap); //send of a request to the Last.fm webservice.

    void sendScrobbles(); //private method handling the actual sending of scrobbled tracks to Last.Fm webservice

    void cacheScrobbles(); //handles caching of currently queued scrobbles to disk

    void restoreScrobbles(); //restore cached scrobbles from disk

    bool getErrorResponse(const QByteArray& response, QString& strErrorDesc, qint32& code); //in case of error response, retrieve the associated last.fm error description and code. Returns TRUE in case error response can be parsed ok.

private slots:

    void OnRequestFinished(QNetworkReply *reply); //a network request has finished

signals:

    void authenticateSessionStatus(bool, const QString&); //signal emitted to notify authentication status

    void updateNowPlayingStatus(bool, const QString&); //signal emitted to notify status of now playing request

    void scrobbleTracksStatus(bool, const QString&); //signal emitted to notify status of scrobble tracks request


};

//non-member serialisation methods

QDataStream &operator<<(QDataStream &stream, const TrackData& track);
QDataStream &operator>>(QDataStream &stream, TrackData &track);

#endif // LASTFMSCROBBLER_H
