/*
 *  Nameday desktop applet (widget) for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <glib/gi18n-lib.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "nameday_dbparser.h"
#include "nameday_settings.h"

static void
nameday_settings_on_response (GtkDialog *dialog,
			      gint response_id,
			      gpointer data)
{
	NamedayDesktopApplet *priv;
	GtkWidget *picker;
	HildonTouchSelector *selector;
	GtkTreeIter iter;

	g_return_if_fail (data);

	priv = NAMEDAY_DESKTOP_APPLET (data);

	if (response_id == GTK_RESPONSE_OK) {
		if (priv->db)
			nameday_dbparser_close (priv);

		picker = g_object_get_data (G_OBJECT (dialog), "picker");
		
		if (hildon_picker_button_get_active (HILDON_PICKER_BUTTON (picker)) > -1) {
			selector = hildon_picker_button_get_selector (HILDON_PICKER_BUTTON (picker));
			if (hildon_touch_selector_get_selected (selector, 0, &iter)) {
				gtk_tree_model_get (hildon_touch_selector_get_model (selector, 0),
						    &iter, COLUMN_DB_FILENAME, &priv->db_filename, -1);

				/* save db file name */
		    		nameday_settings_save_conf (priv);
			}
		}

		nameday_dbparser_open (priv);

		/* forced update desktop applet */
		priv->previous_day = 0;
		nameday_desktop_applet_on_timeout (priv);
	}

	if (priv->settings_window)
		gtk_widget_destroy (priv->settings_window);

	priv->settings_window = NULL;
}

GtkWidget *
nameday_settings_dialog_create (NamedayDesktopApplet *priv)
{
	GtkWidget *dialog, *button, *selector;
	HildonTouchSelectorColumn *column;
	GtkListStore *calendars_list;
	GtkTreeIter iter;
        GtkTreeModel *model;
        gboolean iter_valid = TRUE;
        gchar *db_filename = NULL;

	g_return_val_if_fail (priv, NULL);

	/* settings dialog */
	dialog = gtk_dialog_new_with_buttons (_("Settings"),
	                                      NULL,
	                                      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
		                              _("Save"), GTK_RESPONSE_OK,
	                                      NULL);

	/* list of calendars */
	calendars_list = nameday_dbparser_calendar_list_create (priv);

	/* sort list of calendars */
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE(calendars_list),
					      COLUMN_DB_NAME, GTK_SORT_ASCENDING);

	/* touch selector */
	selector = hildon_touch_selector_new ();
	column = hildon_touch_selector_append_text_column (HILDON_TOUCH_SELECTOR (selector),
							   GTK_TREE_MODEL (calendars_list), TRUE);
	g_object_set (G_OBJECT (column), "text-column", COLUMN_DB_NAME, NULL);

	/* picker button */
	button = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (button), _("Nameday calendar"));
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button), HILDON_TOUCH_SELECTOR (selector));
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), button);
	g_object_set_data (G_OBJECT (dialog), "picker", button);

	g_signal_connect (dialog, "response",
			  G_CALLBACK (nameday_settings_on_response), priv);

	/* show settings dialog */
	gtk_widget_show_all (dialog);

	if (priv->db_filename) {
		/* select right row in the list of calendars */
	        model = hildon_touch_selector_get_model (HILDON_TOUCH_SELECTOR (selector), COLUMN_DB_NAME);
	        for (iter_valid = gtk_tree_model_get_iter_first (model, &iter); iter_valid;
	             iter_valid = gtk_tree_model_iter_next (model, &iter)) {
	                gtk_tree_model_get (model, &iter, COLUMN_DB_FILENAME, &db_filename, -1);
	                if (strcmp (db_filename, priv->db_filename) == 0) {
				hildon_touch_selector_select_iter (HILDON_TOUCH_SELECTOR (selector), COLUMN_DB_NAME,
								   &iter, FALSE);
	                        break;
	                }
	        }
	} else {
		/* unselect calendar */
		hildon_picker_button_set_active (HILDON_PICKER_BUTTON (button), -1);
	}

	return dialog;
}

void
nameday_settings_read_conf (NamedayDesktopApplet *priv)
{
	GError *err = NULL;

	g_return_if_fail (priv);
	g_return_if_fail (priv->gconf_client);

	/* read db file name */
	priv->db_filename = gconf_client_get_string(priv->gconf_client, GCONF_KEY_DB_FILENAME, &err);
	
	if (err != NULL) {
		g_warning ("nameday_settings_read_conf: Unable to read db filename %s", err->message);
	        g_error_free (err);
	        return;
	}
}

void
nameday_settings_save_conf (NamedayDesktopApplet *priv)
{
	GError *err = NULL;

	g_return_if_fail (priv);
	g_return_if_fail (priv->gconf_client);

	/* save db file name */
	gconf_client_set_string(priv->gconf_client, GCONF_KEY_DB_FILENAME, priv->db_filename, &err);
	
	if (err != NULL) {
		g_warning ("nameday_settings_save_conf: Unable to read db filename %s", err->message);
	        g_error_free (err);
	        return;
	}
}	

void
nameday_settings_init (NamedayDesktopApplet *priv)
{
	GConfClient *gconf_client = NULL;

	g_return_if_fail (priv);

        /* get the default client */
        gconf_client = gconf_client_get_default();
        
        if (gconf_client == NULL) {
		g_printerr ("nameday_settings_init: Unable to get GConf client");
		return;
        }

	priv->gconf_client = gconf_client;

        /* read configuration */
        nameday_settings_read_conf (priv);
}

void
nameday_settings_finalize (NamedayDesktopApplet *priv)
{
	g_return_if_fail (priv);

	if (priv->gconf_client) {
		g_object_unref (G_OBJECT (priv->gconf_client));
	}

	priv->gconf_client = NULL;
}
