/*
 *  moonphase home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <math.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>
#include <libhildondesktop/libhildondesktop.h>

#include "moonphase-home-widget.h"
#include "moonphase-util.h"

HD_DEFINE_PLUGIN_MODULE (MoonphaseHomePlugin, moonphase_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)
#define MOONPHASE_HOME_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MOONPHASE_TYPE_HOME_PLUGIN, MoonphaseHomePluginPrivate))
#define MOONPHASE_SETTINGS_FILE "/.moonphase_home_plugin"

struct _MoonphaseHomePluginPrivate
{
  gchar* iD;
  GdkPixbuf* background_moon;
  GdkPixbuf* foreground_moon;
  guint animation_timer_id;
  gint opacity;
  gint size;
  gdouble scale_factor;
  double phases[5];
  gboolean phases_visible;
  GtkWidget* moon_actor[5];
};

struct _moonphases_data_t
{
  int index;
  double fraction;
  MoonphaseHomePlugin* desktop_plugin;
};

typedef struct _moonphases_data_t moonphases_data_t;

static gboolean
moonphase_phases_animation(gpointer user_data);

static GtkWidget*
build_ui(void)
{
  GtkWidget* contents = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  gtk_widget_show(contents);
  return contents;
}

static void
moonphase_update(GtkWidget* widget,
		 gpointer data)
{
  MoonphaseHomePlugin *desktop_plugin = MOONPHASE_HOME_PLUGIN(widget);
  gtk_widget_queue_draw(widget);
}

static double
_get_moonphase_fraction()
{
  time_t now = time(NULL);
  double  pphase;
  double  mage;
  double  dist;
  double  angdia;
  double  sudist;
  double  suangdia;
  struct tm* tm_now = gmtime(&now);
  double pdate = jtime(tm_now);
  return phase(pdate, &pphase, &mage, &dist, &angdia, &sudist, &suangdia);
}

static void
_moonphase_render_phase(cairo_t* cr, double fraction)
{
  cairo_new_path(cr);
  int width = 256;
  int height = 256;
  cairo_move_to(cr, width/2.0, 0);
  if(fraction <= 0.5)
  {
    cairo_curve_to(cr, width * 1.25, 0, 
		   width * 1.25, height, 
		   width/2, height);
  }
  else
  {
    fraction-=0.5;
    cairo_curve_to(cr, 
		   -1/4.0*width, 0,
		   -1/4.0*width, height, 
		   width/2, height);
  }

  cairo_curve_to(cr, 
		 (-3 * fraction +  5/4.0)*width, height, 
		 (-3 * fraction +  5/4.0)*width,0,
		 width/2, 0);

  cairo_close_path(cr);
  cairo_fill(cr);
}

static void
_calculate_phases_date(MoonphaseHomePlugin* desktop_plugin)
{
  time_t now = time(NULL);
  struct tm* tm_now = gmtime(&now);
  double pdate = jtime(tm_now);
  phasehunt(pdate + 0.5, desktop_plugin->priv->phases);
}

static gboolean
moonphase_expose(GtkWidget* widget, GdkEventExpose *event)
{  
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(widget);
  double fraction = _get_moonphase_fraction();
  _calculate_phases_date(desktop_plugin);
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_scale(cr,
	      desktop_plugin->priv->size/256.0, 
	      desktop_plugin->priv->size/256.0);
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->priv->background_moon, 0, 0);
  cairo_paint(cr);
  gdk_cairo_set_source_pixbuf(cr, desktop_plugin->priv->foreground_moon, 0, 0);
  _moonphase_render_phase(cr, fraction);
  cairo_destroy(cr);
  int i;
  for(i=0;i<5;++i)
    gtk_widget_queue_draw(desktop_plugin->priv->moon_actor[i]);

  return GTK_WIDGET_CLASS(moonphase_home_plugin_parent_class)->expose_event(widget, event);
}

static gchar*
_day_month_text_for_phase(double julian_day)
{
  long yy;
  int mm;
  int dd;
  jyear(julian_day, &yy, &mm, &dd);
  struct tm now_tm;
  now_tm.tm_year = yy +1900;
  now_tm.tm_mon = mm-1;
  now_tm.tm_mday = dd;
  now_tm.tm_hour = now_tm.tm_min = now_tm.tm_sec = 0;
  gchar buf[255];
  strftime(buf, 255,dgettext("hildon-libs","wdgt_va_date_short"), &now_tm);
  return g_strdup(buf);
}

static gboolean
_render_phase(GtkWidget* widget, GdkEventExpose *event, gpointer user_data)
{
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  PangoLayout *pangoLayout;
  PangoFontDescription *fontDescription;
  pangoLayout = pango_cairo_create_layout(cr);
  const char* font = "Sans 100";
  fontDescription = pango_font_description_from_string(font);
  pango_layout_set_font_description(pangoLayout, fontDescription);
  pango_font_description_free(fontDescription);

  pango_layout_set_width(pangoLayout, 85*1024);

  moonphases_data_t* moonphases_data = (moonphases_data_t*)user_data;
  int index = moonphases_data->index;
  gchar* label_text = _day_month_text_for_phase(moonphases_data->desktop_plugin->priv->phases[index]);

  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  cairo_scale(cr,
	      moonphases_data->desktop_plugin->priv->size/(5*256.0), 
	      moonphases_data->desktop_plugin->priv->size/(5*256.0));

  gdk_cairo_set_source_pixbuf(cr, moonphases_data->desktop_plugin->priv->background_moon, 0, 0);
  cairo_paint(cr);

  gdk_cairo_set_source_pixbuf(cr, moonphases_data->desktop_plugin->priv->foreground_moon, 0, 0);
  _moonphase_render_phase(cr, moonphases_data->fraction);

  int width;
  int height;
  pango_layout_set_text(pangoLayout, label_text, -1);
  pango_layout_get_pixel_size(pangoLayout, &width, &height);

  cairo_set_operator(cr, CAIRO_OPERATOR_DEST_OVER);
  cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 0.8);
  cairo_rectangle(cr, 128.0, 128-height/2.0, 3*width, height);
  cairo_fill(cr);
  cairo_move_to(cr, 
		256.0+width/2.0, 
		128.0-height/2.0);
  cairo_set_operator(cr, CAIRO_OPERATOR_ADD);
  cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
  pango_cairo_show_layout(cr, pangoLayout);
  g_free(label_text);
  cairo_destroy(cr);
  return TRUE;
}

static void
_read_settings(MoonphaseHomePlugin* desktop_plugin)
{
  GKeyFile *keyFile;
  gchar* fileName;
  gboolean fileExists;

  keyFile = g_key_file_new();
  fileName = g_strconcat(g_get_home_dir(), MOONPHASE_SETTINGS_FILE, NULL);
  fileExists = g_key_file_load_from_file (keyFile, fileName, G_KEY_FILE_KEEP_COMMENTS, NULL);
  if(fileExists)
  {

    GError *error = NULL;
    desktop_plugin->priv->size = g_key_file_get_integer(keyFile, desktop_plugin->priv->iD, "size", &error);
  }
  g_key_file_free(keyFile);
  g_free(fileName);
}

static void
_set_colormap_for_actor(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
}

static GtkWidget*
_create_actor(MoonphaseHomePlugin* desktop_plugin, int index, double fraction)
{
  GtkWidget* actor = hildon_animation_actor_new();
  GtkWidget *image = NULL;
  image = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(image), FALSE);
  gtk_widget_set_size_request(actor, desktop_plugin->priv->size, desktop_plugin->priv->size/5.0);
  moonphases_data_t* moonphases_data = g_new0(moonphases_data_t,1);
  moonphases_data->desktop_plugin = desktop_plugin;
  moonphases_data->index = index;
  moonphases_data->fraction = fraction;

  g_signal_connect(GTK_WIDGET(image), "expose_event", G_CALLBACK(_render_phase), moonphases_data);
  gtk_container_add(GTK_CONTAINER(actor), image);
  hildon_animation_actor_set_parent(HILDON_ANIMATION_ACTOR(actor), GTK_WINDOW(desktop_plugin));
  hildon_animation_actor_set_position(HILDON_ANIMATION_ACTOR (actor), 
				      0, 
				      index*desktop_plugin->priv->size/5.0);
  hildon_animation_actor_set_opacity(HILDON_ANIMATION_ACTOR (actor), 0);
  hildon_animation_actor_set_scale(HILDON_ANIMATION_ACTOR (actor), 0.001, 0.001);
  _set_colormap_for_actor(actor);  
  gtk_widget_show_all(actor);
  return actor;
}

static void
_init_moonphases_actor(gpointer user_data)
{
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(user_data);
  desktop_plugin->priv->moon_actor[0] = _create_actor(desktop_plugin, 0, 0);
  desktop_plugin->priv->moon_actor[1] = _create_actor(desktop_plugin, 1, 0.25);
  desktop_plugin->priv->moon_actor[2] = _create_actor(desktop_plugin, 2, 0.50);
  desktop_plugin->priv->moon_actor[3] = _create_actor(desktop_plugin, 3, 0.75);
  desktop_plugin->priv->moon_actor[4] = _create_actor(desktop_plugin, 4, 1.00);
}

static void
moonphase_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(widget);
  desktop_plugin->priv->iD = hd_home_plugin_item_get_applet_id (HD_HOME_PLUGIN_ITEM (widget));
  _read_settings(desktop_plugin);
  gtk_window_resize(GTK_WINDOW(widget),
		    desktop_plugin->priv->size,
		    desktop_plugin->priv->size);
  _init_moonphases_actor(desktop_plugin);
  GTK_WIDGET_CLASS(moonphase_home_plugin_parent_class)->realize(widget);
}

static void
moonphase_finalize(GObject* object)
{
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(object);
  g_free(desktop_plugin->priv->iD);
  g_object_unref(desktop_plugin->priv->background_moon);
  g_object_unref(desktop_plugin->priv->foreground_moon);
  g_free(desktop_plugin->priv);
  G_OBJECT_CLASS(moonphase_home_plugin_parent_class)->finalize(object);
}

static gboolean
_start_animation(gpointer user_data)
{
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(user_data);
  desktop_plugin->priv->animation_timer_id =
    g_timeout_add(50,
		  moonphase_phases_animation, desktop_plugin);
  return FALSE;
}

static gboolean
moonphase_phases_animation(gpointer user_data)
{
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(user_data);
  if(desktop_plugin->priv->opacity <= 256 && 
     desktop_plugin->priv->opacity >= 0)
  {
    int i;
    for(i=0;i<5;++i)
    {
      hildon_animation_actor_set_opacity(HILDON_ANIMATION_ACTOR(desktop_plugin->priv->moon_actor[i]),
					 desktop_plugin->priv->opacity);
      hildon_animation_actor_set_scale(HILDON_ANIMATION_ACTOR(desktop_plugin->priv->moon_actor[i]),
				       desktop_plugin->priv->scale_factor,
				       desktop_plugin->priv->scale_factor);
      hildon_animation_actor_set_position(HILDON_ANIMATION_ACTOR(desktop_plugin->priv->moon_actor[i]),
					  -desktop_plugin->priv->scale_factor*desktop_plugin->priv->size/20.0,
					  desktop_plugin->priv->size/5.0*((desktop_plugin->priv->scale_factor-1)*(i-3) 
									  - desktop_plugin->priv->scale_factor/4.0 + i));

    }
    if(desktop_plugin->priv->phases_visible)
    {
      desktop_plugin->priv->opacity-=32;
      desktop_plugin->priv->scale_factor+=0.125;
    }
    else
    {
      desktop_plugin->priv->opacity+=32;
      desktop_plugin->priv->scale_factor-=0.125;
    }
    return TRUE;
  }
  desktop_plugin->priv->phases_visible = !desktop_plugin->priv->phases_visible;
  if(desktop_plugin->priv->phases_visible)
  {
    g_source_remove(desktop_plugin->priv->animation_timer_id);
    desktop_plugin->priv->opacity = 256;
    desktop_plugin->priv->scale_factor = 1;
    desktop_plugin->priv->animation_timer_id = g_timeout_add(3500,
							     _start_animation, desktop_plugin);
  }
  else
  {
    desktop_plugin->priv->animation_timer_id = 0;
  }
  return FALSE;
}

static void
_show_hide_moonphases(GtkWidget* widget, GdkEventButton *event, gpointer user_data)
{
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(user_data);
  
  if(desktop_plugin->priv->animation_timer_id==0)
  {
    desktop_plugin->priv->opacity = 0;
    desktop_plugin->priv->scale_factor = 2;    
    _start_animation(desktop_plugin);
  }
}

static void
_save_settings(MoonphaseHomePlugin* desktop_plugin)
{
  GKeyFile* keyFile;
  gchar *fileData;
  FILE *iniFile;
  gsize size;
  gchar *filename;

  keyFile = g_key_file_new();
  filename = g_strconcat (g_get_home_dir(), MOONPHASE_SETTINGS_FILE, NULL);
  g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
  g_key_file_set_integer(keyFile, desktop_plugin->priv->iD, "size", desktop_plugin->priv->size);
  
  fileData = g_key_file_to_data (keyFile, &size, NULL);
  g_file_set_contents(filename, fileData, size, NULL);

  g_key_file_free (keyFile); 
  g_free(fileData); 
  g_free(filename); 
}

static void
_show_settings_dialog(GtkWidget* widget, gpointer user_data)
{
  MoonphaseHomePlugin* desktop_plugin = MOONPHASE_HOME_PLUGIN(widget);
  GtkWidget *dialog;
  dialog = gtk_dialog_new_with_buttons("Settings",
				       NULL,
				       0,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_ACCEPT,
				       NULL);
  GtkWidget* slider = hildon_gtk_hscale_new();
  gtk_adjustment_configure(gtk_range_get_adjustment(GTK_RANGE(slider)), 
			   desktop_plugin->priv->size,
			   64, 320,
			   1, 1, 1);
  gtk_scale_set_value_pos(GTK_SCALE(slider), GTK_POS_RIGHT);
  gtk_scale_set_draw_value(GTK_SCALE(slider), TRUE);
  gtk_scale_set_digits(GTK_SCALE(slider), 0);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), slider, TRUE, TRUE, 0);
  gtk_widget_show_all(dialog);

  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
  {
    desktop_plugin->priv->size = gtk_range_get_value(GTK_RANGE(slider));
    gtk_window_resize(GTK_WINDOW (desktop_plugin), 
		      desktop_plugin->priv->size,
		      desktop_plugin->priv->size);
    int i = 0;
    for(i = 0;i<5;++i)
    {
      gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin->priv->moon_actor[i]), 
				  3*desktop_plugin->priv->size/5.0, 
				  desktop_plugin->priv->size/5.0);
      hildon_animation_actor_set_position(HILDON_ANIMATION_ACTOR(desktop_plugin->priv->moon_actor[i]), 
					  0, 
					  i*desktop_plugin->priv->size/5.0);

    }
    _save_settings(desktop_plugin);
  }  
  gtk_widget_destroy (dialog);
}

static void
moonphase_home_plugin_init(MoonphaseHomePlugin* desktop_plugin)
{
  desktop_plugin->priv = MOONPHASE_HOME_PLUGIN_GET_PRIVATE(desktop_plugin);
  GtkWidget* contents = build_ui();
  desktop_plugin->priv->iD = NULL;
  desktop_plugin->priv->phases_visible = FALSE;
  desktop_plugin->priv->foreground_moon = 
    gdk_pixbuf_new_from_file("/usr/share/pixmaps/moon-phase-widget/fullmoon.png", NULL);
  desktop_plugin->priv->background_moon = 
    gdk_pixbuf_new_from_file("/usr/share/pixmaps/moon-phase-widget/newmoon.png", NULL);
  desktop_plugin->priv->size = 256;
  hd_home_plugin_item_set_settings(HD_HOME_PLUGIN_ITEM(desktop_plugin), TRUE);

  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(_show_settings_dialog), NULL);
  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(moonphase_update), NULL);
  g_signal_connect(contents, "button-release-event", G_CALLBACK(_show_hide_moonphases), desktop_plugin);
  _calculate_phases_date(desktop_plugin);

  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
}

static void
moonphase_home_plugin_class_init(MoonphaseHomePluginClass* klass)
{
  GtkWidgetClass* widget_class = GTK_WIDGET_CLASS(klass);
  g_type_class_add_private(klass, sizeof(MoonphaseHomePluginPrivate));
  widget_class->realize = moonphase_realize;
  widget_class->expose_event = moonphase_expose;
  G_OBJECT_CLASS(klass)->finalize = moonphase_finalize;
}

static void
moonphase_home_plugin_class_finalize(MoonphaseHomePluginClass* klass)
{
}
