/*
 *  Medard for Maemo.
 *  Copyright (C) 2011 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <QTimer>
#include <QImage>
#include <QDir>
#include <QFile>

#include "medarddownloader.h"

#define DOWNLOAD_CACHE_DIR ".cache/medard-downloader"

#define MEDARD_URL "http://www.medard-online.cz"
#define MEDARD_IMAGE_URL "http://www.medard-online.cz/scripts/getimage.php?initDate=%1&domain=%2&variable=%3&offset=%4"

#define FC_SEA_LEVEL_PRESSURE "slp"
#define FC_PRECIPITATION "precip"
#define FC_WIND_VELOCITY "wv"
#define FC_CLOUDINESS "cloud"
#define FC_TEMPERATURE "temp"

#define FD_EUROPE "1"
#define FD_CZECH_REPUBLIC "2"

#define MIN_OFFSET 0
#define MAX_OFFSET 72

#define IMAGE_WIDTH 556
#define IMAGE_HEIGHT 408

#define MAX_DOWNLOAD_RETRIES 3
#define RETRY_TIMEOUT 5000

MedardDownloader::MedardDownloader() : QObject()
{
    m_forecastType = FC_SEA_LEVEL_PRESSURE;
    m_forecastDomain = FD_EUROPE;
    m_forecastInitialDateCode.clear();
    m_forecastDateOffset = 0;

    m_network = new QNetworkAccessManager();
    m_reply = 0;

    m_retryCounter = 0;
    m_retryTimer = new QTimer();
    connect(m_retryTimer, SIGNAL(timeout()), this, SLOT(retryTimerEvent()));

    m_cachePath = QString("%1/%2")
                  .arg(QDir().homePath())
                  .arg(DOWNLOAD_CACHE_DIR);

    createCacheDirectory();
}

MedardDownloader::~MedardDownloader()
{
    if (m_retryTimer->isActive())
        m_retryTimer->stop();

    if (m_reply) {
        m_reply->abort();
        delete m_reply;
    }
}

QSize MedardDownloader::imageSize()
{
    return QSize(IMAGE_WIDTH, IMAGE_HEIGHT);
}

void MedardDownloader::setForecastType(ForecastType type)
{
    switch (type) {
        case SeaLevelPressure:
            m_forecastType = FC_SEA_LEVEL_PRESSURE;
            break;

        case Precipitation:
            m_forecastType = FC_PRECIPITATION;
            break;

        case WindVelocity:
            m_forecastType = FC_WIND_VELOCITY;
            break;

        case Cloudiness:
            m_forecastType = FC_CLOUDINESS;
            break;

        case Temperature:
            m_forecastType = FC_TEMPERATURE;
            break;
    }
}

void MedardDownloader::setForecastDomain(ForecastDomain domain)
{
    switch (domain) {
        case Europe:
            m_forecastDomain = FD_EUROPE;
            break;

        case CzechRepublic:
            m_forecastDomain = FD_CZECH_REPUBLIC;
            break;
    }
}

void MedardDownloader::setForecastInitialDate(QDateTime date)
{
    QString offset;

    m_forecastInitialDate = date.toUTC();

    if (date.toUTC().time().hour() >= 18) {
        m_forecastInitialDate.setTime(QTime(18, 0, 0));
        offset = "12";
    } else if (date.toUTC().time().hour() >= 12) {
        m_forecastInitialDate.setTime(QTime(12, 0, 0));
        offset = "06";
    } else if (date.toUTC().time().hour() >= 6) {
        m_forecastInitialDate.setTime(QTime(6, 0, 0));
        offset = "00";
    } else {
        m_forecastInitialDate.setTime(QTime(0, 0, 0));
        offset = "18";
    }

    if (offset == "18") {
        /* use previous day */
        m_forecastInitialDateCode = QString("%1_%2")
                                    .arg(date.addDays(-1).toUTC().toString("yyMMdd"))
                                    .arg(offset);
    } else {
        /* use current day */
        m_forecastInitialDateCode = QString("%1_%2")
                                    .arg(date.toUTC().toString("yyMMdd"))
                                    .arg(offset);
    }

    cleanCacheDirectory();
}

QDateTime MedardDownloader::forecastInitialDate()
{
    return m_forecastInitialDate.toLocalTime();
}

QDateTime MedardDownloader::forecastDate()
{
    return m_forecastInitialDate.addSecs(3600 * m_forecastDateOffset).toLocalTime();
}

void MedardDownloader::setForecastDateOffset(int offset)
{
    m_forecastDateOffset = offset;
    if (m_forecastDateOffset > MAX_OFFSET)
        m_forecastDateOffset = MAX_OFFSET;

    if (m_forecastDateOffset < MIN_OFFSET)
        m_forecastDateOffset = MIN_OFFSET;
}

void MedardDownloader::retryTimerEvent()
{
    if (m_retryTimer->isActive())
        m_retryTimer->stop();

    downloadImage();
}

void MedardDownloader::tryDownloadImageAgain()
{
    m_retryCounter++;

    if (m_retryCounter < MAX_DOWNLOAD_RETRIES) {
        m_retryTimer->setInterval(RETRY_TIMEOUT * (m_retryCounter + 1));
        m_retryTimer->start();
    } else {
        m_retryCounter = 0;
        emit downloadFailed();
    }
}

void MedardDownloader::clearDownloadRequest()
{
    delete m_reply;
    m_reply = 0;
}

void MedardDownloader::downloadImageFinished()
{ 
    QByteArray picture = m_reply->readAll();

    if (picture.isNull() || picture.size() <= 0)
        return;

    m_retryCounter = 0;

    QImage image;
    if (!image.loadFromData(picture, "png"))
        return;

    QString filename = QString("%1/%2_%3_%4_%5.png")
                       .arg(m_cachePath)
                       .arg(m_forecastType)
                       .arg(m_forecastDomain)
                       .arg(m_forecastInitialDateCode)
                       .arg(QString().number(m_forecastDateOffset));

    if ((image.width() == 512) && (image.height() == 512)) {
        QImage croped(512, 400, QImage::Format_ARGB32_Premultiplied);
        croped = image.copy(0, 52, 512, IMAGE_HEIGHT);
        croped.save(filename, "png");
    } else {
        QImage croped(560, 400, QImage::Format_ARGB32_Premultiplied);
        croped = image.copy(10, 96, IMAGE_WIDTH, IMAGE_HEIGHT);
        croped.save(filename, "png");
    }

    emit downloadFinished(filename, forecastDate());

    QTimer::singleShot(0, this, SLOT(clearDownloadRequest()));
}

void MedardDownloader::downloadImageError(QNetworkReply::NetworkError /* code */)
{
    tryDownloadImageAgain();
    QTimer::singleShot(0, this, SLOT(clearDownloadRequest()));
}

void MedardDownloader::downloadImage()
{
    if (m_forecastInitialDateCode.isNull()) {
        retrieveForecastInitialDate();
        tryDownloadImageAgain();
        return;
    }

    QString filename = QString("%1/%2_%3_%4_%5.png")
                       .arg(m_cachePath)
                       .arg(m_forecastType)
                       .arg(m_forecastDomain)
                       .arg(m_forecastInitialDateCode)
                       .arg(QString().number(m_forecastDateOffset));

    if (isDownloaded(filename)) {
        emit downloadFinished(filename, forecastDate());
        return;
    }

    QString imageUrl = QString(MEDARD_IMAGE_URL)
                               .arg(m_forecastInitialDateCode)
                               .arg(m_forecastDomain)
                               .arg(m_forecastType)
                               .arg(QString().number(m_forecastDateOffset));

    QUrl url(imageUrl);
    QNetworkRequest request(url);

    if (m_reply)
        clearDownloadRequest();
    m_reply = m_network->get(request);

    connect(m_reply, SIGNAL(finished()), this, SLOT(downloadImageFinished()));
    connect(m_reply, SIGNAL(error(QNetworkReply::NetworkError)), this,
            SLOT(downloadImageError(QNetworkReply::NetworkError)));
}

bool MedardDownloader::isDownloaded(const QString filename)
{
    return QFile(filename).exists();
}

int MedardDownloader::forecastDateOffset()
{
    return m_forecastDateOffset;
}

int MedardDownloader::minForecastDateOffset()
{
    return MIN_OFFSET;
}

int MedardDownloader::maxForecastDateOffset()
{
    return MAX_OFFSET;
}

void MedardDownloader::retrieveForecastInitialDateFinished()
{
    QByteArray data = m_reply->readAll();

    int index1 = data.indexOf("var fcst_initDatestamp=\"", 0);
    int index2 = data.indexOf("\";", index1);
    if (index1 != -1) {
        QString temp;
        for (int i = index1 + 24; i < index2; i++) {
            temp.append(data.at(i));
        }
        QDateTime date = QDateTime::fromTime_t(temp.toULong() + 6 * 3600);
        if (!date.isNull()) {
            setForecastInitialDate(date.toLocalTime());

            int forecastDateOffset = date.toLocalTime().secsTo(QDateTime().currentDateTime()) / 3600;
            setForecastDateOffset(forecastDateOffset);
        }
        m_retryCounter = 0;
    }

    QTimer::singleShot(0, this, SLOT(clearDownloadRequest()));
}

void MedardDownloader::retrieveForecastInitialDateError(QNetworkReply::NetworkError /* code */)
{
}

void MedardDownloader::retrieveForecastInitialDate()
{
    QString serverUrl = QString(MEDARD_URL);

    QUrl url(serverUrl);
    QNetworkRequest request(url);

    if (m_reply)
        clearDownloadRequest();
    m_reply = m_network->get(request);

    connect(m_reply, SIGNAL(finished()), this, SLOT(retrieveForecastInitialDateFinished()));
    connect(m_reply, SIGNAL(error(QNetworkReply::NetworkError)), this,
            SLOT(retrieveForecastInitialDateError(QNetworkReply::NetworkError)));
}

void MedardDownloader::createCacheDirectory()
{
    QDir cacheDir(m_cachePath);
    if (!cacheDir.exists())
        cacheDir.mkpath(cacheDir.path());
}

void MedardDownloader::cleanCacheDirectory()
{
    QDir cacheDir(m_cachePath);
    QStringList list = cacheDir.entryList();
    for (int i = 0; i < list.size(); i++) {
        if (!list.at(i).contains(m_forecastInitialDateCode)) {
            QFile(m_cachePath + "/" + list.at(i)).remove();
        }
    }
}
