#include "startwindow.h"
#include "mainwindow.h"

#include <QScrollArea>
#include <QFileDialog>
#include <QMenuBar>
#include <QGLWidget>

#include <QMaemo5InformationBox>

#include "shared.h"


StartWindow::StartWindow(QWidget *parent) : QMainWindow(parent), 
        readingUI(0), 
        searchCatalogsView(0),
        sorting(None)
{
    setWindowTitle("MeBook");
    QPalette palette;
    QPixmap bg(":/startwindow_bg.png");
    QBrush brush(bg);
    palette.setBrush(QPalette::Window, brush);
    setPalette(palette);
    
    scrollArea = new QScrollArea(this);
//    palette.setBrush(scrollArea->backgroundRole(), QBrush(Qt::transparent));
    scrollArea->setPalette(palette);
    scrollArea->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(scrollArea, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(openContextMenu(const QPoint&)));
    setCentralWidget(scrollArea);
//    scrollArea->setViewport(new QGLWidget(QGLFormat(QGL::DoubleBuffer | QGL::SampleBuffers | QGL::HasOverlay | QGL::StencilBuffer | QGL::AccumBuffer | QGL::DirectRendering)));
    //Scroller is used to detect if user is tapping an item or scrolling the QAbstractKineticScroller.
    //TODO: There might be a better way to do this..
    scroller = scrollArea->property("kineticScroller").value<QAbstractKineticScroller*>();
    
    setAttribute(Qt::WA_Maemo5StackedWindow);

    bookList = new QWidget(this);
    bookList->setContextMenuPolicy(Qt::NoContextMenu);
    scrollArea->setWidget(bookList);
    bookList->setContentsMargins(QMargins(10, 10, 10, 10));
    scrollArea->setWidgetResizable(true);

    listLayout = new QVBoxLayout(bookList);

    QString buttonStyleSheet =  
        "QPushButton{ \
        height: 70px; \
        border-image: url(:/button_borderimage.png) 3 10 3 10; \
        border-image-repeat: round; \
        border-top: 3px transparent; \
        border-bottom: 3px transparent; \
        border-left: 10px transparent; \
        border-right: 10px transparent; \
        } \
        QPushButton:pressed{ \
        border-image: url(:/button_borderimage_pressed.png) 3 10 3 10; \
        border-top: 3px transparent; \
        border-bottom: 3px transparent; \
        border-left: 10px transparent; \
        border-right: 10px transparent; \
        }";


    QPushButton *openFileButton = new QPushButton(tr("Open file"), this);
    openFileButton->setStyleSheet(buttonStyleSheet);
    listLayout->addWidget(openFileButton);
    listLayout->setAlignment(openFileButton, Qt::AlignTop);
    listLayout->setAlignment(Qt::AlignTop);
    connect(openFileButton, SIGNAL(pressed()), this, SLOT(openFile()));

    QPushButton *searchCatalogsButton = new QPushButton(tr("Search catalogs"), this);
    searchCatalogsButton->setStyleSheet(buttonStyleSheet);
    listLayout->addWidget(searchCatalogsButton);
    listLayout->setAlignment(searchCatalogsButton, Qt::AlignTop);
    connect(searchCatalogsButton, SIGNAL(clicked()), this, SLOT(searchCatalogs()));

    QMenuBar *menuBar = new QMenuBar(this);
    QAction *settings = menuBar->addAction(tr("Settings"));
    connect(settings, SIGNAL(triggered()), this, SLOT(openSettingsDialog()));
    QAction *about = menuBar->addAction(tr("About"));
    connect(about, SIGNAL(triggered()), this, SLOT(openAboutDialog()));

    QActionGroup *sortGroup = new QActionGroup(menuBar);
    sortByAuthorAction = sortGroup->addAction(tr("Author"));
    sortByAuthorAction->setCheckable(true);
    menuBar->addAction(sortByAuthorAction);
    connect(sortByAuthorAction, SIGNAL(triggered()), this, SLOT(sortByAuthor()));

    sortByTitleAction = sortGroup->addAction(tr("Title"));
    sortByTitleAction->setCheckable(true);
    menuBar->addAction(sortByTitleAction);
    connect(sortByTitleAction, SIGNAL(triggered()), this, SLOT(sortByTitle()));

    sortByDateAddedAction = sortGroup->addAction(tr("Date"));
    sortByDateAddedAction->setCheckable(true);
    menuBar->addAction(sortByDateAddedAction);
    connect(sortByDateAddedAction, SIGNAL(triggered()), this, SLOT(sortByDateAdded()));

    setMenuBar(menuBar);

    
    bookListLayout = new QVBoxLayout();
    bookListLayout->setSpacing(14);
    listLayout->addLayout(bookListLayout);

    library = new Library("/home/user/MyDocs/books", "/home/user/MyDocs/books/bookdb.db", this);

    connect(library, SIGNAL(bookAdded(Book*)), this, SLOT(addBook(Book*)));
    connect(library, SIGNAL(bookRemoved(Book*)), this, SLOT(removeBook(Book*)));
    library->loadLibrary();

    feedBooksCatalog = new FeedBooks(this);
    connect(feedBooksCatalog, SIGNAL(bookSaved(Book*)), library, SLOT(addToLibrary(Book*)));
    connect(feedBooksCatalog, SIGNAL(message(const QString)), this, SLOT(showMessage(const QString)));
    connect(feedBooksCatalog, SIGNAL(openBook(Book*)), this, SLOT(openBook(Book*)));

    osso_context = osso_initialize(OSSO_NAME.toUtf8(), "0.0.1", TRUE, NULL);

    readSettings();
}

StartWindow::~StartWindow()
{
    if(osso_context)
        osso_deinitialize(osso_context);
}


void StartWindow::readSettings()
{

    settings = new QSettings("durriken.org", "MeBook");

    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation", static_cast<int>(Qt::WA_Maemo5PortraitOrientation)).toInt()), true);

    sorting = static_cast<Sorting>(settings->value("sorting", static_cast<int>(None)).toInt());

    if(sorting == byAuthor)
        sortByAuthorAction->setChecked(true);
    else if(sorting == byTitle)
        sortByTitleAction->setChecked(true);
    else
        sortByDateAddedAction->setChecked(true);

}

void StartWindow::sortBooks(Sorting s)
{
    sorting = s;
    titles.clear();
    authors.clear();
    settings->setValue("sorting", static_cast<int>(sorting));
    QList<Book*> books = library->getBooks();
    //Remove BookListItems from bookListLayout
    bookList->hide();
    for(QList<BookListItem*>::iterator iter = bookListItems.begin(); iter != bookListItems.end(); ++iter){
        bookListLayout->removeWidget(*iter);
        bookListItems.removeOne(*iter);
        delete (*iter);
    }
    bookList->adjustSize();

    for(QList<Book*>::iterator iter = books.begin(); iter != books.end(); ++iter){
        Book *book = *iter;
        addBook(book);
    }

    bookList->show();
}

void StartWindow::openFile()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Open book"), "/home/user/MyDocs", tr("eBook Files (*.epub)"));
    
    if(!filename.isEmpty()){
        Book *book = new Book(filename);
        library->addToLibrary(book);
        openBook(book);
    }
}

void StartWindow::openBook(Book *book)
{
    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);

    readingUI = new MainWindow(osso_context, this);
    readingUI->setFont(settings->value("font-family", "Sans Serif").toString(), settings->value("font-size", 24).toInt());
    readingUI->setColorMode(!settings->value("colors", true).toBool());
    readingUI->enableBacklightForcing(settings->value("backlight", true).toBool());
    readingUI->setViewMode(static_cast<BookViewMode>(settings->value("viewmode", 1).toInt()), static_cast<Qt::WidgetAttribute>(settings->value("orientation", static_cast<int>(Qt::WA_Maemo5PortraitOrientation)).toInt()));

    readingUI->openBook(book);

    //Hide other windows 
    if(searchCatalogsView){
        if(searchCatalogsView->isVisible())
            searchCatalogsView->hide();
    }

}

void StartWindow::openSettingsDialog()
{
    SettingsDialog *settingsDialog = new SettingsDialog(this);    
    settingsDialog->setOrientation(settings->value("orientation", static_cast<int>(Qt::WA_Maemo5PortraitOrientation)).toInt());
    settingsDialog->setColors(settings->value("colors", true).toInt());
    settingsDialog->setFontFamily(settings->value("font-family", "Sans Serif").toString());
    settingsDialog->setFontSize(settings->value("font-size", 24).toInt());
    settingsDialog->setBacklightForce(settings->value("backlight", true).toBool());
    settingsDialog->setViewMode(static_cast<BookViewMode>(settings->value("viewmode", 1).toInt()));


    if(settingsDialog->exec() == QDialog::Accepted){
        int orientation = settingsDialog->getOrientation();
        if(orientation != settings->value("orientation")){
            settings->setValue("orientation", orientation); 
            setAttribute(static_cast<Qt::WidgetAttribute>(orientation), true);
        }

        int colors = settingsDialog->getColors();
        if(colors != settings->value("colors")){
            settings->setValue("colors", colors);
        }

        settings->setValue("font-family", settingsDialog->getFontFamily());
        settings->setValue("font-size", settingsDialog->getFontSize());
        settings->setValue("backlight", settingsDialog->getBacklighForce());
        settings->setValue("viewmode", settingsDialog->getViewMode());

        settings->sync();
    }
}

void StartWindow::openAboutDialog()
{
    QDialog dialog(this);
    dialog.setWindowTitle(tr("About MeBook"));
    QHBoxLayout horizontalLayout;

    QLabel icon;
    QPixmap pixmap = QIcon::fromTheme("mebook").pixmap(48, 48);
    icon.setPixmap(pixmap.scaled(QSize(72, 72)));
    horizontalLayout.addWidget(&icon);
    horizontalLayout.setAlignment(&icon, Qt::AlignTop);

    QVBoxLayout verticalLayout;
    horizontalLayout.addLayout(&verticalLayout);
    horizontalLayout.setAlignment(&verticalLayout, Qt::AlignLeft);
    horizontalLayout.setStretch(1, 2);

    QLabel name("MeBook " + QString(APP_VERSION));
    name.setFont(QFont("Sans", 22));
    verticalLayout.addWidget(&name);
    verticalLayout.setAlignment(&name, Qt::AlignLeft | Qt::AlignTop);

    QLabel copyright("<qt>&copy; 2010 by Juhana Jauhiainen</qt>");
    QFont copyrightFont("Sans", 12);
    QPalette palette = QApplication::palette();
    palette.setColor(QPalette::Normal, QPalette::WindowText, Qt::darkGray);
    copyright.setPalette(palette);
    copyright.setFont(copyrightFont);
    verticalLayout.addWidget(&copyright);
    verticalLayout.setStretch(2, 3);
    verticalLayout.setAlignment(&copyright, Qt::AlignLeft | Qt::AlignBottom);

    dialog.setLayout(&horizontalLayout);

    QPushButton reportBug(tr("Report bugs"));
    connect(&reportBug, SIGNAL(clicked()), this, SLOT(reportBugs()));
    horizontalLayout.addWidget(&reportBug);
    dialog.exec();
}

void StartWindow::searchCatalogs()
{
    setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    if(!searchCatalogsView){
        searchCatalogsView = new SearchCatalogsView(feedBooksCatalog, this);
        connect(searchCatalogsView, SIGNAL(closing()), this, SLOT(searchCatalogsClosed()));
    }
    searchCatalogsView->show();
}

void StartWindow::searchCatalogsClosed()
{
    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);
}

void StartWindow::bookViewClosed()
{
}

void StartWindow::showMessage(const QString message)
{
    QMaemo5InformationBox::information(this, message, QMaemo5InformationBox::DefaultTimeout); 
}

void StartWindow::addBook(Book *book)
{

    BookListItem *item;
    item = new BookListItem(book, scroller, scrollArea);
    QString author = book->getAuthor();
    if(author.contains(" ") && !author.contains(",")){
        QStringList temp = author.split(" ");
        author = temp.last();
    }
    authors.push_back(author);
    
    titles.push_back(book->getTitle());

    if(sorting == byAuthor){
        authors.sort();
        int index = authors.indexOf(author);
        bookListItems.insert(index, item);
        bookListLayout->insertWidget(index, item);
    } else if(sorting == byTitle){
        titles.sort();
        int index = titles.indexOf(book->getTitle());
        bookListItems.insert(index, item);
        bookListLayout->insertWidget(index, item);
    } else{
        bookListItems.push_back(item);
        bookListLayout->addWidget(item);
    }
    connect(item, SIGNAL(selected(Book*)), this, SLOT(openBook(Book*)));
    bookListLayout->setAlignment(item, Qt::AlignTop);
    //Call adjust size so widget's size is recalculated.
    bookList->adjustSize();

}

void StartWindow::removeBook(Book *book)
{
    if(!book->getFilename().isEmpty()){
        for(QList<BookListItem*>::iterator iter = bookListItems.begin(); iter != bookListItems.end(); ++iter){
            BookListItem *item = *iter;
            if(item->book()->getFilename() == book->getFilename()){
                bookListItems.removeOne(item);
                bookListLayout->removeWidget(item);
                delete item;
                bookList->adjustSize();
                return;
            
            }
        }
    }
}

void StartWindow::reportBugs()
{
    QDesktopServices::openUrl(QUrl("http://code.google.com/p/mebook/issues/"));
}

void StartWindow::openContextMenu(const QPoint &point)
{
    QPoint globalPoint = scrollArea->mapToGlobal(point);
    QMenu *menu = new QMenu();
    QAction *deleteAction = menu->addAction(tr("Delete"));

    for(QList<BookListItem*>::iterator iter = bookListItems.begin(); iter != bookListItems.end(); ++iter){
        BookListItem *item = *iter;
        if(item->rect().contains(item->mapFromGlobal(globalPoint))){
            if(menu->exec(globalPoint) == deleteAction){
                item->setSelected(false);
                QDialog dialog(this);
                QHBoxLayout dialogLayout;
                QDialogButtonBox buttonBox(QDialogButtonBox::Yes);
                connect(&buttonBox, SIGNAL(accepted()), &dialog, SLOT(accept()));
                QLabel dialogText(tr("Do you really want to delete this book? <br> The file will be removed from your mobile device."));
                dialogText.setWordWrap(true);
                dialogLayout.addWidget(&dialogText);
                dialogLayout.addWidget(&buttonBox);
                dialog.setLayout(&dialogLayout);
                dialog.setWindowTitle(tr("Confirm deletion"));
                if(dialog.exec() == QDialog::Accepted)
                    library->removeFromLibrary(item->book());
                break;
            }
        }
    }


}

void StartWindow::sortByAuthor()
{
    sortBooks(byAuthor);
}

void StartWindow::sortByTitle()
{
    sortBooks(byTitle);
}

void StartWindow::sortByDateAdded()
{
    sortBooks(None);
}


void StartWindow::search(const QString &searchString)
{
    
}
