#ifndef SETTINGSDIALOG_H
#define SETTINGSDIALOG_H

#include <QDialog>
#include <QRadioButton>
#include <QCheckBox>
#include <QtMaemo5>

#include "shared.h"

struct CustomListItem{
    QString text;
    QString fontFamily;
    int fontSize;
    int row;
};

class CustomListItemDelegate : public QStyledItemDelegate{
    Q_OBJECT
public:
    CustomListItemDelegate(QWidget *parent = 0) : QStyledItemDelegate(parent){
    }

    void paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const{
        QStyleOptionViewItem op;
        initStyleOption(&op, index);
        QFont font(index.model()->data(index, Qt::UserRole).toString(), index.model()->data(index, Qt::UserRole + 1).toInt());
        painter->setFont(font);
        painter->save();
        QApplication::style()->drawPrimitive(QStyle::PE_PanelItemViewItem, &option, painter); 
        painter->drawText(option.rect, Qt::AlignCenter, index.model()->data(index, Qt::DisplayRole).toString());
        painter->restore();
    }
    
    QSize sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const{
        QStyleOptionViewItem op;
        initStyleOption(&op, index);
        op.font = QFont(index.model()->data(index, Qt::UserRole).toString(), index.model()->data(index, Qt::UserRole + 1).toInt());
        QFontMetrics metrics(op.font);
        int height = metrics.height();
        if(height < 70)
            height = 70;
        return QSize(0, height);
    }
};

class CustomListModel : public QAbstractListModel{
    Q_OBJECT
public:
    CustomListModel(QList<CustomListItem> items, QObject *parent = 0) : QAbstractListModel(parent), listItems(items) {}

    int rowCount(const QModelIndex &parent = QModelIndex()) const{
        Q_UNUSED(parent);

        return listItems.count();
    }

    QVariant data(const QModelIndex &index, int role) const{
        if(!index.isValid())
            return QVariant();

        if(index.row() >= listItems.count())
            return QVariant();

        //DisplayRole returns the text.
        if(role == Qt::DisplayRole)
            return listItems[index.row()].text;
 
        //UserRole returns the font.
        if(role == Qt::UserRole)
            return listItems[index.row()].fontFamily;

        if(role == Qt::UserRole + 1)
            return listItems[index.row()].fontSize;

        return QVariant();
    }

    CustomListItem *item(int row)
    {
        if(row >= 0 && row < listItems.count())
            return &listItems[row];

        return 0;
    }

    QList<CustomListItem*> findItems(const QString &s)
    {
        QList<CustomListItem*> ritems;

        for(QList<CustomListItem>::iterator iter = listItems.begin(); iter != listItems.end(); ++iter){
            if((*iter).text.contains(s))
                ritems.push_back(&(*iter));
        }

        return ritems;
    }

private:
    QList<CustomListItem> listItems;
};

class SettingsDialog : public QDialog
{
    Q_OBJECT
public:
    SettingsDialog(QWidget *parent = 0);

    int getOrientation();
    void setOrientation(int);

    int getColors();
    void setColors(int);

    void setFontFamily(const QString &family);
    QString getFontFamily();

    void setFontSize(int size);
    int getFontSize();

    void setBacklightForce(bool);
    bool getBacklighForce();

    void setViewMode(BookViewMode);
    BookViewMode getViewMode();

protected:
    void closeEvent(QCloseEvent*);
    

private:
    QButtonGroup *orientationButtonGroup;
    QRadioButton *portraitRadioButton;
    QRadioButton *landscapeRadioButton;
    QButtonGroup *colorsButtonGroup;
    QRadioButton *normalRadioButton;
    QRadioButton *invertedRadioButton;
    QButtonGroup *viewModeButtonGroup;
    QRadioButton *scrollRadioButton;
    QRadioButton *pagesRadioButton;
    QMaemo5ListPickSelector *fontSelector;
    CustomListModel *model;
    QHash<QString, CustomListModel*> fontSizes;
    QMaemo5ListPickSelector *fontSizeSelector;
    QCheckBox *backlightButton;
    QWidget *containerWidget;

public slots:
    void fontChanged(const QString&);
};

#endif
