#include "searchcatalogsview.h"

#include <QDialog>
#include <QDialogButtonBox>
#include <QCheckBox>
#include <QCloseEvent>
#include <QMaemo5InformationBox>

SearchCatalogsView::SearchCatalogsView(Catalog* c, QWidget *parent) : QWidget(parent), catalog(c)
{ 
    QVBoxLayout *mainLayout = new QVBoxLayout(this);
    scrollArea = new QScrollArea(this);
    mainLayout->addWidget(scrollArea);

    listContainer = new QWidget(this);
    scrollArea->setWidget(listContainer);
    scrollArea->setWidgetResizable(true);

    scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    QVBoxLayout *listButtonLayout = new QVBoxLayout(listContainer);

    listLayout = new QVBoxLayout();

    searchEdit = new QLineEdit();
    listButtonLayout->addWidget(searchEdit);
    listButtonLayout->setAlignment(searchEdit, Qt::AlignTop);

    QPushButton *searchButton = new QPushButton(tr("Search"));
    listButtonLayout->addWidget(searchButton);
    listButtonLayout->setAlignment(searchButton, Qt::AlignTop);

    listLayout->addStretch(1);
    listLayout->setAlignment(Qt::AlignTop);
    listButtonLayout->addLayout(listLayout);

    connect(searchButton, SIGNAL(clicked()), this, SLOT(searchButtonPressed())); 
    connect(catalog, SIGNAL(addBook(Book*)), this, SLOT(addBook(Book*)));
    connect(catalog, SIGNAL(foundBooksCount(int)), this, SLOT(bookCount(int)));

    setWindowFlags(windowFlags() | Qt::Window);
    setAttribute(Qt::WA_Maemo5StackedWindow);

    setWindowTitle(tr("Search feedbooks.com"));

}

void SearchCatalogsView::searchButtonPressed()
{
    //Remove all items from book list
    QLayoutItem *child;
    while((child = listLayout->takeAt(0)) != 0){
        QWidget *widget = child->widget();
        if(widget)
            delete widget;
        delete child;
    }
    listContainer->adjustSize();
    listLayout->addStretch(1);

    QString searchPhrase = searchEdit->text();
    catalog->searchBooks(searchPhrase);

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);

}

void SearchCatalogsView::addBook(Book *book)
{
    BookListItem *item;
    //If scroller is not null pass it to BookListItem. It's used to detect if user is tapping an item or scrolling the QAbstractKineticScroller.
    QAbstractKineticScroller *scroller = scrollArea->property("kineticScroller").value<QAbstractKineticScroller*>();
    if(scroller)
        item = new BookListItem(book, scroller);
    else
        item = new BookListItem(book, 0, this);
    listLayout->insertWidget(listLayout->count()-1, item);
    listLayout->setAlignment(item, Qt::AlignTop);
    //Call adjust size so widget's size is recalculated.
    listContainer->adjustSize();

    connect(item, SIGNAL(selected(Book*)), this, SLOT(openBookInfo(Book*)));
}

void SearchCatalogsView::openBookInfo(Book *book)
{
    QDialog dialog(this);

    QHBoxLayout dialogLayout(&dialog);
    QScrollArea dialogScrollArea;
    QWidget bookInfoContainer;
    QVBoxLayout layout(&bookInfoContainer);

    QPixmap cover = book->getCoverImage();
    QLabel coverLabel;
    if(!cover.isNull()){
        coverLabel.setPixmap(cover);
        coverLabel.setFixedWidth(cover.width());
        layout.addWidget(&coverLabel);
        layout.setAlignment(&coverLabel, Qt::AlignCenter);
    }

    QLabel nameLabel(book->getTitle());
    QFont font = nameLabel.font();
    font.setBold(true);
    nameLabel.setFont(font);
    layout.addWidget(&nameLabel);
    layout.setAlignment(&nameLabel, Qt::AlignCenter);

    QLabel authorLabel(book->getAuthor());
    authorLabel.setFont(font);
    layout.addWidget(&authorLabel);
    layout.setAlignment(&authorLabel, Qt::AlignCenter);

    QLabel summaryLabel(book->getDescription());
    summaryLabel.setWordWrap(true);
    summaryLabel.setAlignment(Qt::AlignJustify);
    layout.addWidget(&summaryLabel);

    unsigned int words = book->getLengthInWords();
    QLabel length;
    if(words > 0){
        length.setText(QString::number(words) + tr(" words"));
        length.setFont(font);
        layout.addWidget(&length);
    }

    QLabel publishedLabel;
    QString published = book->getPublishDate();
    if(!published.isEmpty()){
        publishedLabel.setText(tr("Published: ") + published);
        publishedLabel.setFont(font);
        layout.addWidget(&publishedLabel);
    }

    QDialogButtonBox buttons(&dialog);
    QPushButton downloadButton(tr("Download"));
    buttons.addButton(&downloadButton, QDialogButtonBox::AcceptRole);
    connect(&downloadButton, SIGNAL(clicked()), &dialog, SLOT(accept()));

    QCheckBox openWhenLoaded(tr("Open book after download"));
    openWhenLoaded.setChecked(false);
    layout.addWidget(&openWhenLoaded);
    
    dialogScrollArea.setWidget(&bookInfoContainer);
    dialogScrollArea.setWidgetResizable(true);

    dialogScrollArea.setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    dialogLayout.addWidget(&dialogScrollArea);
    dialogLayout.setStretch(0, 1);
    dialogLayout.addWidget(&buttons);
    dialogLayout.setAlignment(&buttons, Qt::AlignBottom);

    dialog.setWindowTitle(tr("Book info"));

    if(dialog.exec() == QDialog::Accepted){
        QMaemo5InformationBox::information(this, tr("Downloading ") + book->getTitle(), QMaemo5InformationBox::DefaultTimeout);
        catalog->download(book, openWhenLoaded.checkState());
    }

}

void SearchCatalogsView::bookCount(int number)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    if(number == 0){
        QLabel *notFoundLabel = new QLabel(tr("No books found."));
        listLayout->insertWidget(listLayout->count()-1, notFoundLabel);
        listLayout->setAlignment(notFoundLabel, Qt::AlignCenter);
        listContainer->adjustSize();
    } else if(number > 20){
        //TODO: Add "More"-button for loading remaining pages of books
    }
}

void SearchCatalogsView::closeEvent(QCloseEvent *event)
{
    emit closing();
    event->accept();
}
