#include "bookview.h"

#include <QDebug>
#include <QAction>
#include <QWebElement>
#include <QWebFrame>
#include <QMouseEvent>
#include <QWebSettings>
#include <QApplication>
#include <QDesktopWidget>
#include <QAbstractKineticScroller>
#include <QFontMetrics>
#include <QFontDatabase>
#include <math.h>
#include <QTime>
#include <QMenu>
#include <QMessageBox>
#include <QGraphicsProxyWidget>
#include <QGraphicsLinearLayout>
#include <QMaemo5InformationBox>
#include <QDBusConnection>
#include <QDBusMessage>

#include "addbookmarkdialog.h"
#include "addannotationdialog.h"
#include "listviewdialog.h"

#include "libosso.h"

//Q_DECLARE_METATYPE(RenderTile)
const int topMargin = 90;

BookView::BookView(QWidget *parent) : QGraphicsView(parent), 
        autoScrollStep(1), 
        autoScrollBy(0), 
        isTap(false), 
        tapMove(0)
{
    setMouseTracking(false); 

    currentPosition.section.number = 0;
    currentPosition.page = 0;
    currentPosition.multiplier = 0.00;

    scene = new QGraphicsScene(this);
    setScene(scene);

    pageWidgetCurrent = NULL;
    pageWidgetNext = NULL;
    scrollablePage = NULL;




    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    
    renderer = new Renderer(this);

    //int type = qRegisterMetaType<RenderTile>("RenderTile");

    connect(renderer, SIGNAL(tileRendered(RenderTile*)), this, SLOT(addTile(RenderTile*)));
    connect(this, SIGNAL(requestRender(QString, int)), renderer, SLOT(addToQueue(QString, int)));

    renderer->setRenderSettings(&renderSettings);

  
    //Set PageMode as default.
    setViewMode(PageMode, Qt::WA_Maemo5PortraitOrientation);

    //Autoscroll every 200ms
    autoScrollTimer.setInterval(200);
    connect(&autoScrollTimer, SIGNAL(timeout()), this, SLOT(handleAutoScroll()));

    setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(openContextMenu(const QPoint&)));
    
}

BookView::~BookView()
{
    deleteTiles();
}

void BookView::setBook(Book *openBook)
{
    if(openBook){
        book = openBook;
        

        if(!book->isLoaded())
            book->load();

        renderer->setAnnotations(book->getAnnotations());
        renderer->setResourceManager(book->getResourceManager());

        progressSlider->setRange(0, book->getBookLength());

        if(book->getProgress().isValid()){
            currentPosition.multiplier = book->getProgress().getPercentage();
            currentPosition.section.number = book->getProgress().getSection();
        }else{ 
            currentPosition.multiplier = 0;
            currentPosition.section.number = 0;
        }  

        goToSection(currentPosition.section.number, currentPosition.multiplier);

    }
}

void BookView::deleteTiles()
{
    for(QHash< int, QHash<int, RenderTile*> >::iterator iter = tiles.begin(); iter != tiles.end(); ++iter)
    {
        qDeleteAll(*iter);
        (*iter).clear();
    }

    tiles.clear();
}

void BookView::goToSection(int section, float multiplier)
{ 
    history.push_back(currentPosition);
    if(viewMode == PageMode){
        if(tiles.contains(section)){
            currentPosition.section.number = section;
            currentPosition.multiplier = multiplier;
            currentPosition.page = ((float)currentPosition.section.numPages) * ((float)currentPosition.multiplier);
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }else{
            QString content = book->getSection(section);
            currentPosition.section.number = section;
            currentPosition.multiplier = multiplier;
            currentPosition.page = -1;
            pageWidgetCurrent->setPixmap(loadingPixmap());
            loadingBar->setValue(0);
            loadingBarProxy->setVisible(true);
            deleteTiles();
            emit requestRender(content, currentPosition.section.number);
        }
    }else{
        QString content = book->getSection(section);
        currentPosition.section.number = section;
        currentPosition.multiplier = multiplier;
        emit requestRender(content, currentPosition.section.number);
    }

    int value = ((float)book->getSectionLength(currentPosition.section.number) * ((float)currentPosition.multiplier)) + book->getSectionStart(currentPosition.section.number);
    progressSlider->setValue(value);
}

void BookView::goToTOCIndex(int index)
{
    goToSection(book->getSectionNumberOfTOCIndex(index), 0);
}

void BookView::goToBookmark(int id)
{
    QList<Bookmark> bookmarks = book->getBookmarks();
    if(id < bookmarks.count() && id >= 0){
        Bookmark bm = bookmarks.at(id);
        goToSection(bm.getSection(), bm.getMultiplier());
    }else{
        qDebug() << "Can't find bookmark!";
    }
}

void BookView::goToAnnotation(int id)
{
    QList<Annotation> annotations = book->getAnnotations();
    if(id < annotations.count() && id >= 0){
        Annotation annotation = annotations[id];
        goToSection(annotation.getSection(), annotation.getPercentage());

    }
}

void BookView::backInHistory()
{
    PositionInBook last = history.last();
    goToSection(last.section.number, last.multiplier);
    history.removeLast();
}

void BookView::nextSection()
{
    if(currentPosition.section.number < (book->getNumberOfSections() - 1)){
        goToSection(currentPosition.section.number + 1, 0);
    } else{
        goToSection(0, 0);
    }
}

void BookView::previousSection()
{
    if(currentPosition.section.number > 0){
        goToSection(currentPosition.section.number - 1, 0);
    }
}

void BookView::setFont(const QString &family, int size)
{
    renderSettings.setFont(QFont(family, size));    
}

void BookView::setNightMode(bool toggle)
{
    if(toggle){
        renderSettings.setTextColor(QColor(Qt::white));
        renderSettings.setBackgroundColor(QColor(Qt::black));
        scene->setBackgroundBrush(QBrush(Qt::black));
    }
    else{
        renderSettings.setTextColor(QColor(Qt::black));
        renderSettings.setBackgroundColor(QColor("#EDEADB"));
        scene->setBackgroundBrush(QBrush(QColor("#EDEADB")));
    }
    
}

void BookView::setViewMode(BookViewMode mode, Qt::WidgetAttribute orientation)
{

    viewMode = mode;    

    int pageWidth = 0;
    int pageHeight = 0;
    int screenWidth = 0;
    int screenHeight = 0;
    if(orientation == Qt::WA_Maemo5PortraitOrientation){
        pageWidth = 480;
        pageHeight = 730 - 70;

        screenWidth = 480;
        screenHeight = 800;
    }
    else{
        pageWidth = 800;
        pageHeight = 400 - 70;

        screenWidth = 800;
        screenHeight = 480;
    }


    if(viewMode == PageMode){

        scene->clear();

        pageWidgetCurrent = new QGraphicsPixmapItem();
        pageWidgetNext = new QGraphicsPixmapItem();

        scene->addItem(pageWidgetCurrent);
        scene->addItem(pageWidgetNext);
        connect(renderer, SIGNAL(contentHeight(int)), this, SLOT(contentHeight(int)));
        connect(renderer, SIGNAL(numberOfPages(int)), this, SLOT(numberOfPages(int)));
        scene->setSceneRect(QRectF(0, 0, screenWidth, screenHeight));

        deleteTiles();

        pageWidgetCurrent->setPos(0, topMargin);
        pageWidgetNext->setPos(pageWidth, topMargin);

        pageWidgetCurrent->setPixmap(loadingPixmap());

        renderer->enablePagination(true);
        renderer->enablePixmapRendering(true);

        

    } else{
        scene->clear();

        scrollablePage = new ScrollablePage();
        scene->addItem(scrollablePage);
        connect(renderer, SIGNAL(webPageLoaded(QWebPage*)), this, SLOT(webPageLoaded(QWebPage*)));
        connect(scrollablePage, SIGNAL(positionChanged(int)), this, SLOT(sectionScrolled(int)));
        scrollablePage->setGeometry(0, topMargin, pageWidth, pageHeight);
        scene->setSceneRect(0, 0, screenWidth, screenHeight);
        renderer->enablePagination(false);
        renderer->enablePixmapRendering(false);

    }

    const int sliderHeight = 32;
    const int leftMargin = 5;


    progressSlider = new BookProgressSlider();
    progressSlider->setMinimumWidth(pageWidth - 10);
    progressSlider->setMaximum(100);

    progressSlider->setColors(QColor("#3d342e"), renderSettings.getBackgroundColor());

    connect(progressSlider, SIGNAL(sliderReleased()), this, SLOT(sliderMoved()));
    sliderProxy = scene->addWidget(progressSlider);
    sliderProxy->setPos(leftMargin, screenHeight - (sliderHeight + leftMargin));

    loadingBar = new QProgressBar();
    loadingBar->setStyleSheet("QProgressBar{ border: 2px solid #3d342e; border-radius: 10px; background: " + renderSettings.getBackgroundColor().name() + "} QProgressBar::chunk{ background: #3d342e; }");
    loadingBar->setMaximumSize(QSize(240, 20));
    loadingBar->setMinimumSize(QSize(240, 20));
    loadingBar->setTextVisible(false);
    loadingBar->setMinimum(0);


    loadingBarProxy = scene->addWidget(loadingBar);
    loadingBarProxy->setPos((screenWidth / 2) - 120, (screenHeight / 2) - 10);
    loadingBarProxy->setVisible(false);

    toolbarWidget = new GraphicsWidget();
    QPalette palette;
    palette.setColor(QPalette::Window, renderSettings.getBackgroundColor());
    toolbarWidget->setPalette(palette);
    QGraphicsLinearLayout *toolbarLayout = new QGraphicsLinearLayout();
    toolbarWidget->setMinimumWidth(screenWidth);
    toolbarWidget->setMinimumHeight(70);
    toolbarLayout->setPreferredSize(screenWidth, 70);
    toolbarLayout->setMinimumWidth(screenWidth);

    wmDashboardIcon = new ToolbarIcon(":/wmdashboardicon_normal.png", ":/wmdashboardicon_pressed.png");
    connect(wmDashboardIcon, SIGNAL(released()), this, SLOT(wmDashboard()));
    wmBackIcon = new ToolbarIcon(":/wmbackicon_normal.png", ":/wmbackicon_pressed.png");
    connect(wmBackIcon, SIGNAL(released()), this, SLOT(wmClose()));

    TOCIcon = new ToolbarIcon(":/tocicon_normal.png", ":/tocicon_pressed.png");
    connect(TOCIcon, SIGNAL(released()), this, SLOT(tocNavigation()));

    bookmarkIcon = new ToolbarIcon(":/bookmarkicon_normal.png", ":/bookmarkicon_pressed.png");
    connect(bookmarkIcon, SIGNAL(released()), this, SLOT(viewBookmarks()));

    annotationIcon = new ToolbarIcon(":/annotationicon_normal.png", ":/annotationicon_pressed.png");
    connect(annotationIcon, SIGNAL(released()), this, SLOT(viewAnnotations()));

    historyBackIcon = new ToolbarIcon(":/historybackicon_normal.png", ":/historybackicon_pressed.png");
    connect(historyBackIcon, SIGNAL(released()), this, SLOT(backInHistory()));

    nextSectionIcon = new ToolbarIcon(":/nextsectionicon_normal.png", ":/nextsectionicon_pressed.png");
    connect(nextSectionIcon, SIGNAL(released()), this, SLOT(nextSection()));
    previousSectionIcon = new ToolbarIcon(":/previoussectionicon_normal.png", ":/previoussectionicon_pressed.png");
    connect(previousSectionIcon, SIGNAL(released()), this, SLOT(previousSection()));

    toolbarLayout->addItem(wmDashboardIcon);
    toolbarLayout->addItem(TOCIcon);
    if(viewMode == PageMode){
        toolbarLayout->addItem(historyBackIcon);
    } else {
        toolbarLayout->addItem(previousSectionIcon);
        toolbarLayout->addItem(nextSectionIcon);
    }
    toolbarLayout->addItem(bookmarkIcon);
    toolbarLayout->addItem(annotationIcon);
    toolbarLayout->addItem(wmBackIcon);
    

    toolbarWidget->setLayout(toolbarLayout);

    scene->addItem(toolbarWidget);
    toolbarWidget->setPos(0, 0);

    pageSize = QSizeF(pageWidth, pageHeight); 
    renderer->setPageSize(pageSize);
    renderSettings.setMaxImageSize(pageWidth-(2*leftMargin), pageHeight-(2*topMargin));

}

void BookView::addBookmark(QString name, QColor color)
{
    float multiplier = currentPosition.multiplier;
    if(viewMode == PageMode) 
        multiplier = ((float)currentPosition.page / ((float)currentPosition.section.numPages));
    Bookmark bookmark(name, color, currentPosition.section.number, multiplier);
    book->addBookmark(bookmark);
}

void BookView::mousePressEvent(QMouseEvent *event)
{
    if(viewMode == PageMode && event->pos().y() < sliderProxy->pos().y())
        swipeStart = event->pos();    

    isTap = true;
    tapMove = 0;

    QGraphicsView::mousePressEvent(event);  
}

void BookView::mouseMoveEvent(QMouseEvent *event)
{
   if(viewMode == PageMode){
       if(!swipeStart.isNull() && event->pos().x() > 0){
            float moveX = event->pos().x() - swipeStart.x();
            swipeStart = event->pos();
            pageWidgetCurrent->moveBy(moveX, 0);

            //If moved enough user is not tapping
            tapMove += moveX;
            if((tapMove > 10 || tapMove < -10) && isTap){
                isTap = false;
                tapMove = 0;
            }

            int pageWidth = pageWidgetNext->boundingRect().width();
            //Calculate new position of pageWidgetNext.
            //It is either on the left or right side of pageWidgetCurrent
            float nextX = (pageWidgetCurrent->pos().x() >= 0) ? pageWidgetCurrent->pos().x() - pageWidth : pageWidgetCurrent->pos().x() + pageWidth;
            pageWidgetNext->setPos(nextX, pageWidgetNext->pos().y());

            if(nextX > 0){
                pageWidgetNext->setPixmap(getNextPagePixmap());
            }else{
                pageWidgetNext->setPixmap(getPreviousPagePixmap());
            }
       }
   }
    QGraphicsView::mouseMoveEvent(event);  
}

void BookView::mouseReleaseEvent(QMouseEvent *event)
{
    if(viewMode == PageMode){
        int pageWidth = pageSize.width();
        if(pageWidgetCurrent->x() < -(pageWidth/2)){
            pageWidgetCurrent->setPos(-pageWidth, pageWidgetCurrent->pos().y());
            pageWidgetNext->setPos(0, pageWidgetNext->pos().y());
            //Switch pointers of current and next page widget
            QGraphicsPixmapItem *temp = pageWidgetCurrent;
            pageWidgetCurrent = pageWidgetNext;
            pageWidgetNext = temp;
            //Handle changing page logic.
            nextPage();
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }else if(pageWidgetCurrent->x() > (pageWidth/2) && (currentPosition.section.number != 0 || currentPosition.page != 0)){
            pageWidgetCurrent->setPos(pageWidth, pageWidgetCurrent->pos().y());
            pageWidgetNext->setPos(0, pageWidgetNext->pos().y());
            //Switch pointers of current and next page widget
            QGraphicsPixmapItem *temp = pageWidgetCurrent;
            pageWidgetCurrent = pageWidgetNext;
            pageWidgetNext = temp;
            //Handle changing page logic.
            previousPage(); 
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
            
        }else{
            //If user didn't drag far enought snap current page back to position.
            pageWidgetCurrent->setPos(0, pageWidgetCurrent->pos().y());
            pageWidgetNext->setPos(pageWidth, pageWidgetNext->pos().y());
        }
        swipeStart = QPoint();
    }


    //Test if the user tapped a annotation
    if(isTap){
        QString annotationHit;
        bool hit = false;
        if(viewMode == PageMode){
            hit = renderer->hitAnnotation(event->pos() - QPoint(0, topMargin), annotationHit, tiles[currentPosition.section.number][currentPosition.page]);
        } else{
            QPoint t = QPoint(event->pos().x(), scrollablePage->getPosition() + event->pos().y() + topMargin);
            hit = renderer->hitAnnotation(t, annotationHit);
        }

        if(hit){
            Annotation *a = book->getAnnotation(annotationHit);
            if(a){
                QMessageBox annotationBox(this);
                annotationBox.setText(a->getAnnotation());
                annotationBox.setWindowTitle(tr("Annotation"));
                annotationBox.setStandardButtons(QMessageBox::Close);
                annotationBox.exec();
            }
        }
    }


    QGraphicsView::mouseReleaseEvent(event);
}


bool BookView::eventFilter(QObject *object, QEvent *event)
{
    if(event->type() == QEvent::Close){
        delete object;
//        parentWidget()->showNormal();
        QApplication::setActiveWindow(parentWidget());
        parentWidget()->setFocus();
        return true;
    }

    return QObject::eventFilter(object, event);
}


QPixmap BookView::loadingPixmap()
{
    //TODO: Replace with a more useful loading picture.
/*    QPixmap tile(20, 20);
    QPainter painter;
    painter.begin(&tile);
    painter.setPen(Qt::NoPen);
    painter.setBrush(Qt::lightGray);
    painter.drawRect(0, 0, 20, 20);
    painter.setBrush(Qt::white);
    painter.drawRect(0, 0, 10, 10);
    painter.drawRect(10, 10, 10, 10);
    painter.end();*/
    QPixmap pixmap(pageSize.toSize());
    pixmap.fill(renderSettings.getBackgroundColor());
/*    painter.begin(&pixmap);
    painter.drawTiledPixmap(0, 0, pageSize.width(), pageSize.height(), tile);
    painter.end();*/
    return pixmap;
}

void BookView::nextPage()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        history.push_back(currentPosition);
        if((currentPosition.page + 1) < currentPosition.section.numPages){
            currentPosition.page++;
            currentPosition.multiplier = (((float)currentPosition.page) / ((float)currentPosition.section.numPages));
            int value = ((float)book->getSectionLength(currentPosition.section.number) * ((float)currentPosition.multiplier)) + book->getSectionStart(currentPosition.section.number);
            progressSlider->setValue(value);
        }else{
            goToSection(currentPosition.section.number + 1, 0);
        }
    }


}

void BookView::previousPage()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        history.push_back(currentPosition);
        if(0 < currentPosition.page){
            currentPosition.page--;
            currentPosition.multiplier = (((float)currentPosition.page) / ((float)currentPosition.section.numPages));
            int value = ((float)book->getSectionLength(currentPosition.section.number) * ((float)currentPosition.multiplier)) + book->getSectionStart(currentPosition.section.number);

            progressSlider->setValue(value);

        }else{
            goToSection(currentPosition.section.number - 1, 1.0);
        }
    }

    if(currentPosition.section.number < 0)
        currentPosition.section.number = 0;

}

QPixmap BookView::getCurrentPagePixmap()
{
    if(tiles.contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        if(pages.contains(currentPosition.page)){
            if(!pages[currentPosition.page]->isRendered())
                renderer->renderTile(pages[currentPosition.page]);
            return QPixmap::fromImage(pages[currentPosition.page]->getPixmap());
        }
    }    
    
    return loadingPixmap();
}

QPixmap BookView::getNextPagePixmap()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        if(pages.contains(currentPosition.page+1)){
            if(!pages[currentPosition.page+1]->isRendered())
                renderer->renderTile(pages[currentPosition.page+1]);
            return QPixmap::fromImage(pages[currentPosition.page+1]->getPixmap());
        }else if(tiles.keys().contains(currentPosition.section.number+1)){
            pages = tiles[currentPosition.section.number+1];
            if(!pages[0]->isRendered())
                renderer->renderTile(pages[0]);
            return QPixmap::fromImage(pages[0]->getPixmap());
        }
    }
    return loadingPixmap();
}

QPixmap BookView::getPreviousPagePixmap()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        if(pages.contains(currentPosition.page-1)){
            if(!pages[currentPosition.page-1]->isRendered())
                renderer->renderTile(pages[currentPosition.page-1]);
            return QPixmap::fromImage(pages[currentPosition.page-1]->getPixmap());
        }
    }
    return loadingPixmap();
}

void BookView::volumeUpPressed(){
   if(viewMode == PageMode){
        nextPage();
        pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
   }else{
        autoScrollBy += autoScrollStep; 
        autoScrollTimer.start();
   }
}

void BookView::volumeDownPressed()
{
   if(viewMode == PageMode){
        previousPage();
        pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
   }else{
        autoScrollBy -= autoScrollStep; 
        autoScrollTimer.start();
   }
}


void BookView::saveProgress()
{
    if(viewMode == PageMode){
        int section = currentPosition.section.number;
        float multiplier = ((float)currentPosition.page / ((float)currentPosition.section.numPages));

        BookProgress progress;
        progress.setProgress(section, multiplier);
        book->setProgress(progress);
    }else{
        int scrollPosition = scrollablePage->getPosition();
        int section = currentPosition.section.number; 
        float multiplier = ((float)scrollPosition / (float)currentPosition.section.contentHeight);

        BookProgress progress;
        progress.setProgress(section, multiplier);
        book->setProgress(progress);
    }

}

void BookView::addTile(RenderTile *tile)
{
    if(tiles.keys().contains(tile->getSection()))
        tiles[tile->getSection()].insert(tile->getID(), tile);
    else{
        QHash<int, RenderTile*> pages;
        pages.insert(tile->getID(), tile);
        tiles.insert(tile->getSection(), pages);
    }

    if(tile->getID() == currentPosition.page && tile->getSection() == currentPosition.section.number){
//        pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
    }
    
    //Set first page as cover if no cover image found.
   /* if(tile->getSection() == 0 && tile->getID() == 0 && book->getCoverImage().isNull()){
        book->addCoverImage(QPixmap::fromImage(tile->getPixmap()));
    }*/
    loadingBar->setValue(tile->getID() + 1);
    QApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

    if(tile->getID() >= loadingBar->maximum()- 1){
        loadingBarProxy->setVisible(false);
    }


}

void BookView::contentHeight(int height)
{
    if(viewMode == PageMode){
        if(currentPosition.page == -1){
            currentPosition.page = ((float)height / pageSize.height()) * ((float)currentPosition.multiplier);
            currentPosition.section.contentHeight = height;
            currentPosition.section.numPages = (float)height / (float)pageSize.height();

            loadingBar->setMaximum(currentPosition.section.numPages);
        }
        
    }else{
        currentPosition.section.contentHeight = height;
        int scrollPosition = (float)height * ((float)currentPosition.multiplier);
        scrollablePage->setPosition(scrollPosition);
    }
        
}

void BookView::numberOfPages(int n)
{
    currentPosition.section.numPages = n;
    currentPosition.page = ((float)currentPosition.section.numPages) * ((float)currentPosition.multiplier);
    if(currentPosition.page >= currentPosition.section.numPages)
        currentPosition.page = currentPosition.section.numPages - 1;
    pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
}


void BookView::webPageLoaded(QWebPage *page){
    scrollablePage->setPage(page);
}

void BookView::handleAutoScroll()
{
    //Only scroll if MainWindow is active.
    if(parentWidget()->isActiveWindow()){
        if(autoScrollBy != 0)
            scrollablePage->scrollBy(autoScrollBy);
        else
            autoScrollTimer.stop();
    }
}

void BookView::openContextMenu(const QPoint &pos)
{
    QMenu *menu = new QMenu();
    menu->addAction(tr("Add bookmark"), this, SLOT(addBookmark()));
    menu->addAction(tr("Add annotation"), this, SLOT(addAnnotation()));

    QString temp;
    if(viewMode == PageMode){
        temp = renderer->getParagraphAt(pos - QPoint(0, 20), tiles[currentPosition.section.number][currentPosition.page]);
    } else{
        QPoint t = QPoint(pos.x(), scrollablePage->getPosition() + pos.y());
        temp = renderer->getParagraphAt(t);
    }

    if(temp.isEmpty()) temp = "Couldn't get paragraph!";

    annotationParagraph = temp;

    menu->exec(mapToGlobal(pos));
}

void BookView::addBookmark()
{
    AddBookmarkDialog dialog;
    if(dialog.exec() == QDialog::Accepted){
        addBookmark(dialog.getName(), dialog.getColor());        
    }
}

void BookView::addAnnotation()
{
    AddAnnotationDialog dialog;
    dialog.setParagraph(annotationParagraph);
    if(dialog.exec() == QDialog::Accepted){
        float multiplier = currentPosition.multiplier;
        if(viewMode == PageMode) 
            multiplier = ((float)currentPosition.page / ((float)currentPosition.section.numPages));
        Annotation annotation(currentPosition.section.number, multiplier, annotationParagraph, dialog.getSelectedText(), dialog.getAnnotation(), dialog.getColor());

        book->addAnnotation(annotation);
        renderer->setAnnotations(book->getAnnotations());
        if(viewMode == PageMode){
            renderer->renderTile(tiles[currentPosition.section.number][currentPosition.page]);
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }
    }
}

void BookView::sectionScrolled(int pos)
{
    currentPosition.multiplier = ((float)pos / (float)currentPosition.section.contentHeight);
    int value = ((float)book->getSectionLength(currentPosition.section.number) * ((float)currentPosition.multiplier)) + book->getSectionStart(currentPosition.section.number);
    progressSlider->setValue(value);
}

void BookView::sliderMoved()
{
    int sliderPosition = progressSlider->sliderPosition();
    int section = book->getSectionAtPosition(sliderPosition);

    if(section != -1){
        currentPosition.multiplier = (((float)sliderPosition - (float)book->getSectionStart(section)) / (float)book->getSectionLength(section));
        goToSection(section, currentPosition.multiplier);
    }

}

void BookView::tocNavigation()
{
    QList<QString> toc = book->getTOC();
    if(toc.isEmpty()){
        QMaemo5InformationBox::information(this, tr("This book doesn't have table of contents information."), QMaemo5InformationBox::DefaultTimeout); 
        return;
    }
    //TODO: Show notofication banner if no TOC
    QList<ListItem > list;
    for(QList<QString>::iterator iter = toc.begin(); iter != toc.end(); ++iter){
        ListItem item;
        item.text = *iter;
        list.push_back(item);
    }

    ListViewDialog dialog(this);
    dialog.setWindowTitle(tr("Table of content"));
    ListModel *model = new ListModel(list);
    dialog.setModel(model);
    dialog.setAttribute(Qt::WA_Maemo5StackedWindow);
    dialog.setWindowFlags(dialog.windowFlags() | Qt::Window);
    connect(&dialog, SIGNAL(selectedIndex(int)), this, SLOT(goToTOCIndex(int)));
//    dialog.installEventFilter(this);
    dialog.exec();
}

void BookView::viewBookmarks()
{
    QList<Bookmark> bm = book->getBookmarks();
    if(bm.isEmpty()){
        QMaemo5InformationBox::information(this, tr("No bookmarks found. Press and hold to add a bookmark."), QMaemo5InformationBox::DefaultTimeout); 
        return;
    }
    QList<ListItem > list;
    for(QList<Bookmark>::iterator iter = bm.begin(); iter != bm.end(); ++iter){
        Bookmark bookmark = *iter;
        QPixmap ip(32, 32); 
        ip.fill(Qt::transparent);
        QPainter painter(&ip);
        painter.setRenderHint(QPainter::Antialiasing, true);
        painter.setPen(Qt::NoPen);
        painter.setBrush(bookmark.getColor());
        painter.drawRoundedRect(ip.rect(), 3, 3);
        ListItem item;
        item.text = bookmark.getName();
        QString sectionName = book->getSectionName(bookmark.getSection());
        if(sectionName.isEmpty())
            sectionName = tr("Section") + " " + QString::number(bookmark.getSection());
        item.subText =  sectionName + ", " + bookmark.percentageAsString() + "%";
        item.icon = ip;

        list.push_back(item);
    }
    
    ListViewDialog dialog(this, true);
    dialog.setWindowTitle(tr("Bookmarks"));
    ListModel *model = new ListModel(list);
    dialog.setModel(model);
    dialog.setAttribute(Qt::WA_Maemo5StackedWindow);
    dialog.setWindowFlags(dialog.windowFlags() | Qt::Window);
    connect(&dialog, SIGNAL(selectedIndex(int)), this, SLOT(goToBookmark(int)));
    connect(&dialog, SIGNAL(deleteIndex(int)), book, SLOT(removeBookmark(int)));
    dialog.exec();
}


void BookView::viewAnnotations()
{
    QList<Annotation> a = book->getAnnotations();
    if(a.isEmpty()){
        
        QMaemo5InformationBox::information(this, tr("No annotations found. Press and hold paragraph to add an annotation."), QMaemo5InformationBox::DefaultTimeout); 
        return;
    }
    QList<ListItem> list;
    for(QList<Annotation>::iterator iter = a.begin(); iter != a.end(); ++iter){
        Annotation annotation = *iter;
        QPixmap ip(32, 32); 
        ip.fill(Qt::transparent);
        QPainter painter(&ip);
        painter.setRenderHint(QPainter::Antialiasing, true);
        painter.setPen(Qt::NoPen);
        painter.setBrush(annotation.getColor());
        painter.drawRoundedRect(ip.rect(), 3, 3);

        ListItem item;
        item.icon = ip;
        item.text = annotation.getAnnotation();
        item.subText = book->getSectionName(annotation.getSection());

        list.push_back(item);
    }
    ListViewDialog dialog(this, true);
    dialog.setWindowTitle(tr("Annotations"));
    ListModel *model = new ListModel(list);
    dialog.setModel(model);
    connect(&dialog, SIGNAL(selectedIndex(int)), this, SLOT(goToAnnotation(int)));
    connect(&dialog, SIGNAL(deleteIndex(int)), book, SLOT(removeAnnotation(int)));
    dialog.setAttribute(Qt::WA_Maemo5StackedWindow);
    dialog.setWindowFlags(dialog.windowFlags() | Qt::Window);
    dialog.exec();
    
}

void BookView::wmDashboard()
{ 
    QDBusConnection c = QDBusConnection::sessionBus();
    QDBusMessage m = QDBusMessage::createSignal("/", "com.nokia.hildon_desktop", "exit_app_view");
    c.send(m);
}

void BookView::wmClose()
{
    emit closeWindow();
}
