#include "startwindow.h"
#include "mainwindow.h"

#include <QScrollArea>
#include <QFileDialog>
#include <QMenuBar>

#include <QMaemo5InformationBox>

#include "shared.h"


StartWindow::StartWindow(QWidget *parent) : QMainWindow(parent), readingUI(0), searcCatalogsView(0)
{
    setWindowTitle("MeBook");
    
    scrollArea = new QScrollArea(this);
    setCentralWidget(scrollArea);

    //Scroller is used to detect if user is tapping an item or scrolling the QAbstractKineticScroller.
    //TODO: There might be a better way to do this..
    scroller = scrollArea->property("kineticScroller").value<QAbstractKineticScroller*>();
    
    setAttribute(Qt::WA_Maemo5StackedWindow);

    bookList = new QWidget();
    bookList->setContextMenuPolicy(Qt::NoContextMenu);
    scrollArea->setWidget(bookList);
    scrollArea->setWidgetResizable(true);

    listLayout = new QVBoxLayout(bookList);
    listLayout->setSpacing(14);

    QPushButton *openFileButton = new QPushButton(tr("Open file"), this);
    listLayout->addWidget(openFileButton);
    listLayout->setAlignment(openFileButton, Qt::AlignTop);
    listLayout->setAlignment(Qt::AlignTop);
    connect(openFileButton, SIGNAL(pressed()), this, SLOT(openFile()));

    QPushButton *searchCatalogsButton = new QPushButton(tr("Search catalogs"), this);
    listLayout->addWidget(searchCatalogsButton);
    listLayout->setAlignment(searchCatalogsButton, Qt::AlignTop);
    connect(searchCatalogsButton, SIGNAL(clicked()), this, SLOT(searchCatalogs()));

    QMenuBar *menuBar = new QMenuBar(this);
    QAction *settings = menuBar->addAction(tr("Settings"));
    connect(settings, SIGNAL(triggered()), this, SLOT(openSettingsDialog()));
    QAction *about = menuBar->addAction(tr("About"));
    connect(about, SIGNAL(triggered()), this, SLOT(openAboutDialog()));

    setMenuBar(menuBar);

    readSettings();

    library = new Library("/home/user/MyDocs/books", "/home/user/MyDocs/books/bookdb.db", this);

    connect(library, SIGNAL(bookAdded(Book*)), this, SLOT(addBook(Book*)));
    connect(library, SIGNAL(bookRemoved(Book*)), this, SLOT(removeBook(Book*)));
    library->loadLibrary();

    feedBooksCatalog = new FeedBooks(this);
    connect(feedBooksCatalog, SIGNAL(bookSaved(Book*)), library, SLOT(addToLibrary(Book*)));
    connect(feedBooksCatalog, SIGNAL(message(const QString)), this, SLOT(showMessage(const QString)));
    connect(feedBooksCatalog, SIGNAL(openBook(Book*)), this, SLOT(openBook(Book*)));

    osso_context = osso_initialize(OSSO_NAME.toUtf8(), "0.0.1", TRUE, NULL);

}

StartWindow::~StartWindow()
{
    if(osso_context)
        osso_deinitialize(osso_context);
}


void StartWindow::readSettings()
{

    settings = new QSettings("lue", "MeBook-0.0.8");

    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);


}

void StartWindow::openFile()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Open book"), "/home/user/MyDocs", tr("eBook Files (*.epub)"));
    
    if(!filename.isEmpty()){
        Book *book = new Book(filename);
        library->addToLibrary(book);
        openBook(book);
    }
}

void StartWindow::openBook(Book *book)
{
    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);

    if(!readingUI){
        readingUI = new MainWindow(osso_context, this);
        connect(readingUI, SIGNAL(closing()), this, SLOT(bookViewClosed()));
    }

    readingUI->setFont(settings->value("font-family", "Sans Serif").toString(), settings->value("font-size", 24).toInt());
    readingUI->setColorMode(!settings->value("colors", true).toBool());
    readingUI->enableBacklightForcing(settings->value("backlight", false).toBool());
    readingUI->setViewMode(static_cast<BookViewMode>(settings->value("viewmode", 1).toInt()), static_cast<Qt::WidgetAttribute>(settings->value("orientation", static_cast<int>(Qt::WA_Maemo5PortraitOrientation)).toInt()));

    readingUI->openBook(book);
    readingUI->show();

    //Hide other windows 
    if(searcCatalogsView){
        if(searcCatalogsView->isVisible())
            searcCatalogsView->hide();
    }

}

void StartWindow::openSettingsDialog()
{
    SettingsDialog *settingsDialog = new SettingsDialog(this);    
    settingsDialog->setOrientation(settings->value("orientation").toInt());
    settingsDialog->setColors(settings->value("colors").toInt());
    settingsDialog->setFontFamily(settings->value("font-family").toString());
    settingsDialog->setFontSize(settings->value("font-size").toInt());
    settingsDialog->setBacklightForce(settings->value("backlight").toBool());
    settingsDialog->setViewMode(static_cast<BookViewMode>(settings->value("viewmode").toInt()));


    if(settingsDialog->exec() == QDialog::Accepted){
        int orientation = settingsDialog->getOrientation();
        if(orientation != settings->value("orientation")){
            settings->setValue("orientation", orientation); 
            setAttribute(static_cast<Qt::WidgetAttribute>(orientation), true);
        }

        int colors = settingsDialog->getColors();
        if(colors != settings->value("colors")){
            settings->setValue("colors", colors);
        }

        settings->setValue("font-family", settingsDialog->getFontFamily());
        settings->setValue("font-size", settingsDialog->getFontSize());
        settings->setValue("backlight", settingsDialog->getBacklighForce());
        settings->setValue("viewmode", settingsDialog->getViewMode());

        settings->sync();
    }
}

void StartWindow::openAboutDialog()
{
    QDialog dialog(this);
    dialog.setWindowTitle(tr("About MeBook"));
    QHBoxLayout horizontalLayout;

    QLabel icon;
    QPixmap pixmap = QIcon::fromTheme("mebook").pixmap(48, 48);
    icon.setPixmap(pixmap.scaled(QSize(72, 72)));
    horizontalLayout.addWidget(&icon);
    horizontalLayout.setAlignment(&icon, Qt::AlignTop);

    QVBoxLayout verticalLayout;
    horizontalLayout.addLayout(&verticalLayout);
    horizontalLayout.setAlignment(&verticalLayout, Qt::AlignLeft);
    horizontalLayout.setStretch(1, 2);

    //TODO: Get version number automagically.
    QLabel name("MeBook " + QString(APP_VERSION));
    name.setFont(QFont("Sans", 22));
    verticalLayout.addWidget(&name);
    verticalLayout.setAlignment(&name, Qt::AlignLeft | Qt::AlignTop);

    QLabel copyright("<qt>&copy; 2010 by Juhana Jauhiainen</qt>");
    QFont copyrightFont("Sans", 12);
    QPalette palette = QApplication::palette();
    palette.setColor(QPalette::Normal, QPalette::WindowText, Qt::darkGray);
    copyright.setPalette(palette);
    copyright.setFont(copyrightFont);
    verticalLayout.addWidget(&copyright);
    verticalLayout.setStretch(2, 3);
    verticalLayout.setAlignment(&copyright, Qt::AlignLeft | Qt::AlignBottom);

    dialog.setLayout(&horizontalLayout);

    QPushButton reportBug(tr("Report bugs"));
    connect(&reportBug, SIGNAL(clicked()), this, SLOT(reportBugs()));
    horizontalLayout.addWidget(&reportBug);
    dialog.exec();
}

void StartWindow::searchCatalogs()
{
    if(!searcCatalogsView){
        searcCatalogsView = new SearchCatalogsView(feedBooksCatalog, this);
        connect(searcCatalogsView, SIGNAL(closing()), this, SLOT(searchCatalogsClosed()));
    }
    setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    searcCatalogsView->show();
}

void StartWindow::searchCatalogsClosed()
{
    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);
}

void StartWindow::bookViewClosed()
{
    delete readingUI;
    readingUI = 0;
}

void StartWindow::showMessage(const QString message)
{
    QMaemo5InformationBox::information(this, message, QMaemo5InformationBox::DefaultTimeout); 
}

void StartWindow::addBook(Book *book)
{

    BookListItem *item;
    item = new BookListItem(book, scroller, scrollArea);
    listLayout->addWidget(item);
    connect(item, SIGNAL(selected(Book*)), this, SLOT(openBook(Book*)));
    listLayout->setAlignment(item, Qt::AlignTop);
    //Call adjust size so widget's size is recalculated.
    bookList->adjustSize();
    bookListItems.push_back(item);
}

void StartWindow::removeBook(Book *book)
{
    if(!book->getFilename().isEmpty()){
        for(QList<BookListItem*>::iterator iter = bookListItems.begin(); iter != bookListItems.end(); ++iter){
            BookListItem *item = *iter;
            if(item->book()->getFilename() == book->getFilename()){
                bookListItems.removeOne(item);
                listLayout->removeWidget(item);
                delete item;
                bookList->adjustSize();
                return;
            
            }
        }
    }
}

void StartWindow::reportBugs()
{
    QDesktopServices::openUrl(QUrl("http://code.google.com/p/mebook/issues/"));
}
