/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-provider/microfeederror.h>
#include <microfeed-common/microfeedprotocol.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdint.h>

typedef struct _ItemDataProperty ItemDataProperty;

typedef struct {
	time_t last_updated;
} PermanentData;

struct _MicrofeedFeed {
	MicrofeedObject object;

	char* uri;
	char* name;
	MicrofeedPublisher* publisher;
	char* object_path;
	MicrofeedFeedPermission feed_permission;
	MicrofeedItem* metadata_item;
	MicrofeedFeedCallbacks callbacks;
	void* user_data;
	MicrofeedFeedIterator* iterators;

	int updating;
	MicrofeedDatabaseTable* database_table;
	MicrofeedDatabaseIndex* uid_index;
	MicrofeedDatabaseIndex* timestamp_uid_index;
	MicrofeedStore* data_properties;
	MicrofeedStore* virtual_feed_properties;
	MicrofeedFeed* master_feed;
	MicrofeedStore* virtual_feeds;
	PermanentData* permanent_data;
	unsigned long int update_interval;
};

struct _MicrofeedFeedIterator {
	MicrofeedFeedIterator* previous;
	MicrofeedFeedIterator* next;

	MicrofeedFeed* feed;
	MicrofeedDatabaseIterator* database_iterator;
	int backwards : 1;
	int timeline : 1;
};

typedef struct {
	char* property_name;
	char* feed_uri_prefix;
	int create : 1;
} VirtualFeedProperty;

static void free_function(MicrofeedFeed* feed);
static const char* virtual_feed_property_get_property_name(VirtualFeedProperty* virtual_feed_property);
static void virtual_feed_item_added(MicrofeedFeed* master_feed, const char* feed_uri_prefix, const char* uri_postfix, MicrofeedItem* item, int create);
static void virtual_feed_item_removed(MicrofeedFeed* master_feed, const char* feed_uri_prefix, const char* uri_postfix, MicrofeedItem* item);
static void virtual_feed_item_changed(MicrofeedFeed* master_feed, const char* feed_uri_prefix, const char* uri_postfix, MicrofeedItem* item, int create);
static void handle_item_property_change(MicrofeedFeed* feed, MicrofeedItem* old_item, MicrofeedItem* new_item);
static void handle_item_status_change(MicrofeedFeed* feed, MicrofeedItem* item);
static void remove_old_items(MicrofeedFeed* feed);
static void send_item_signal(MicrofeedFeed* feed, const char* destination, const char* signal_name, MicrofeedItem* item);

static MicrofeedClass microfeed_feed_class = {
	"MicrofeedFeed",
	(MicrofeedObjectFreeFunction)free_function
};


static MicrofeedFeed* feed_new(MicrofeedPublisher* publisher, MicrofeedFeed* master_feed, const char* uri, const char* name, MicrofeedFeedPermission feed_permission, int create, MicrofeedFeedCallbacks* callbacks, void* user_data) {
	MicrofeedFeed* feed = NULL;
	MicrofeedDatabaseTable* database_table = NULL;
	MicrofeedDatabaseIndex* uid_index = NULL;
	MicrofeedDatabaseIndex* timestamp_uid_index = NULL;
	char* database_name;
	char* s;
	void* data;
	size_t data_size;

	if ((database_table = microfeed_database_get_table(microfeed_publisher_get_database(publisher), uri)) &&
	    (uid_index = microfeed_database_table_get_index(database_table, "uid", create, (MicrofeedDatabaseIndexFunction)microfeed_item_marshalled_get_uid, microfeed_database_compare_keys_direct)) &&
	    (timestamp_uid_index = microfeed_database_table_get_index(database_table, "timestamp", create, (MicrofeedDatabaseIndexFunction)microfeed_item_marshalled_get_timestamp_uid, (MicrofeedDatabaseCompareFunction)microfeed_database_compare_keys_direct))) {
		feed = microfeed_object_new(&microfeed_feed_class, MicrofeedFeed);
		feed->uri = strdup(uri);
		feed->name = strdup(name);
		feed->publisher = microfeed_object_ref(publisher, MicrofeedPublisher);
		feed->object_path = strdup(microfeed_publisher_get_object_path(publisher));
		feed->feed_permission = feed_permission;
		feed->metadata_item = microfeed_item_new(MICROFEED_ITEM_UID_FEED_METADATA, time(NULL));
		microfeed_item_set_property(feed->metadata_item, MICROFEED_ITEM_PROPERTY_NAME_FEED_NAME, name);
		s = microfeed_feed_permission_to_string(feed_permission);
		if (s && *s) {
			microfeed_item_set_property(feed->metadata_item, MICROFEED_ITEM_PROPERTY_NAME_FEED_PERMISSION, s);
		}
		free(s);
		feed->iterators = NULL;
		if (callbacks) {
			feed->callbacks = *callbacks;
		}
		feed->user_data = user_data;

		feed->database_table = database_table;
		feed->uid_index = uid_index;
		feed->timestamp_uid_index = timestamp_uid_index;

		if (master_feed) {
			feed->master_feed = microfeed_object_ref(master_feed, MicrofeedFeed);
		} else {
			feed->data_properties = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)microfeed_store_get_key_direct);
			feed->virtual_feed_properties = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)virtual_feed_property_get_property_name);
			feed->virtual_feeds = microfeed_store_new_sorted_weak_references((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)microfeed_feed_get_uri);
		}
		
		if (microfeed_database_get_named_data(microfeed_publisher_get_database(publisher), feed->uri, &data, &data_size)) {
			feed->permanent_data = (PermanentData*)data;
			if (data_size != sizeof(PermanentData)) {
				feed->permanent_data = (PermanentData*)realloc(feed->permanent_data, sizeof(PermanentData));
			}
		} else {
			feed->permanent_data = microfeed_memory_allocate(PermanentData);
		}
	}
	if (!feed) {
		if (timestamp_uid_index) {
			microfeed_object_unref(timestamp_uid_index, MicrofeedDatabaseIndex);
		}
		if (uid_index) {
			microfeed_object_unref(uid_index, MicrofeedDatabaseIndex);
		}
		if (database_table) {
			microfeed_object_unref(database_table, MicrofeedDatabaseTable);
		}
	}

	return feed;	
}

MicrofeedFeed* microfeed_feed_new(MicrofeedPublisher* publisher, const char* uri, const char* name, MicrofeedFeedPermission feed_permission, int create, MicrofeedFeedCallbacks* callbacks, void* user_data) {

	return feed_new(publisher, NULL, uri, name, feed_permission, create, callbacks, user_data);
}

MicrofeedFeed* microfeed_feed_new_virtual(MicrofeedFeed* master_feed, const char* uri, const char* name, MicrofeedFeedPermission feed_permission, int create, MicrofeedFeedCallbacks* callbacks, void* user_data) {
	microfeed_assert(!master_feed->master_feed);
	
	return feed_new(master_feed->publisher, master_feed, uri, name, feed_permission, create, callbacks, user_data);
}

/* Does NOT remove subscribers */
static void free_function(MicrofeedFeed* feed) {
	MicrofeedFeedIterator* iterator;
	MicrofeedStoreIterator* store_iterator;
	char* data_property;
	VirtualFeedProperty* virtual_feed_property;

	microfeed_assert(!feed->iterators);

	if (feed->callbacks.destroy) {
		feed->callbacks.destroy(feed, feed->user_data);
	}

	microfeed_database_set_named_data(microfeed_publisher_get_database(feed->publisher), feed->uri, feed->permanent_data, sizeof(PermanentData));
	free(feed->permanent_data);
	
	free(feed->uri);
	free(feed->name);
	microfeed_object_unref(feed->publisher, MicrofeedPublisher);
	free(feed->object_path);
	microfeed_item_free(feed->metadata_item);	
	microfeed_object_unref(feed->uid_index, MicrofeedDatabaseIndex);
	microfeed_object_unref(feed->timestamp_uid_index, MicrofeedDatabaseIndex);
	microfeed_object_unref(feed->database_table, MicrofeedDatabaseTable);

	if (feed->master_feed) {
		microfeed_object_unref(feed->master_feed, MicrofeedFeed);
	} else {
		for (store_iterator = microfeed_store_iterate(feed->data_properties, NULL);
		     (data_property = microfeed_store_iterator_get(store_iterator, char));
		     microfeed_store_iterator_next(store_iterator)) {
			free(data_property);
		}
		microfeed_store_iterator_free(store_iterator);
		microfeed_store_free(feed->data_properties);

		for (store_iterator = microfeed_store_iterate(feed->virtual_feed_properties, NULL);
		     (virtual_feed_property = microfeed_store_iterator_get(store_iterator, VirtualFeedProperty));
		     microfeed_store_iterator_next(store_iterator)) {
			free(virtual_feed_property->property_name);
			free(virtual_feed_property->feed_uri_prefix);
			microfeed_memory_free(virtual_feed_property);
		}
		microfeed_store_iterator_free(store_iterator);
		microfeed_store_free(feed->virtual_feed_properties);
	
		microfeed_store_free(feed->virtual_feeds);
	}
	
	feed->uri = NULL;
	feed->callbacks.destroy = NULL;
	feed->callbacks.update = NULL;
	feed->iterators = NULL;
}

MicrofeedPublisher* microfeed_feed_get_publisher(MicrofeedFeed* feed) {

	return feed->publisher;
}

void microfeed_feed_replace_item(MicrofeedFeed* feed, MicrofeedItem* item) {
	const void* data;
	size_t data_size;
	const char* uid;
	size_t uid_size;
	void* existing_data;
	size_t existing_data_size;
	MicrofeedItem* existing_item;
	const void* timestamp_uid;
	size_t timestamp_uid_size;

	if (feed->master_feed) {
		feed = feed->master_feed;
	}

	microfeed_object_lock(feed, MicrofeedFeed);

	microfeed_item_marshal(item, &data, &data_size);	
	microfeed_item_marshalled_get_uid(data, data_size, &uid, &uid_size);
	if (microfeed_database_index_get_data(feed->uid_index, uid, uid_size, &existing_data, &existing_data_size)) {
		if (data_size != existing_data_size || memcmp(data, existing_data, existing_data_size)) {
			existing_item = microfeed_item_new_from_marshalled(existing_data, existing_data_size);
			handle_item_property_change(feed, existing_item, item);
			
			send_item_signal(feed, NULL, MICROFEED_SIGNAL_NAME_ITEM_CHANGED, item);

			if (microfeed_item_get_timestamp(item) != microfeed_item_get_timestamp(existing_item)) {
				microfeed_item_marshalled_get_timestamp_uid(existing_data, existing_data_size, &timestamp_uid, &timestamp_uid_size); 
				microfeed_database_index_remove_data(feed->timestamp_uid_index, timestamp_uid, timestamp_uid_size);
			}
			microfeed_database_index_replace_data(feed->uid_index, data, data_size);

			microfeed_item_free(existing_item);
		}
		free(existing_data);
	} else {	
		handle_item_property_change(feed, NULL, item);

		send_item_signal(feed, NULL, MICROFEED_SIGNAL_NAME_ITEM_ADDED, item);

		microfeed_database_index_replace_data(feed->uid_index, data, data_size);
	}
	
	microfeed_object_unlock(feed, MicrofeedFeed);
}

void microfeed_feed_remove_item(MicrofeedFeed* feed, const char* uid) {	
	size_t uid_size;
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;

	if (feed->master_feed) {
		feed = feed->master_feed;
	}

	microfeed_object_lock(feed, MicrofeedFeed);

	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(feed->uid_index, uid, uid_size, &data, &data_size)) {
		item = microfeed_item_new_from_marshalled(data, data_size);
		handle_item_property_change(feed, item, NULL);
		microfeed_item_free(item);
		free(data);
		
		microfeed_database_index_remove_data(feed->uid_index, uid, uid_size);
		microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->object_path, feed->uri, uid);
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
}

void microfeed_feed_remove_items(MicrofeedFeed* feed, const char* start_uid, const char* end_uid) {
	MicrofeedFeed* real_feed;
	size_t start_uid_size;
	size_t end_uid_size;
	MicrofeedDatabaseIterator* iterator;
	void* data;
	size_t data_size;
	const char* uid;
	size_t uid_size;
	void* master_data;
	uint64_t timestamp;
	MicrofeedItem* item;
	MicrofeedItem* real_item;
	
	if (feed->master_feed) {
		microfeed_object_lock(feed->master_feed, MicrofeedFeed);
		real_feed = feed->master_feed;
	} else {
		real_feed = feed;
	}
	microfeed_object_lock(feed, MicrofeedFeed);
	
	start_uid_size = (start_uid ? strlen(start_uid) + 1 : 0);
	end_uid_size = (end_uid ? strlen(end_uid) + 1 : 0);
	for (iterator = microfeed_database_index_iterate(feed->uid_index, start_uid, uid_size, 0);
	     (microfeed_database_iterator_get(iterator, &data, &data_size));
	     microfeed_database_iterator_next(iterator)) {
		microfeed_item_marshalled_get_uid(data, data_size, &uid, &uid_size);
		if (end_uid && uid_size == end_uid_size && memcmp(uid, end_uid, uid_size) > 0) {
			free(data);
			break;
		}
		if (feed->master_feed) {
			microfeed_database_index_get_data(feed->master_feed->uid_index, uid, uid_size, &master_data, &data_size);
			item = microfeed_item_new_from_marshalled(master_data, data_size);
			free(master_data);
		} else {
			item = microfeed_item_new_from_marshalled(data, data_size);
		}
		handle_item_property_change(real_feed, item, NULL);
		
		microfeed_database_index_remove_data(real_feed->uid_index, uid, uid_size);
		microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(real_feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, real_feed->object_path, real_feed->uri, uid);

		free(data);
		microfeed_item_free(item);
	}
	microfeed_database_iterator_free(iterator);
	
	microfeed_object_unlock(feed, MicrofeedFeed);
	if (feed->master_feed) {
		microfeed_object_unlock(feed->master_feed, MicrofeedFeed);
	}
}

MicrofeedItem* microfeed_feed_get_item(MicrofeedFeed* feed, const char* uid) {
	void* data;
	size_t data_size;
	MicrofeedItem* item;

	if (feed->master_feed) {
		feed = feed->master_feed;
	}
	/* TODO: Should we check that the item really exists in the _virtual_ feed? */

	microfeed_object_lock(feed, MicrofeedFeed);

	if (!microfeed_database_index_get_data(feed->uid_index, uid, strlen(uid) + 1, &data, &data_size)) {
		item = NULL;
	} else {
		item = microfeed_item_new_from_marshalled(data, data_size);
		free(data);
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
	
	return item;	
}

const char* microfeed_feed_get_uri(MicrofeedFeed* feed) {

	return feed->uri;
}

const char* microfeed_feed_get_name(MicrofeedFeed* feed) {

	/* TODO: What if the name changes? */
	return feed->name;
}

void microfeed_feed_set_name(MicrofeedFeed* feed, const char* name) {
	microfeed_object_lock(feed, MicrofeedFeed);

	if (strcmp(name, feed->name)) {
		free(feed->name);
		feed->name = strdup(name);
		/* TODO: Change metadata item. */
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
}

MicrofeedItem* microfeed_feed_get_metadata_item(MicrofeedFeed* feed) {
	MicrofeedItem* item;

	microfeed_object_lock(feed, MicrofeedFeed);

	item = microfeed_item_duplicate(feed->metadata_item);

	microfeed_object_unlock(feed, MicrofeedFeed);

	return item;
}

void microfeed_feed_send_metadata_item(MicrofeedFeed* feed, const char* destination) {
	microfeed_object_lock(feed, MicrofeedFeed);

	send_item_signal(feed, destination, MICROFEED_SIGNAL_NAME_ITEM_ADDED, feed->metadata_item);

	microfeed_object_unlock(feed, MicrofeedFeed);
}


MicrofeedFeedIterator* microfeed_feed_iterate(MicrofeedFeed* feed, const char* start_uid, int backwards) {
	MicrofeedFeedIterator* iterator;

	microfeed_object_lock(feed, MicrofeedFeed);
	
	iterator = (MicrofeedFeedIterator*)microfeed_memory_allocate(MicrofeedFeedIterator);
	iterator->feed = microfeed_object_ref(feed, MicrofeedFeed);
	iterator->backwards = backwards;
	iterator->timeline = 0;
	iterator->database_iterator = microfeed_database_index_iterate(feed->uid_index, start_uid, (start_uid ? strlen(start_uid) + 1 : 0), backwards);

	if (feed->iterators) {
		feed->iterators->previous = iterator;
		iterator->next = feed->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	feed->iterators = iterator;

	microfeed_object_unlock(feed, MicrofeedFeed);

	return iterator;
}

MicrofeedFeedIterator* microfeed_feed_iterate_timeline(MicrofeedFeed* feed, time_t start_timestamp, int backwards) {
	MicrofeedFeedIterator* iterator;

	microfeed_object_lock(feed, MicrofeedFeed);
	
	iterator = (MicrofeedFeedIterator*)microfeed_memory_allocate(MicrofeedFeedIterator);
	iterator->feed = microfeed_object_ref(feed, MicrofeedFeed);
	iterator->backwards = backwards;
	iterator->timeline = 1;
	if (start_timestamp) {
		iterator->database_iterator = microfeed_database_index_iterate(feed->timestamp_uid_index, &start_timestamp, sizeof(time_t), backwards);
	} else {
		iterator->database_iterator = microfeed_database_index_iterate(feed->timestamp_uid_index, NULL, 0, backwards);
	}		

	if (feed->iterators) {
		feed->iterators->previous = iterator;
		iterator->next = feed->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	feed->iterators = iterator;

	microfeed_object_unlock(feed, MicrofeedFeed);

	return iterator;
}

void microfeed_feed_update(MicrofeedFeed* feed, const char* bus_name) {
	MicrofeedError* error;
	long timestamp;

	microfeed_object_lock(feed, MicrofeedFeed);

	if (feed->updating) {
		if (bus_name) {
			microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, feed->object_path, feed->uri);
		}
	} else {
		microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, feed->object_path, feed->uri);
		feed->updating = 1;
	
		if (feed->callbacks.update) {

			microfeed_object_unlock(feed, MicrofeedFeed);
				
			error = feed->callbacks.update(feed, (bus_name ? 1 : 0), feed->user_data);

			microfeed_object_lock(feed, MicrofeedFeed);
				
			if (error) {
				microfeed_provider_send_error_signal(microfeed_publisher_get_provider(feed->publisher), NULL, microfeed_error_get_name(error), feed->object_path, feed->uri, NULL, microfeed_error_get_message(error));
				microfeed_error_free(error);
			} else {
				feed->permanent_data->last_updated = time(NULL);
			}
		}

		microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED, feed->object_path, feed->uri);
		feed->updating = 0;
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
}

void microfeed_feed_republish(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, unsigned int max_count, const char* bus_name) {
	MicrofeedFeed* real_feed;
	size_t end_uid_size;
	unsigned int count;
	MicrofeedDatabaseIterator* iterator;
	void* data;
	size_t data_size;
	const char* uid;
	size_t uid_size;
	void* master_data;
	MicrofeedItem* item;
	
	if (feed->master_feed) {
		microfeed_object_lock(feed->master_feed, MicrofeedFeed);
		real_feed = feed->master_feed;
	} else {
		real_feed = feed;
	}
	microfeed_object_lock(feed, MicrofeedFeed);
	
	microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_STARTED, feed->object_path, feed->uri);

	end_uid_size = (end_uid ? strlen(end_uid) + 1 : 0);
	count = 0;
	for (iterator = microfeed_database_index_iterate(feed->uid_index, start_uid, (start_uid ? strlen(start_uid) + 1 : 0), 0);
	     microfeed_database_iterator_get(iterator, &data, &data_size);
	     microfeed_database_iterator_next(iterator)) {
		if (feed->master_feed) {
			microfeed_item_marshalled_get_uid(data, data_size, &uid, &uid_size);
			if (microfeed_database_index_get_data(feed->master_feed->uid_index, uid, uid_size, &master_data, &data_size)) {
				item = microfeed_item_new_from_marshalled(master_data, data_size);
				free(master_data);
			} else {
				fprintf(stderr, "ERROR: Item not found from master feed: %s\n", uid);
				break;
			}
		} else {
			item = microfeed_item_new_from_marshalled(data, data_size);
		}

		send_item_signal(feed, bus_name, MICROFEED_SIGNAL_NAME_ITEM_REPUBLISHED, item);

		microfeed_item_free(item);
		free(data);

		if (end_uid && end_uid_size == uid_size && memcmp(end_uid, uid, uid_size) <= 0) {
			break;
		}
		if (++count == max_count) {
			break;
		}
	}
	microfeed_database_iterator_free(iterator);

	microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_ENDED, feed->object_path, feed->uri);

	microfeed_object_unlock(feed, MicrofeedFeed);
	if (feed->master_feed) {
		microfeed_object_unlock(feed->master_feed, MicrofeedFeed);
	}
}

MicrofeedError* microfeed_feed_set_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_set) {
	MicrofeedError* error = NULL;
	size_t uid_size;
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	MicrofeedItemStatus status;
	MicrofeedItem* item;
	
	if (feed->master_feed) {
		feed = feed->master_feed;
	}
	/* TODO: Should we check that the item really exists in the _virtual_ feed? */

	microfeed_object_lock(feed, MicrofeedFeed);	

	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(feed->uid_index, uid, uid_size, &data, &data_size)) {
		status = microfeed_item_marshalled_get_status(data, data_size);
		if (!(status & status_to_set)) {
			if (status_to_set == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				error = feed->callbacks.mark_item(feed, uid, 1, feed->user_data);
			}
			if (!error) {
				status |= status_to_set;
				microfeed_item_marshalled_set_status(data, data_size, status);
				microfeed_database_index_replace_data(feed->uid_index, data, data_size);

				microfeed_provider_send_status_changed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, feed->object_path, feed->uri, uid, status);

				item = microfeed_item_new_from_marshalled(data, data_size);
				handle_item_status_change(feed, item);
				microfeed_item_free(item);
			}
		}
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "No such item when setting status.");
	}

	microfeed_object_unlock(feed, MicrofeedFeed);	

	return error;
}

MicrofeedError* microfeed_feed_unset_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_unset) {
	MicrofeedError* error = NULL;
	size_t uid_size;
	void* data;
	size_t data_size;
	char status;
	MicrofeedItem* item;
	
	if (feed->master_feed) {
		feed = feed->master_feed;
	}
	/* TODO: Should we check that the item really exists in the _virtual_ feed? */

	microfeed_object_lock(feed, MicrofeedFeed);	

	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(feed->uid_index, uid, uid_size, &data, &data_size)) {
		status = microfeed_item_marshalled_get_status(data, data_size);
		if (status & status_to_unset) {
			if (status_to_unset == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				error = feed->callbacks.mark_item(feed, uid, 0, feed->user_data);
			}
			if (!error) {
				status &= ~status_to_unset;
				microfeed_item_marshalled_set_status(data, data_size, status);
				microfeed_database_index_replace_data(feed->uid_index, data, data_size);

				microfeed_provider_send_status_changed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, feed->object_path, feed->uri, uid, status);

				item = microfeed_item_new_from_marshalled(data, data_size);
				handle_item_status_change(feed, item);
				microfeed_item_free(item);
			}
		}		
		free(data);	
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "No such item when unsetting status.");
	}

	microfeed_object_unlock(feed, MicrofeedFeed);	

	return error;
}

int microfeed_feed_get_item_count(MicrofeedFeed* feed) {
	
	return microfeed_database_table_get_data_count(feed->database_table);
}

/* TODO: Should use defines. */
MicrofeedFeedPermission microfeedfeed_permission_from_string(const char* string) {
	MicrofeedFeedPermission feed_permission = MICROFEED_FEED_PERMISSION_NONE;

	while (*string == ':') {
		string++;
		if (!strncmp(string, "add", 3) && (string[3] == 0 || string[3] == ':')) {
			feed_permission |= MICROFEED_FEED_PERMISSION_ADD;
			string += 3;
		}
	}
	
	return feed_permission;
}

/* TODO: Should use defines. */
char* microfeed_feed_permission_to_string(MicrofeedFeedPermission feed_permission) {
	char buffer[1024];
	char* string = buffer;
	
	if (feed_permission & MICROFEED_FEED_PERMISSION_ADD) {
		memcpy(string, ":add", 4);
		string += 4;
	}
	*string = 0;
	
	return strdup(buffer);
}

void microfeed_feed_add_data_property(MicrofeedFeed* feed, const char* property_name) {
	ItemDataProperty* item_data_property;
	
	microfeed_assert(!feed->master_feed);
	
	microfeed_store_insert(feed->data_properties, strdup(property_name));
}

void microfeed_feed_add_virtual_feed_property(MicrofeedFeed* feed, const char* property_name, const char* feed_uri_prefix, int create) {
	VirtualFeedProperty* virtual_feed_property;

	microfeed_assert(!feed->master_feed);
	
	virtual_feed_property = microfeed_memory_allocate(VirtualFeedProperty);
	if (property_name) {
		virtual_feed_property->property_name = strdup(property_name);
	}
	virtual_feed_property->feed_uri_prefix = strdup(feed_uri_prefix);
	virtual_feed_property->create = create;
	microfeed_store_insert(feed->virtual_feed_properties, virtual_feed_property);
}

long microfeed_feed_get_last_updated_time(MicrofeedFeed* feed) {

	return (long)feed->permanent_data->last_updated;
}


void microfeed_feed_start_periodical_update(MicrofeedFeed* feed) {
	unsigned long int interval;
	time_t elapsed;

	if ((interval = (unsigned long int)microfeed_publisher_get_setting_value_integer(feed->publisher, "update_interval", 0) * 60) &&
	    interval != feed->update_interval) {
printf("LAST UPDATE: %s %ld\n", feed->uri, feed->permanent_data->last_updated);
		if (feed->update_interval) {
			microfeed_provider_remove_timeout(microfeed_publisher_get_provider(feed->publisher), (MicrofeedProviderTimeoutFunction)microfeed_feed_update, microfeed_object_get_weak_reference(feed, MicrofeedFeed), NULL);
		}
		elapsed = time(NULL) - feed->permanent_data->last_updated;
		microfeed_provider_add_timeout(microfeed_publisher_get_provider(feed->publisher), (elapsed > interval ? 0 : interval - elapsed), interval, (MicrofeedProviderTimeoutFunction)microfeed_feed_update, microfeed_object_get_weak_reference(feed, MicrofeedFeed), NULL);
		feed->update_interval = interval;
	}
}

void microfeed_feed_iterator_free(MicrofeedFeedIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->feed) {
		iterator->feed->iterators = iterator->next;
	}

	microfeed_object_unref(iterator->feed, MicrofeedFeed);
	iterator->feed = NULL;
	microfeed_database_iterator_free(iterator->database_iterator);
	microfeed_memory_free(iterator);
}

MicrofeedFeed* microfeed_feed_iterator_get_feed(MicrofeedFeedIterator* iterator) {

	return iterator->feed;
}

MicrofeedItem* microfeed_feed_iterator_get_item(MicrofeedFeedIterator* iterator) {
	void* data;
	size_t data_size;
	const char* uid;
	size_t uid_size;
	void* master_data;
	uint64_t timestamp;
	MicrofeedItem* item;

	if (iterator->feed->master_feed) {
		microfeed_object_lock(iterator->feed->master_feed, MicrofeedFeed);
	}
	microfeed_object_lock(iterator->feed, MicrofeedFeed);
	
	if (!microfeed_database_iterator_get(iterator->database_iterator, &data, &data_size)) {
		item = NULL;
	} else if (iterator->feed->master_feed) {
		microfeed_item_marshalled_get_uid(data, data_size, &uid, &uid_size);
		microfeed_database_index_get_data(iterator->feed->master_feed->uid_index, uid, uid_size, &master_data, &data_size);
		item = microfeed_item_new_from_marshalled(master_data, data_size);
		free(master_data);
		free(data);
	} else {
		item = microfeed_item_new_from_marshalled(data, data_size);
		free(data);
	}
	
	microfeed_object_unlock(iterator->feed, MicrofeedFeed);
	if (iterator->feed->master_feed) {
		microfeed_object_unlock(iterator->feed->master_feed, MicrofeedFeed);
	}
	
	return item;
}

void microfeed_feed_iterator_next(MicrofeedFeedIterator* iterator) {
	if (iterator->feed->master_feed) {
		microfeed_object_lock(iterator->feed->master_feed, MicrofeedFeed);
	}
	microfeed_object_lock(iterator->feed, MicrofeedFeed);

	microfeed_database_iterator_next(iterator->database_iterator);

	microfeed_object_unlock(iterator->feed, MicrofeedFeed);
	if (iterator->feed->master_feed) {
		microfeed_object_unlock(iterator->feed->master_feed, MicrofeedFeed);
	}
}

int microfeed_feed_iterator_jump_and_remove_previous_items(MicrofeedFeedIterator* iterator, const char* uid) {
	int retval = 0;
	size_t uid_size;
	const char* key;
	size_t key_size;
	void* data;
	size_t data_size;
	char* start_key;
	size_t start_key_size;
	char* end_key;
	size_t end_key_size;
	MicrofeedItem* item;

	if (iterator->feed->master_feed) {
		microfeed_object_lock(iterator->feed->master_feed, MicrofeedFeed);
	}
	microfeed_object_lock(iterator->feed, MicrofeedFeed);
	
	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(iterator->feed->uid_index, uid, uid_size, &data, &data_size)) {
		retval = 1;
		free(data);
		if (microfeed_database_iterator_get(iterator->database_iterator, &data, &data_size)) {
			microfeed_item_marshalled_get_uid(data, data_size, &key, &key_size);
			if (key_size != uid_size || memcmp(key, uid, uid_size)) {
				start_key = malloc(key_size);
				memcpy(start_key, key, key_size);
				start_key_size = key_size;
				end_key = malloc(key_size);
				memcpy(end_key, key, key_size);
				end_key_size = key_size;
				while (microfeed_database_iterator_get(iterator->database_iterator, &data, &data_size)) {
					microfeed_item_marshalled_get_uid(data, data_size, &key, &key_size);
					if (key_size == uid_size && !memcmp(key, uid, uid_size)) {
						free(data);
						break;
					}
					end_key = realloc(end_key, key_size);
					memcpy(end_key, key, key_size);
					end_key_size = key_size;
					free(data);

					if (iterator->feed->master_feed) {
						microfeed_database_index_get_data(iterator->feed->master_feed->uid_index, end_key, end_key_size, &data, &data_size);
						item = microfeed_item_new_from_marshalled(data, data_size);
						free(data);
	
						microfeed_object_unlock(iterator->feed, MicrofeedFeed);
	
						handle_item_property_change(iterator->feed->master_feed, item, NULL);
	
						microfeed_object_lock(iterator->feed, MicrofeedFeed);
	
						microfeed_item_free(item);

						microfeed_database_index_remove_data(iterator->feed->master_feed->uid_index, key, key_size);
						microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(iterator->feed->master_feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, iterator->feed->master_feed->object_path, iterator->feed->master_feed->uri, key);
					} else {
						microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(iterator->feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, iterator->feed->object_path, iterator->feed->uri, key);
					}

					microfeed_database_iterator_next(iterator->database_iterator);
				}
				if (!iterator->feed->master_feed) {
					if (iterator->backwards) {		
						if (iterator->timeline) {
							microfeed_database_index_remove_data_range(iterator->feed->uid_index, end_key, end_key_size, start_key, start_key_size);
						} else {
							microfeed_database_index_remove_data_range(iterator->feed->timestamp_uid_index, end_key, end_key_size, start_key, start_key_size);			
						}
					} else {
						if (iterator->timeline) {
							microfeed_database_index_remove_data_range(iterator->feed->uid_index, start_key, start_key_size, end_key, end_key_size);
						} else {
							microfeed_database_index_remove_data_range(iterator->feed->timestamp_uid_index, start_key, start_key_size, end_key, end_key_size);
						}
					}
				}
				free(start_key);
				free(end_key);
			} else {
				free(data);
			}
		}
	}
	
	microfeed_object_unlock(iterator->feed, MicrofeedFeed);
	if (iterator->feed->master_feed) {
		microfeed_object_unlock(iterator->feed->master_feed, MicrofeedFeed);
	}

	return retval;
}

MicrofeedError* microfeed_feed_call_modify_item_callback(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item) {
	MicrofeedError* error;
	
	microfeed_object_lock(feed, MicrofeedFeed);

/* TODO: There must NOT be on-going update. Wait somewhere until update ends... */

	if (feed->callbacks.modify_item) {

		microfeed_object_unlock(feed, MicrofeedFeed);

		error = feed->callbacks.modify_item(feed, existing_item, new_item, feed->user_data);

		microfeed_object_lock(feed, MicrofeedFeed);
	
		if (error) {
			microfeed_provider_send_error_signal(microfeed_publisher_get_provider(feed->publisher), NULL, microfeed_error_get_name(error), feed->object_path, feed->uri, microfeed_item_get_uid((existing_item ? existing_item : new_item)), microfeed_error_get_message(error));
			microfeed_error_free(error);
		}
	} else {
		/* TODO: Send CannotModifyItem signal */
	}
	
	microfeed_object_unlock(feed, MicrofeedFeed);
}

static const char* virtual_feed_property_get_property_name(VirtualFeedProperty* virtual_feed_property) {

	return (virtual_feed_property->property_name ? virtual_feed_property->property_name : "");
}

static void virtual_feed_item_added(MicrofeedFeed* master_feed, const char* feed_uri_prefix, const char* uri_postfix, MicrofeedItem* item, int create) {
	char* uri;
	MicrofeedFeed* feed;
	const void* data;
	size_t data_size;
	
	uri = microfeed_util_string_concatenate(feed_uri_prefix, uri_postfix, NULL);
	if (!(feed = microfeed_store_get(master_feed->virtual_feeds, uri, MicrofeedFeed))) {
		if ((feed = microfeed_publisher_get_or_instantiate_feed(master_feed->publisher, uri, create))) {
			microfeed_store_insert(master_feed->virtual_feeds, feed);
		}
	}
printf("via: %s %p\n", uri, feed);
	free(uri);

	if (feed) {
		microfeed_object_lock(feed, MicrofeedFeed);

		microfeed_item_marshal_without_properties(item, &data, &data_size);
		microfeed_database_index_replace_data(feed->uid_index, data, data_size);

		send_item_signal(feed, NULL, MICROFEED_SIGNAL_NAME_ITEM_ADDED, item);

		microfeed_object_unlock(feed, MicrofeedFeed);	

		microfeed_object_unref(feed, MicrofeedFeed);
	}
}

static void virtual_feed_item_removed(MicrofeedFeed* master_feed, const char* feed_uri_prefix, const char* uri_postfix, MicrofeedItem* item) {
	char* uri;
	MicrofeedFeed* feed;
	const char* uid;
	size_t uid_size;
	
	uri = microfeed_util_string_concatenate(feed_uri_prefix, uri_postfix, NULL);
	if (!(feed = microfeed_store_get(master_feed->virtual_feeds, uri, MicrofeedFeed))) {
		if ((feed = microfeed_publisher_get_or_instantiate_feed(master_feed->publisher, uri, 0))) {
			microfeed_store_insert(master_feed->virtual_feeds, feed);
		}
	}
printf("vir: %s %p\n", uri, feed);
	free(uri);

	if (feed) {
		microfeed_object_lock(feed, MicrofeedFeed);

		uid = microfeed_item_get_uid(item);
		uid_size = strlen(uid) + 1;
		microfeed_database_index_remove_data(feed->uid_index, uid, uid_size);

		microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->object_path, feed->uri, uid);

		microfeed_object_unlock(feed, MicrofeedFeed);

		microfeed_object_unref(feed, MicrofeedFeed);
	}
}

static void virtual_feed_item_changed(MicrofeedFeed* master_feed, const char* feed_uri_prefix, const char* uri_postfix, MicrofeedItem* item, int create) {
	char* uri;
	MicrofeedFeed* feed;
	
	uri = microfeed_util_string_concatenate(feed_uri_prefix, uri_postfix, NULL);
	if (!(feed = microfeed_store_get(master_feed->virtual_feeds, uri, MicrofeedFeed))) {
		if ((feed = microfeed_publisher_get_or_instantiate_feed(master_feed->publisher, uri, create))) {
			microfeed_store_insert(master_feed->virtual_feeds, feed);
		}
	}
printf("vic: %s %p\n", uri, feed);
	free(uri);
	
	if (feed) {
		send_item_signal(feed, NULL, MICROFEED_SIGNAL_NAME_ITEM_CHANGED, item);

		microfeed_object_unref(feed, MicrofeedFeed);
	}
}

static void handle_item_property_change(MicrofeedFeed* feed, MicrofeedItem* old_item, MicrofeedItem* new_item) {
	MicrofeedStoreIterator* iterator;
	char* data_property;
	const char* old_value;
	const char* new_value;
	VirtualFeedProperty* virtual_feed_property;

	for (iterator = microfeed_store_iterate(feed->data_properties, NULL);
	     (data_property = microfeed_store_iterator_get(iterator, char));
	     microfeed_store_iterator_next(iterator)) {
		if (old_item && (old_value = microfeed_item_get_property(old_item, data_property))) {
			if (new_item && (new_value = microfeed_item_get_property(new_item, data_property))) {
				if (strcmp(old_value, new_value)) {
					microfeed_publisher_unref_data(feed->publisher, old_value);
					microfeed_publisher_ref_data(feed->publisher, new_value); 
				}
			} else {
				microfeed_publisher_unref_data(feed->publisher, old_value);
			}
		} else if (new_item && (new_value = microfeed_item_get_property(new_item, data_property))) {
			microfeed_publisher_ref_data(feed->publisher, new_value);
		}
	}
	microfeed_store_iterator_free(iterator);
	
	for (iterator = microfeed_store_iterate(feed->virtual_feed_properties, NULL);
	     (virtual_feed_property = microfeed_store_iterator_get(iterator, VirtualFeedProperty));
	     microfeed_store_iterator_next(iterator)) {
	     	if (virtual_feed_property->property_name) {
			if (old_item && (old_value = microfeed_item_get_property(old_item, virtual_feed_property->property_name))) {
				if (new_item && (new_value = microfeed_item_get_property(new_item, virtual_feed_property->property_name))) {
					if (strcmp(old_value, new_value)) {
						virtual_feed_item_removed(feed, virtual_feed_property->feed_uri_prefix, old_value, old_item);
						virtual_feed_item_added(feed, virtual_feed_property->feed_uri_prefix, new_value, new_item, virtual_feed_property->create);
					} else {
						virtual_feed_item_changed(feed, virtual_feed_property->feed_uri_prefix, old_value, old_item, virtual_feed_property->create);
					}
				} else {
					virtual_feed_item_removed(feed, virtual_feed_property->feed_uri_prefix, old_value, old_item);
				}
			} else if (new_item && (new_value = microfeed_item_get_property(new_item, virtual_feed_property->property_name))) {
				virtual_feed_item_added(feed, virtual_feed_property->feed_uri_prefix, new_value, new_item, virtual_feed_property->create);
			}
		} else {
			if (new_item) {
				if (old_item) {
					virtual_feed_item_changed(feed, virtual_feed_property->feed_uri_prefix, microfeed_item_get_uid(new_item), new_item, virtual_feed_property->create);
				} else {
					virtual_feed_item_added(feed, virtual_feed_property->feed_uri_prefix, microfeed_item_get_uid(new_item), new_item, virtual_feed_property->create);
				}
			} else {
				virtual_feed_item_removed(feed, virtual_feed_property->feed_uri_prefix, microfeed_item_get_uid(old_item), old_item);
			}
		}
	}
	microfeed_store_iterator_free(iterator);
}

static void handle_item_status_change(MicrofeedFeed* feed, MicrofeedItem* item) {
	MicrofeedStoreIterator* iterator;
	VirtualFeedProperty* virtual_feed_property;
	const char* value;
	char* uri;
	MicrofeedFeed* virtual_feed;

	for (iterator = microfeed_store_iterate(feed->virtual_feed_properties, NULL);
	     (virtual_feed_property = microfeed_store_iterator_get(iterator, VirtualFeedProperty));
	     microfeed_store_iterator_next(iterator)) {
	     	if (virtual_feed_property->property_name && (value = microfeed_item_get_property(item, virtual_feed_property->property_name))) {
			uri = microfeed_util_string_concatenate(virtual_feed_property->feed_uri_prefix, value, NULL);
			if (!(virtual_feed = microfeed_store_get(feed->virtual_feeds, uri, MicrofeedFeed))) {
				if ((virtual_feed = microfeed_publisher_get_or_instantiate_feed(feed->publisher, uri, virtual_feed_property->create))) {
					microfeed_store_insert(feed->virtual_feeds, virtual_feed);
				}
			}
printf("visc: %s %p\n", uri, virtual_feed);
			free(uri);

			if (virtual_feed) {

				microfeed_provider_send_status_changed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, virtual_feed->object_path, virtual_feed->uri, microfeed_item_get_uid(item), microfeed_item_get_status(item));

				microfeed_object_unref(virtual_feed, MicrofeedFeed);
			}


		}
	}
	microfeed_store_iterator_free(iterator);

}

static void remove_old_items(MicrofeedFeed* feed) {
	time_t timestamp;
	int keep_old_items;
	
/*	keep_old_items = atoi(microfeed_publisher_get_setting_value(feed->publisher, "keep_old_items", "0"));
	timestamp = time(NULL) + 60 * 60 * keep_old_items;
*/	
	/* TODO! */
	
}

static void send_item_signal(MicrofeedFeed* feed, const char* destination, const char* signal_name, MicrofeedItem* item) {
	MicrofeedItem* duplicate_item;
	
	duplicate_item = microfeed_item_duplicate(item);
	if (feed->master_feed && feed->master_feed->callbacks.complete_item_properties) {
		feed->master_feed->callbacks.complete_item_properties(feed, duplicate_item, feed->master_feed->user_data);
	}
	if (feed->callbacks.complete_item_properties) {
		feed->callbacks.complete_item_properties(feed, duplicate_item, feed->user_data);
	}
	microfeed_provider_send_item_signal(microfeed_publisher_get_provider(feed->publisher), destination, signal_name, feed->object_path, feed->uri, duplicate_item);
	microfeed_item_free(duplicate_item);
}
