/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDOBJECT_H
#define MICROFEEDOBJECT_H

#include <sys/types.h>
#include <microfeed-common/microfeedthread.h>

/**
 * @addtogroup common libmicrofeed-common - Common modules used both in the publisher and in the subscriber side
 * @{
 * @addtogroup MicrofeedObject
 *
 * Basic services for Microfeed objects.
 *
 * @{
 */

/**
 * Opaque data type representing an object implementation.
 */
typedef struct _MicrofeedObjectImpl MicrofeedObjectImpl;

/**
 * A data type representing an object.
 */
typedef struct _MicrofeedObject MicrofeedObject;
struct _MicrofeedObject {
	MicrofeedObjectImpl* object_impl;
	unsigned long int reference_count;
	MicrofeedMutex* mutex;
};

typedef void (*MicrofeedObjectFreeFunction)(MicrofeedObject* object);

typedef struct _MicrofeedClass MicrofeedClass;
struct _MicrofeedClass {
	const char* name;
	MicrofeedObjectFreeFunction free_function;
};

/**
 * Opaque data type representing a weak reference.
 */
typedef struct _MicrofeedWeakReference MicrofeedWeakReference;

MicrofeedObject* microfeed_object_new_impl(MicrofeedClass* class, size_t size, const char* class_name);
#define microfeed_object_new(c, t) ((t*)microfeed_object_new_impl(c, sizeof(t), #t))
MicrofeedObject* microfeed_object_ref_impl(MicrofeedObject* object, const char* class_name);
#define microfeed_object_ref(o, t) ((t*)microfeed_object_ref_impl((MicrofeedObject*)o, #t))
MicrofeedObject* microfeed_object_ref_generic(MicrofeedObject* object);
void microfeed_object_unref_impl(MicrofeedObject* object, const char* class_name);
#define microfeed_object_unref(o, t) (microfeed_object_unref_impl((MicrofeedObject*)o, #t))
void microfeed_object_unref_generic(MicrofeedObject* object);
MicrofeedWeakReference* microfeed_object_get_weak_reference_impl(MicrofeedObject* object, const char* class_name);
#define microfeed_object_get_weak_reference(o, t) (microfeed_object_get_weak_reference_impl((MicrofeedObject*)o, #t))
MicrofeedWeakReference* microfeed_object_get_weak_reference_generic(MicrofeedObject* object);
void microfeed_object_lock_impl(MicrofeedObject* object, const char* class_name);
#define microfeed_object_lock(o, t) (microfeed_object_lock_impl((MicrofeedObject*)o, #t))
void microfeed_object_unlock_impl(MicrofeedObject* object, const char* class_name);
#define microfeed_object_unlock(o, t) (microfeed_object_unlock_impl((MicrofeedObject*)o, #t))

void microfeed_weak_reference_unref(MicrofeedWeakReference* weak_refeference);
MicrofeedObject* microfeed_weak_reference_get_object_impl(MicrofeedWeakReference* weak_reference, const char* class_name);
#define microfeed_weak_reference_get_object(w, t) ((t*)microfeed_weak_reference_get_object_impl(w, #t))
MicrofeedObject* microfeed_weak_reference_get_object_generic(MicrofeedWeakReference* weak_reference);

#define MICROFEED_OBJECT(o) ((MicrofeedObject*)(o))
#endif
