/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-common/microfeedmain.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedthread.h>

#include <sys/time.h>
#include <time.h>
#include <stdio.h>
#include <poll.h>
#include <string.h>
#include <unistd.h>

struct _MicrofeedTimeout {
	MicrofeedMain* main;
	MicrofeedTimeout* next;
	MicrofeedTimeout* previous;
	unsigned long int milliseconds;
	MicrofeedTimeoutCallback callback;
	void* user_data;
	int in_handler : 1;
};

struct _MicrofeedWatch {
	MicrofeedMain* main;
	MicrofeedWatch* next;
	MicrofeedWatch* previous;
	int fd;
	MicrofeedWatchType type;
	struct pollfd* pollfd;
	MicrofeedWatchCallback callback;
	void* user_data;
};

struct _MicrofeedMain {
	MicrofeedMutex* mutex;
	DBusConnection* connection;
	MicrofeedTimeout* timeouts;
	MicrofeedWatch* watches;
	unsigned int watches_count;
	MicrofeedWatch* watch_iterator;
	struct pollfd* poll_fds;
	int polling;
	int wakeup_pipe[2];
	int exit_requested : 1;
	int poll_fds_valid : 1;
};

static void dispatch_status_changed(DBusConnection* connection, DBusDispatchStatus new_status, void* data);
static dbus_bool_t add_timeout(DBusTimeout *timeout, void *data);
static void remove_timeout(DBusTimeout *timeout, void *data);
static void toggle_timeout(DBusTimeout *timeout, void *data);
static void handle_timeout(MicrofeedMain* main, void* user_data);
static dbus_bool_t add_watch(DBusWatch *timeout, void *data);
static void remove_watch(DBusWatch *timeout, void *data);
static void toggle_watch(DBusWatch *timeout, void *data);
static void handle_watch(MicrofeedMain* main, int fd, MicrofeedWatchType type, void* user_data);
static void wakeup_poll(MicrofeedMain* microfeed_main);
static void wakeup_main(void* data);
static void real_remove_timeout(MicrofeedTimeout* timeout);

/**
 * Instantiates a new main loop with a shared session-wide DBus connection.
 * 
 * Note that this function uses dbus_bus_get() to get a DBus connection, and then initializes
 * its callbacks. See #microfeed_main_new_with_dbus_connection for details.
 *
 * This is the preferred way to initialize a main loop.
 * 
 * @return 
 */
MicrofeedMain* microfeed_main_new() {
	DBusError error;
	DBusConnection* connection;

	dbus_threads_init_default();
	dbus_error_init(&error);
	connection = dbus_bus_get(DBUS_BUS_SESSION, &error);
	dbus_connection_set_exit_on_disconnect(connection, FALSE);
	dbus_error_free(&error);
	
	return microfeed_main_new_with_dbus_connection(connection);
}

/**
 * Instantiates a new main loop with a given DBus connection.
 * 
 * This function binds the DBus connection into the main loop with the following functions:
 * dbus_connection_set_dispatch_status_function,
 * dbus_connection_set_timeout_functions,
 * dbus_connection_set_watch_functions, and
 * dbus_connection_set_wakeup_main_function. Do not call those functions in your application.
 * 
 * This function should be only called, if the shared session-wide DBus connection is not used,
 * but an application wants to use, say, system-wide DBus connection.
 * 
 * @param connection A DBus connection to use in the main loop.
 * @return Instantiated MicrofeedMain or NULL if instantiation failed.
 */
MicrofeedMain* microfeed_main_new_with_dbus_connection(DBusConnection* connection) {
	MicrofeedMain* microfeed_main;
	
	microfeed_main = microfeed_memory_allocate(MicrofeedMain);
	microfeed_main->mutex = microfeed_mutex_new();
	microfeed_main->connection = connection;
	if (pipe(microfeed_main->wakeup_pipe)) {
		fprintf(stderr, "ERROR: Pipe failed.\n");
		
		exit(126);
	}
	
	dbus_connection_set_dispatch_status_function(connection, dispatch_status_changed, microfeed_main, NULL);
	dbus_connection_set_timeout_functions(connection, add_timeout, remove_timeout, toggle_timeout, microfeed_main, NULL);
	dbus_connection_set_watch_functions(connection, add_watch, remove_watch, toggle_watch, microfeed_main, NULL);
	dbus_connection_set_wakeup_main_function(connection, wakeup_main, microfeed_main, NULL);

	return microfeed_main;
}

/**
 * Frees the resources allocated for the main loop.
 * 
 * Note that the DBus connection callbacks are not unset. If the DBus connection is still active,
 * a new functions must be set by the application itself.
 * 
 * @param microfeed_main Instantiated MicrofeedMain.
 */
void microfeed_main_free(MicrofeedMain* microfeed_main) {
	MicrofeedTimeout* timeout;
	MicrofeedTimeout* next_timeout;
	MicrofeedWatch* watch;
	MicrofeedWatch* next_watch;

	microfeed_mutex_free(microfeed_main->mutex);
	microfeed_main->mutex = NULL;
	dbus_connection_close(microfeed_main->connection);
	microfeed_main->connection = NULL;
	for (timeout = microfeed_main->timeouts; timeout; timeout = next_timeout) {
		timeout->main = NULL;
		next_timeout = timeout->next;
		microfeed_memory_free(timeout);
	}
	microfeed_main->timeouts = NULL;
	for (watch = microfeed_main->watches; watch; watch = next_watch) {
		watch->main = NULL;
		next_watch = watch->next;
		microfeed_memory_free(watch);
	}
	microfeed_main->watches = NULL;
	microfeed_main->watches_count = 0;
	microfeed_memory_free(microfeed_main->poll_fds);
	microfeed_memory_free(microfeed_main);
}

/**
 * Returns the DBus connection used in the main loop.
 *
 * @param Instantiated MicrofeedMain.
 * @return DBus connection.
 */
DBusConnection* microfeed_main_get_dbus_connection(MicrofeedMain* microfeed_main) {

	return microfeed_main->connection;
}

/**
 * Executes a main loop repeatedly until a #microfeed_main_exit is called.
 * 
 * @param microfeed_main Instantiated MicrofeedMain.
 */
void microfeed_main_loop(MicrofeedMain* microfeed_main) {
	struct timeval timeval;
	int milliseconds;
	MicrofeedWatch* watch;
	struct pollfd* pollfd;
	MicrofeedWatchType type;
	MicrofeedWatchCallback watch_callback;
	void* user_data;
	int fd;
	char buffer[1024];
	ssize_t dummy;
	MicrofeedTimeout* timeout;

	microfeed_mutex_lock(microfeed_main->mutex);
	
	while (!microfeed_main->exit_requested) {
		microfeed_mutex_unlock(microfeed_main->mutex);
				
		while (dbus_connection_get_dispatch_status(microfeed_main->connection) == DBUS_DISPATCH_DATA_REMAINS) {
			dbus_connection_dispatch(microfeed_main->connection);
		}
		if (dbus_connection_has_messages_to_send(microfeed_main->connection)) {
			dbus_connection_flush(microfeed_main->connection);
		}

		microfeed_mutex_lock(microfeed_main->mutex);

		if (microfeed_main->timeouts) {
			gettimeofday(&timeval, NULL);
			milliseconds = microfeed_main->timeouts->milliseconds - timeval.tv_sec * 1000 - timeval.tv_usec / 1000;
			if (milliseconds < 0) {
				milliseconds = 0;
			}
		} else {
			milliseconds = -1;
		}
		
		if (!microfeed_main->poll_fds_valid) {
			if (microfeed_main->poll_fds) {
				microfeed_memory_free(microfeed_main->poll_fds);
			}
			microfeed_main->poll_fds = pollfd = microfeed_memory_allocate_bytes((microfeed_main->watches_count + 1) * sizeof(struct pollfd));
			pollfd->fd = microfeed_main->wakeup_pipe[0];
			pollfd->events = POLLIN;
			for (watch = microfeed_main->watches, pollfd++; watch; watch = watch->next, pollfd++) {
				watch->pollfd = pollfd;
				pollfd->fd = watch->fd;
				if (watch->type == MICROFEED_WATCH_TYPE_READ_WRITE) {
					pollfd->events = POLLIN | POLLOUT;
				} else if (watch->type == MICROFEED_WATCH_TYPE_WRITE) {
					pollfd->events = POLLOUT;
				} else {
					pollfd->events = POLLIN;
				}
			}
			microfeed_main->poll_fds_valid = 1;
		}

		if (microfeed_main->exit_requested) {
			break;
		}

		microfeed_main->polling = 1;
		
		microfeed_mutex_unlock(microfeed_main->mutex);
		
		poll(microfeed_main->poll_fds, microfeed_main->watches_count + 1, milliseconds);
		
		microfeed_mutex_lock(microfeed_main->mutex);
		
		microfeed_main->polling = 0;

		gettimeofday(&timeval, NULL);
		milliseconds = timeval.tv_sec * 1000 + timeval.tv_usec / 1000;
		if (microfeed_main->timeouts && microfeed_main->timeouts->milliseconds < milliseconds) {
			timeout = microfeed_main->timeouts;
			timeout->in_handler = 1;

			microfeed_mutex_unlock(microfeed_main->mutex);

			timeout->callback(microfeed_main, timeout->user_data);

			microfeed_mutex_lock(microfeed_main->mutex);

			real_remove_timeout(timeout);		
		}

		if (microfeed_main->poll_fds->revents & POLLIN) {
			dummy = read(microfeed_main->wakeup_pipe[0], buffer, 1024);
		}
		
		if (microfeed_main->poll_fds_valid) {
			for (watch = microfeed_main->watches; watch; watch = microfeed_main->watch_iterator) {
				microfeed_main->watch_iterator = watch->next;
				if ((pollfd = watch->pollfd)) {
					if (pollfd->revents & (POLLIN | POLLOUT)) {
						type = MICROFEED_WATCH_TYPE_READ_WRITE;
					} else if (pollfd->revents & POLLOUT) {
						type = MICROFEED_WATCH_TYPE_WRITE;
					} else if (pollfd->revents & POLLIN) {
						type = MICROFEED_WATCH_TYPE_READ;
					} else {
						type = MICROFEED_WATCH_TYPE_NONE;
					}
					if (type != MICROFEED_WATCH_TYPE_NONE) {
						watch_callback = watch->callback;
						fd = watch->fd;
						user_data = watch->user_data;
								
						microfeed_mutex_unlock(microfeed_main->mutex);
				
						watch_callback(microfeed_main, fd, type, user_data);

						microfeed_mutex_unlock(microfeed_main->mutex);
					}
				}
			}
		}
	}
	
	microfeed_main->exit_requested = 0;
	
	microfeed_mutex_unlock(microfeed_main->mutex);
}

/**
 * Asks the main loop to stop.
 * 
 * @param microfeed_main Instantiated MicrofeedMain.
 */
void microfeed_main_exit(MicrofeedMain* microfeed_main) {
	microfeed_mutex_lock(microfeed_main->mutex);

	microfeed_main->exit_requested = 1;
	wakeup_poll(microfeed_main);

	microfeed_mutex_unlock(microfeed_main->mutex);	
}

/**
 * Adds a new timeout into the main loop.
 * 
 * The callback is called after the specified time - not before - with the given user data.
 * 
 * @param microfeed_main Instantiated MicrofeedMain.
 * @param milliseconds A specified time in milliseconds.
 * @param callback A function to be called after the specified time.
 * @param user_data A pointer to user data that is given as a parameter when the callback function is called.
 * @return Added timeout.
 */ 
MicrofeedTimeout* microfeed_main_add_timeout(MicrofeedMain* microfeed_main, unsigned long int milliseconds, MicrofeedTimeoutCallback callback, void* user_data) {
	MicrofeedTimeout* timeout;
	struct timeval timeval;
	MicrofeedTimeout* current;
	MicrofeedTimeout* previous;

	microfeed_mutex_lock(microfeed_main->mutex);

	gettimeofday(&timeval, NULL);	
	milliseconds += timeval.tv_sec * 1000 + timeval.tv_usec / 1000;
	
	timeout = microfeed_memory_allocate(MicrofeedTimeout);
	timeout->main = microfeed_main;
	timeout->milliseconds = milliseconds;
	timeout->callback = callback;
	timeout->user_data = user_data;

	for (previous = NULL, current = microfeed_main->timeouts; current; previous = current, current = current->next) {
		if (milliseconds < current->milliseconds) {
			timeout->previous = current->previous;
			if (current->previous) {
				current->previous->next = timeout;
			} else {
				microfeed_main->timeouts = timeout;
			}
			current->previous = timeout;
			timeout->next = current;				
			break;
		}
	}
	if (!current) {
		if (previous) {
			timeout->previous = previous;
			previous->next = timeout;
		} else {
			timeout->previous = NULL;
			microfeed_main->timeouts = timeout;
		}
		timeout->next = NULL;
	}
	
	wakeup_poll(microfeed_main);

	microfeed_mutex_unlock(microfeed_main->mutex);

	return timeout;
}

/**
 * Adds a new file descriptor to watch into the main loop.
 * 
 * The main loop will monitor the given file if it is ready to be read, written or both, and call the
 * given function with the given user data when the file is ready.
 * 
 * @param microfeed_main Instantiated MicrofeedMain.
 * @param fd an Unix file descriptor.
 * @param type The event that should be monitored.
 * @param callback A fucntion to be called when the file is ready.
 * @param user_data A pointer to user data that is given as a parameter when the callback function is called.
 */
MicrofeedWatch* microfeed_main_add_watch(MicrofeedMain* microfeed_main, int fd, MicrofeedWatchType type, MicrofeedWatchCallback callback, void* user_data) {
	MicrofeedWatch* watch;

	microfeed_mutex_lock(microfeed_main->mutex);

	watch = microfeed_memory_allocate(MicrofeedWatch);
	watch->main = microfeed_main;
	watch->fd = fd;
	watch->type = type;
	watch->pollfd = NULL;
	watch->callback = callback;
	watch->user_data = user_data;

	watch->next = microfeed_main->watches;
	if (watch->next) {
		watch->next->previous = watch;
	}
	microfeed_main->watches = watch;
	microfeed_main->watches_count++;

	microfeed_main->poll_fds_valid = 0;
	wakeup_poll(microfeed_main);

	microfeed_mutex_unlock(microfeed_main->mutex);

	return watch;
}

/**
 * Removes a previously added timeout from the main loop.
 * 
 * @param microfeed_main Instantiated MicrofeedMain.
 * @param timeout A timeout previously added but not yet removed.
 */
void microfeed_main_remove_timeout(MicrofeedMain* microfeed_main, MicrofeedTimeout* timeout) {
	microfeed_mutex_lock(microfeed_main->mutex);

	if (microfeed_main == timeout->main && !timeout->in_handler) {
		real_remove_timeout(timeout);
	}

	microfeed_mutex_unlock(microfeed_main->mutex);
}

/**
 * Removes a previously added file descriptor watch from the main loop.
 * 
 * @param microfeed_main Instantiated MicrofeedMain.
 * @param watch A watch previously added but not yet removed.
 */
void microfeed_main_remove_watch(MicrofeedMain* microfeed_main, MicrofeedWatch* watch) {	
	microfeed_mutex_lock(microfeed_main->mutex);

	if (microfeed_main == watch->main) {
		if (microfeed_main->watch_iterator == watch) {
			microfeed_main->watch_iterator = watch->next;
		}
	
		if (watch->previous) {
			watch->previous->next = watch->next;
		} else if (watch->main) {
			watch->main->watches = watch->next;
		}
		if (watch->next) {
			watch->next->previous = watch->previous;
		}
		microfeed_memory_free(watch);	
		microfeed_main->watches_count--;
	
		microfeed_main->poll_fds_valid = 0;
		wakeup_poll(microfeed_main);
	}

	microfeed_mutex_unlock(microfeed_main->mutex);
}

static void dispatch_status_changed(DBusConnection* connection, DBusDispatchStatus new_status, void* data) {
	MicrofeedMain* microfeed_main;
	
	if (new_status == DBUS_DISPATCH_DATA_REMAINS) {
		microfeed_main = (MicrofeedMain*)data;
		wakeup_poll(microfeed_main);
	}
}

static dbus_bool_t add_timeout(DBusTimeout* timeout, void* data) {
	MicrofeedMain* microfeed_main;
	MicrofeedTimeout* microfeed_timeout;
	
	if (dbus_timeout_get_enabled(timeout) && !dbus_timeout_get_data(timeout)) {
		microfeed_main = (MicrofeedMain*)data;
		microfeed_timeout = microfeed_main_add_timeout(microfeed_main, dbus_timeout_get_interval(timeout), handle_timeout, timeout);
		dbus_timeout_set_data(timeout, microfeed_timeout, NULL);
	}

	return TRUE;
}

static void remove_timeout(DBusTimeout* timeout, void* data) {
	MicrofeedMain* microfeed_main;
	MicrofeedTimeout* microfeed_timeout;
	
	microfeed_main = (MicrofeedMain*)data;
	if ((microfeed_timeout = (MicrofeedTimeout*)dbus_timeout_get_data(timeout))) {
		microfeed_main_remove_timeout(microfeed_main, microfeed_timeout);
		dbus_timeout_set_data(timeout, NULL, NULL);
	}
}

static void toggle_timeout(DBusTimeout* timeout, void* data) {
	if (dbus_timeout_get_enabled(timeout)) {
		add_timeout(timeout, data);
	} else {
		remove_timeout(timeout, data);
	}
}

static void handle_timeout(MicrofeedMain* microfeed_main, void* user_data) {
	DBusTimeout* timeout;
	
	timeout = (DBusTimeout*)user_data;
	if (dbus_timeout_get_enabled(timeout)) {
		dbus_timeout_handle(timeout);
		microfeed_main_add_timeout(microfeed_main, dbus_timeout_get_interval(timeout), handle_timeout, timeout);
	}
}

static dbus_bool_t add_watch(DBusWatch *watch, void *data) {
	MicrofeedMain* microfeed_main;
	int fd;
	MicrofeedWatch* microfeed_watch;
	unsigned int flags;
	MicrofeedWatchType type;
	
	if (dbus_watch_get_enabled(watch) && !dbus_watch_get_data(watch)) {
		microfeed_main = (MicrofeedMain*)data;
		flags = dbus_watch_get_flags(watch);
		if (flags & DBUS_WATCH_READABLE && flags & DBUS_WATCH_WRITABLE) {
			type = MICROFEED_WATCH_TYPE_READ_WRITE;
		} else if (flags & DBUS_WATCH_WRITABLE) {
			type = MICROFEED_WATCH_TYPE_WRITE;
		} else {
			type = MICROFEED_WATCH_TYPE_READ;
		}
#if (DBUS_VERSION_MAJOR == 1 && DBUS_VERSION_MINOR == 1 && DBUS_VERSION_MICRO >= 1) || (DBUS_VERSION_MAJOR == 1 && DBUS_VERSION_MAJOR > 1) || (DBUS_VERSION_MAJOR > 1) 
		fd = dbus_watch_get_unix_fd(watch); 
#else
		fd = dbus_watch_get_fd(watch); 
#endif 
		microfeed_watch = microfeed_main_add_watch(microfeed_main, fd, type, handle_watch, watch);
		dbus_watch_set_data(watch, microfeed_watch, NULL);
	}

	return TRUE;
}

static void remove_watch(DBusWatch *watch, void *data) {
	MicrofeedMain* microfeed_main;
	MicrofeedWatch* microfeed_watch;
	
	microfeed_main = (MicrofeedMain*)data;
	if ((microfeed_watch = (MicrofeedWatch*)dbus_watch_get_data(watch))) {
		microfeed_main_remove_watch(microfeed_main, microfeed_watch);
		dbus_watch_set_data(watch, NULL, NULL);
	}
}

static void toggle_watch(DBusWatch *watch, void *data) {
	if (dbus_watch_get_enabled(watch)) {
		add_watch(watch, data);
	} else {
		remove_watch(watch, data);
	}
}

static void handle_watch(MicrofeedMain* main, int fd, MicrofeedWatchType type, void* user_data) {
	DBusWatch* watch;
	unsigned int flags;
	
	watch = (DBusWatch*)user_data;
	if (dbus_watch_get_enabled(watch)) {
		if (type == MICROFEED_WATCH_TYPE_READ_WRITE) {
			flags = DBUS_WATCH_READABLE | DBUS_WATCH_WRITABLE;
		} else if (type == MICROFEED_WATCH_TYPE_WRITE) {
			flags = DBUS_WATCH_WRITABLE;
		} else {
			flags = DBUS_WATCH_READABLE;
		}
		dbus_watch_handle(watch, flags);
	}
}

static void wakeup_poll(MicrofeedMain* microfeed_main) {
	static char c = 0;
	ssize_t dummy;

	if (microfeed_main->polling) {
		dummy = write(microfeed_main->wakeup_pipe[1], &c, 1);
		c++;
	}
}

static void wakeup_main(void* data) {
	MicrofeedMain* microfeed_main;
	
	microfeed_main = (MicrofeedMain*)data;
	
	microfeed_mutex_lock(microfeed_main->mutex);
	
	wakeup_poll(microfeed_main);
	
	microfeed_mutex_unlock(microfeed_main->mutex);
}

void real_remove_timeout(MicrofeedTimeout* timeout) {
	if (timeout->previous) {
		timeout->previous->next = timeout->next;
	} else if (timeout->main) {
		timeout->main->timeouts = timeout->next;
	}
	if (timeout->next) {
		timeout->next->previous = timeout->previous;
	}
	microfeed_memory_free(timeout);	
}
