/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDFILEINDEXTABLE_H
#define MICROFEEDFILEINDEXTABLE_H

#include <microfeed-common/microfeedfile.h>
#include <time.h>

/**
 * @addtogroup common libmicrofeed-common - Common modules used both in the publisher and in the subscriber side
 * @{
 * @addtogroup MicrofeedFileIndexTable
 *
 * @{
 */

/**
 * Opaque data type representing a file index index table.
 */
typedef struct _MicrofeedFileIndexTable MicrofeedFileIndexTable;

typedef struct _MicrofeedFileIndexTableIterator MicrofeedFileIndexTableIterator;

typedef int (*MicrofeedFileIndexTableCompareKeysFunction)(const void* key1, size_t key1_size, const void* key2, size_t key2_size);
typedef void (*MicrofeedFileIndexTableGetKeyFunction)(const void* block, size_t block_size, const void** key_return, size_t* key_size_return);

MicrofeedFileIndexTable* microfeed_file_index_table_new_sorted(MicrofeedFile* file, MicrofeedFileIndex index, int create, MicrofeedFileIndexTableCompareKeysFunction compare_keys, MicrofeedFileIndexTableGetKeyFunction get_key);
MicrofeedFileIndexTable* microfeed_file_index_table_new_sorted_with_delta(MicrofeedFile* file, MicrofeedFileIndex index, int create, unsigned int delta, MicrofeedFileIndexTableCompareKeysFunction compare_keys, MicrofeedFileIndexTableGetKeyFunction get_key);
MicrofeedFileIndexTable* microfeed_file_index_table_new_named_indices(MicrofeedFile* file, MicrofeedFileIndex index, int create);
void microfeed_file_index_table_free(MicrofeedFileIndexTable* table);

MicrofeedFileIndex microfeed_file_index_table_get_file_index(MicrofeedFileIndexTable* table);
int microfeed_file_index_table_insert_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index);
int microfeed_file_index_table_insert_block(MicrofeedFileIndexTable* table, void* block);
MicrofeedFileIndex microfeed_file_index_table_replace_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index);
#define microfeed_file_index_table_replace_block(f,b,t) ((t*)microfeed_file_index_table_replace_block_impl((f), (b)))
void* microfeed_file_index_table_replace_block_impl(MicrofeedFileIndexTable* table, void* block);
MicrofeedFileIndex microfeed_file_index_table_update_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index, const void* new_data, size_t new_data_size);
#define microfeed_file_index_table_update_block(f,b,t,d,s) ((t*)microfeed_file_index_table_replace_block_impl((f), (b), (d), (s)))
void* microfeed_file_index_table_update_block_impl(MicrofeedFileIndexTable* table, void* block, const void* new_data, size_t new_data_size);
int microfeed_file_index_table_remove_index(MicrofeedFileIndexTable* table, MicrofeedFileIndex index);
int microfeed_file_index_table_remove_block(MicrofeedFileIndexTable* table, void* block);
MicrofeedFileIndex microfeed_file_index_table_remove_key(MicrofeedFileIndexTable* table, const void* key, size_t key_size);
#define microfeed_file_index_table_get_block(f,k,s,t) ((t*)microfeed_file_index_table_get_block_impl((f), (k), (s)))
void* microfeed_file_index_table_get_block_impl(MicrofeedFileIndexTable* table, const void* key, size_t key_size);
#define microfeed_file_index_table_get_existing_block(f,k,s,t) ((t*)microfeed_file_index_table_get_existing_block_impl((f), (k), (s)))
MicrofeedFileIndex microfeed_file_index_table_get_index(MicrofeedFileIndexTable* table, const void* key, size_t key_size);
MicrofeedFileIndexTableIterator* microfeed_file_index_table_iterate(MicrofeedFileIndexTable* table, const void* start_key, size_t start_key_size, int backwards);
void microfeed_file_index_table_get_block_key(MicrofeedFileIndexTable* table, const void* block, size_t block_size, const void** key_return, size_t* key_size_return);
int microfeed_file_index_table_compare_keys(MicrofeedFileIndexTable* table, const void* key1, size_t key1_size, const void* key2, size_t key2_size);
time_t microfeed_file_index_table_get_last_updated_time(MicrofeedFileIndexTable* table);
void microfeed_file_index_table_set_last_updated_time(MicrofeedFileIndexTable* table, time_t timestamp);
int microfeed_file_index_table_get_size(MicrofeedFileIndexTable* table);

int microfeed_file_index_table_insert_named_index(MicrofeedFileIndexTable* table, const char* name, MicrofeedFileIndex index);
int microfeed_file_index_table_remove_named_index(MicrofeedFileIndexTable* table, const char* name);
MicrofeedFileIndex microfeed_file_index_table_get_named_index(MicrofeedFileIndexTable* table, const char* name);

void microfeed_file_index_table_iterator_free(MicrofeedFileIndexTableIterator* iterator);
MicrofeedFileIndexTable* microfeed_file_index_table_iterator_get_table(MicrofeedFileIndexTableIterator* iterator);
#define microfeed_file_index_table_iterator_get_block(i,t) ((t*)microfeed_file_index_table_iterator_get_block_impl(i))
void* microfeed_file_index_table_iterator_get_block_impl(MicrofeedFileIndexTableIterator* iterator);
MicrofeedFileIndex microfeed_file_index_table_iterator_get_index(MicrofeedFileIndexTableIterator* iterator);
void microfeed_file_index_table_iterator_next(MicrofeedFileIndexTableIterator* iterator);

int microfeed_file_index_table_compare_keys_direct(const void* key1, size_t key1_size, const void* key2, size_t key2_size);
void microfeed_file_index_table_get_key_direct(const void* block, size_t block_size, const void** key_return, size_t* key_size_return);

/**
 * @}
 * @}
 */

#endif
