#include <math.h>
#include <SDL_thread.h>
#include "sge_core.h"
#include "graphics.h"
#include "board.h"
#include "sound.h"
#include "gweled.h"

#define BOARD_WIDTH 8
#define BOARD_HEIGHT 8
#define TILE_WIDTH 54
#define TILE_HEIGHT 54
gint	game_timer_mode;
#define TIMER_MODE game_timer_mode

void gweled_swap_gems (gint x1, gint y1, gint x2, gint y2);
static T_SGEObject *gweled_draw_character (int x, int y, int layer, char character);
static gboolean gweled_check_for_alignments (void);

gint gpc_game_board[BOARD_WIDTH][BOARD_HEIGHT];
T_SGEObject *g_gem_objects[BOARD_WIDTH][BOARD_HEIGHT];
gchar gi_nb_of_tiles[7];



#define FIRST_BONUS_AT	100	// needs tweaking
#define NB_BONUS_GEMS	8	// same
#define TOTAL_STEPS_FOR_TIMER	60	// seconds

void gweled_remove_gems_and_update_score (void);

enum {
	_IDLE,
	_FIRST_GEM_CLICKED,
	_SECOND_GEM_CLICKED,
	_ILLEGAL_MOVE,
	_MARK_ALIGNED_GEMS,
	_BOARD_REFILLING
};

typedef struct s_alignment {
	gint x;
	gint y;
	gint direction;
	gint length;
} T_Alignment;

SDL_mutex 	*board_mutex = NULL;

void board_lock(void)
{
	SDL_mutexP(board_mutex);
//	pthread_mutex_lock(&board_mutex);
}

void board_unlock(void)
{
	SDL_mutexV(board_mutex);
//	pthread_mutex_unlock(&board_mutex);
}

gint gi_score, gi_current_score, gi_game_running, gi_high_score = 500, gi_notime_high_score = 500;

gint gi_total_gems_removed;
gint gi_gems_removed_per_move;
gint gi_bonus_multiply;
gint gi_previous_bonus_at;
gint gi_next_bonus_at;
gint gi_trigger_bonus;
guint g_steps_for_timer;

gint gi_gem_clicked = 0;
gint gi_x_click = 0;
gint gi_y_click = 0;

gint gi_gem_dragged = 0;
gint gi_x_drag = 0;
gint gi_y_drag = 0;

gboolean g_do_not_score;

unsigned char gpc_bit_n[8] = { 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80 };

static gint gi_state = _IDLE;
static GList *g_alignment_list = NULL;
static int gi_dragging = 0;

void gweled_click(int x, int y)
{
	board_lock();
	if (!gi_game_running)
		goto ret;
	if (x < BOARD_X	|| y < BOARD_Y)
		goto ret;
	x = (x - BOARD_X) / TILE_WIDTH;
	y = (y - BOARD_Y) / TILE_HEIGHT;
	if (x < 0 || x >= BOARD_WIDTH || y<0 || y>=BOARD_HEIGHT)
		goto ret;
	
	if (sge_object_is_moving (g_gem_objects[x][y]))
		goto ret;
		
	gi_x_click = x;
	gi_y_click = y;
	gi_gem_clicked = -1;
	gi_dragging = -1;
ret:	
	board_unlock();
}

void gweled_release(int x, int y)
{
	board_lock();
	if (!gi_game_running)
		goto ret;
	if (x < BOARD_X	|| y < BOARD_Y)
		goto ret;
	if (!gi_dragging)
		goto ret;	
	gi_dragging = 0;
	gi_gem_dragged = 0;		
	x = (x - BOARD_X) / TILE_WIDTH;
	y = (y - BOARD_Y) / TILE_HEIGHT;
	if (x < 0 || x >= BOARD_WIDTH || y<0 || y>=BOARD_HEIGHT)
		goto ret;
	if (sge_object_is_moving (g_gem_objects[x][y]))
		goto ret;
	if( (gi_x_click != x) ||(gi_y_click != y)) {
		gi_x_click = x;
		gi_y_click = y;
		gi_gem_clicked = -1;
	}
ret:
	board_unlock();	
}

int gweled_gems_fall_into_place (void)
{
	gint i, j;
	for (i = 0; i < BOARD_WIDTH; i++)
		for (j = 0; j < BOARD_HEIGHT; j++)
			sge_object_fall_to (g_gem_objects[i][j],
					    j * TILE_HEIGHT);
	return 0;
}

void gweled_draw_board (void)
{
	gint i, j;
	for (i = 0; i < BOARD_WIDTH; i++)
		for (j = 0; j < BOARD_HEIGHT; j++)
			sge_create_object (i * TILE_WIDTH,
					   j * TILE_HEIGHT, 0,
					   gi_tiles_bg_pixbuf[(i + j) % 2]);
}


static gchar get_new_tile (void)
{
	int i;
	int min, max, min_index, max_index, previous_min_index;

	min_index = 0;
	previous_min_index = 0;
	max_index = 0;
	min = gi_nb_of_tiles[0];
	max = gi_nb_of_tiles[0];
	for (i = 0; i < 7; i++) {
		if (gi_nb_of_tiles[i] < min) {
			min = gi_nb_of_tiles[i];
			min_index = i;
			previous_min_index = min_index;
		}
		if (gi_nb_of_tiles[i] > max) {
			max = gi_nb_of_tiles[i];
			max_index = i;
		}
	}

	i = (gint) g_rand_int_range (g_random_generator, 0, 2);

	switch (i) {
	case 0:
		return g_rand_int_range (g_random_generator, 0, 2) ? min_index : previous_min_index;
	default:
		return (max_index + (gchar) g_rand_int_range (g_random_generator, 1, 7)) % 7;
	}
}

void gweled_save_game(void)
{
	FILE *f = fopen(SAVE_PATH, "w");
	if (!f)
		return;
	board_lock();	
	if (fwrite(gpc_game_board, sizeof(gint), BOARD_WIDTH * BOARD_HEIGHT, f) !=
		BOARD_WIDTH * BOARD_HEIGHT)
		goto err;
	if (fwrite(&gi_score, sizeof(gint), 1, f) != 1)
		goto err;
	if (fwrite(&gi_total_gems_removed, sizeof(gint), 1, f) != 1)
		goto err;
	if (fwrite(&gi_bonus_multiply, sizeof(gint), 1, f) != 1)
		goto err;
	if (fwrite(&gi_previous_bonus_at, sizeof(gint), 1, f) != 1)
		goto err;
	if (fwrite(&gi_next_bonus_at, sizeof(gint), 1, f) != 1)
		goto err;
	if (fwrite(&g_steps_for_timer, sizeof(gint), 1, f) != 1)
		goto err;
	if (fwrite(&game_timer_mode, sizeof(game_timer_mode), 1, f) != 1)
		goto err;
err:		
	board_unlock();
	fclose(f);
	return;
}

int gweled_load_game(void)
{
	FILE *f = fopen(SAVE_PATH, "r");
	if (!f)
		return -1;
	if (fread(gpc_game_board, sizeof(gint), 
		BOARD_WIDTH * BOARD_HEIGHT,f) != (BOARD_WIDTH * BOARD_HEIGHT))
		goto err;
	if (fread(&gi_score, sizeof(gint), 1, f) != 1)
		goto err;
	if (fread(&gi_total_gems_removed, sizeof(gint), 1, f) != 1)
		goto err;
	if (fread(&gi_bonus_multiply, sizeof(gint), 1, f) != 1)
		goto err;
	if (fread(&gi_previous_bonus_at, sizeof(gint), 1, f) != 1)
		goto err;
	if (fread(&gi_next_bonus_at, sizeof(gint), 1, f) != 1)
		goto err;
	if (fread(&g_steps_for_timer, sizeof(gint), 1, f) != 1)
		goto err;
	if (fread(&game_timer_mode, sizeof(game_timer_mode), 1, f) != 1)
		goto err;
	gi_current_score = gi_score;
	gi_trigger_bonus = 0;
	fclose(f);
	return 0;
err:
	fclose(f);
	return -1;	
}

void gweled_start_new_game (int load)
{
	int i,j;
	board_lock();
	sge_destroy_all_objects ();
	gweled_draw_board ();
	
	if (load) 
		load = !gweled_load_game();

	memset (gi_nb_of_tiles, 0, 7 * sizeof (gchar));
	
	if (!load) {
		game_timer_mode = g_timer_mode;
		gi_score = 0;
		gi_current_score = 0;
		gi_gems_removed_per_move = 0;
		gi_bonus_multiply = 3;
		gi_previous_bonus_at = 0;
		gi_next_bonus_at = FIRST_BONUS_AT;
		gi_trigger_bonus = 0;
		g_steps_for_timer = FIRST_BONUS_AT / TOTAL_STEPS_FOR_TIMER;

		if (TIMER_MODE)
			gi_total_gems_removed = FIRST_BONUS_AT / 2;
		else
			gi_total_gems_removed = 0;
		for (i = 0; i < BOARD_WIDTH; i++)
			for (j = 0; j < BOARD_HEIGHT; j++)
				gpc_game_board[i][j] = get_new_tile ();
	}

	for (i = 0; i < BOARD_WIDTH; i++)
		for (j = 0; j < BOARD_HEIGHT; j++) {
			gi_nb_of_tiles[gpc_game_board[i][j]]++;
			g_gem_objects[i][j] =
			    sge_create_object (i * TILE_WIDTH,
							(j - BOARD_HEIGHT) * TILE_HEIGHT,
							1,
							gi_gems_pixbuf[gpc_game_board[i][j]]);
		}

	g_do_not_score = TRUE;
	while(gweled_check_for_alignments ()) {
		gweled_remove_gems_and_update_score ();
		gweled_refill_board();
	}
	g_do_not_score = FALSE;
		
	for (i = 0; i < BOARD_WIDTH; i++)
		for (j = 0; j < BOARD_HEIGHT; j++)
			g_gem_objects[i][j] = sge_create_object (i * TILE_WIDTH, (j - BOARD_HEIGHT) * TILE_HEIGHT, 1,
													 gi_gems_pixbuf[gpc_game_board[i][j]]);

	gweled_gems_fall_into_place ();
	gi_game_running = -1;
	gi_state = _MARK_ALIGNED_GEMS;
	display_current_score();
	display_hi_score();
	display_multiply();
	display_time((float)(gi_total_gems_removed -gi_previous_bonus_at)
		/ (float)(gi_next_bonus_at - gi_previous_bonus_at));
	board_unlock();	
}

void gweled_refill_board (void)
{
	gint i, j, k;
	for (i = 0; i < BOARD_WIDTH; i++)
		for (j = 0; j < BOARD_HEIGHT; j++)
			if (gpc_game_board[i][j] == -1) {
				for (k = j; k > 0; k--) {
					gpc_game_board[i][k] =
					    gpc_game_board[i][k - 1];
					g_gem_objects[i][k] =
					    g_gem_objects[i][k - 1];
				}
				gpc_game_board[i][0] = get_new_tile ();
				gi_nb_of_tiles[gpc_game_board[i][0]]++;

				// make sure the new tile appears outside of the screen (1st row is special-cased)
				if (j && g_gem_objects[i][1])
					g_gem_objects[i][0] = sge_create_object (i * TILE_WIDTH,
											g_gem_objects[i][1]->y - TILE_HEIGHT,
											1,
											gi_gems_pixbuf[gpc_game_board[i][0]]);
				else
					g_gem_objects[i][0] = sge_create_object (i * TILE_WIDTH,
											-TILE_HEIGHT,
											1,
											gi_gems_pixbuf[gpc_game_board[i][0]]);
			}
}


gint gweled_is_part_of_an_alignment (gint x, gint y)
{
	gint i, result;
	result = 0;
	for (i = x - 2; i <= x; i++)
		if (i >= 0 && i + 2 < BOARD_WIDTH)
			if (gpc_bit_n[gpc_game_board[i][y]] &
			     gpc_bit_n[gpc_game_board[i + 1][y]] &
			     gpc_bit_n[gpc_game_board[i + 2][y]]) {
				result |= 1;	// is part of an horizontal alignment
				break;
			}

	for (i = y - 2; i <= y; i++)
		if (i >= 0 && i + 2 < BOARD_HEIGHT)
		if (gpc_bit_n[gpc_game_board[x][i]] &
		     gpc_bit_n[gpc_game_board[x][i + 1]] &
		     gpc_bit_n[gpc_game_board[x][i + 2]]) {
				result |= 2;	// is part of a vertical alignment
				break;
		}
		return result;
}

static gboolean gweled_check_for_moves_left (int *pi, int *pj) 
{
	gint i, j;
		for (j = BOARD_HEIGHT - 1; j >= 0; j--)
		for (i = BOARD_WIDTH - 1; i >= 0; i--) {
			if (i > 0) {
				gweled_swap_gems (i - 1, j, i, j);
				if (gweled_is_part_of_an_alignment (i, j)) {
					gweled_swap_gems (i - 1, j, i, j);
					goto move_found;
				}
				gweled_swap_gems (i - 1, j, i, j);
			}
			if (i < 7) {
				gweled_swap_gems (i + 1, j, i, j);
				if (gweled_is_part_of_an_alignment (i, j)) {
					gweled_swap_gems (i + 1, j, i, j);
					goto move_found;
				}
				gweled_swap_gems (i + 1, j, i, j);
			}
			if (j > 0) {
				gweled_swap_gems (i, j - 1, i, j);
				if (gweled_is_part_of_an_alignment (i, j)) {
					gweled_swap_gems (i, j - 1, i, j);
					goto move_found;
				}
				gweled_swap_gems (i, j - 1, i, j);
			}
			if (j < 7) {
				gweled_swap_gems (i, j + 1, i, j);
				if (gweled_is_part_of_an_alignment (i, j)) {
					gweled_swap_gems (i, j + 1, i, j);
					goto move_found;
				}
				gweled_swap_gems (i, j + 1, i, j);
			}
		}		return FALSE;
      move_found:
	if (pi && pj) {
		*pi = i;
		*pj = j;
	}
	return TRUE;
}
void
gweled_swap_gems (gint x1, gint y1, gint x2, gint y2) 
{
	gint i;
	T_SGEObject * object;

	object = g_gem_objects[x1][y1];
	g_gem_objects[x1][y1] = g_gem_objects[x2][y2];
	g_gem_objects[x2][y2] = object;
	i = gpc_game_board[x1][y1];
	gpc_game_board[x1][y1] = gpc_game_board[x2][y2];
	gpc_game_board[x2][y2] = i;
} 

void
delete_alignment_from_board (gpointer alignment_pointer, gpointer user_data)
{
	gint i, i_total_score;
	int xsize, ysize, xhotspot, yhotspot, xpos, ypos;
	char *buffer;
	T_Alignment *alignment;
	T_SGEObject *object;

	alignment = (T_Alignment *) alignment_pointer;
// delete alignment
	if (alignment->direction == 1)	// horizontal
	{
		xhotspot = (alignment->x * TILE_WIDTH + alignment->length * TILE_WIDTH / 2);
		yhotspot = (alignment->y * TILE_HEIGHT + TILE_HEIGHT / 2);
		for (i = alignment->x; i < alignment->x + alignment->length; i++) {
			if (gpc_game_board[i][alignment->y] != -1) {
				gi_gems_removed_per_move++;
				gi_nb_of_tiles[gpc_game_board[i][alignment->y]]--;
				gpc_game_board[i][alignment->y] = -1;
			}
		}
	} else {
		xhotspot = (alignment->x * TILE_WIDTH + TILE_WIDTH / 2);
		yhotspot = (alignment->y * TILE_HEIGHT + alignment->length * TILE_HEIGHT / 2);
		for (i = alignment->y; i < alignment->y + alignment->length; i++) {
			if (gpc_game_board[alignment->x][i] != -1) {
				gi_gems_removed_per_move++;
				gi_nb_of_tiles[gpc_game_board[alignment->x][i]]--;
				gpc_game_board[alignment->x][i] = -1;
			}
		}
	}
//compute score
	if (alignment->length == 1)	//bonus mode
		i_total_score = 10 * g_rand_int_range (g_random_generator, 1, 2);
	else
		i_total_score = 10 * (gi_gems_removed_per_move - 2) * (gi_bonus_multiply >> 1);
	if (g_do_not_score == FALSE) {
		gi_score += i_total_score;
		if (game_timer_mode) {
			if (gi_score > gi_high_score) {
				gi_high_score = gi_score;
				display_hi_score();
			}
		} else {
			if (gi_score > gi_notime_high_score) {
				gi_notime_high_score = gi_score;
				display_hi_score();
			}
		}
//display score
#if 1
		buffer = g_strdup_printf ("%d", i_total_score);
		xsize = strlen (buffer) * SMALL_FONT_WIDTH;
		ysize = SMALL_FONT_HEIGHT;
		for (i = 0; i < strlen (buffer); i++) {
			xpos = xhotspot - xsize / 2 + i * SMALL_FONT_WIDTH;
			ypos = yhotspot - ysize / 2;
			object = gweled_draw_character (xpos, ypos, 4, buffer[i]);
			object->vy = -1.0;
			sge_object_set_lifetime (object, 50);	//1s
		}
		g_free (buffer);
		snd_play(SND_CLICK, 1);
#endif		
	}
}

void
gweled_remove_gems_and_update_score (void)
{
	g_list_foreach (g_alignment_list, delete_alignment_from_board, NULL);
}

void
take_down_alignment (gpointer object, gpointer user_data)
{
	gint i;
	T_Alignment *alignment;

	alignment = (T_Alignment *) object;

	if (alignment->direction == 1)	// horizontal
		for (i = alignment->x; i < alignment->x + alignment->length; i++)
			sge_destroy_object (g_gem_objects[i][alignment->y], NULL);
	else
		for (i = alignment->y; i < alignment->y + alignment->length; i++)
			sge_destroy_object (g_gem_objects[alignment->x][i], NULL);
}

void
gweled_take_down_deleted_gems (void)
{
	g_list_foreach (g_alignment_list, take_down_alignment, NULL);
}

void
destroy_alignment (gpointer object, gpointer user_data)
{
	g_alignment_list = g_list_remove (g_alignment_list, object);
}

void
destroy_all_alignments (void)
{
	g_list_foreach (g_alignment_list, destroy_alignment, NULL);
}

void
gweled_delete_gems_for_bonus (void)
{
	gint i;
	T_Alignment * alignment;
	
	destroy_all_alignments ();
	for (i = 0; i < NB_BONUS_GEMS; i++) {
		alignment = (T_Alignment *) g_malloc (sizeof (T_Alignment));
		alignment->x = g_rand_int_range (g_random_generator, 0, 7);
		alignment->y = g_rand_int_range (g_random_generator, 0, 7);
		alignment->direction = 1;
		alignment->length = 1;
		g_alignment_list = g_list_append (g_alignment_list, (gpointer) alignment);
	}
}

// FIXME!!!
//
// if we have the following pattern:
//
// xxoxoo
//
// and swap the 2 central gems:
//
// xxxooo <- this is counted as 1 alignment of 6
//
// giving a score of 40 appearing in the middle rather than 10 + 40 (combo bonus).
// However the fix implies a significant change in the function below for
// a bug that is unlikely to happen. I will fix it. Just... not now.
gboolean
gweled_check_for_alignments (void)
{
	gint i, j, i_nb_aligned, start_x, start_y;
	T_Alignment *alignment;

	destroy_all_alignments ();
// make a list of vertical alignments
	i_nb_aligned = 0;

	for (i = 0; i < BOARD_WIDTH; i++) {		
		for (j = 0; j < BOARD_HEIGHT; j++)
			if ((gweled_is_part_of_an_alignment (i, j) & 2) == 2) {
				// record the origin of the alignment
				if (i_nb_aligned == 0) {
					start_x = i;
					start_y = j;
				}
				i_nb_aligned++;
			} else {
				// we found one, let's remember it for later use                                
				if (i_nb_aligned > 2) {
					alignment = (T_Alignment *)g_malloc (sizeof (T_Alignment));
					alignment->x = start_x;
					alignment->y = start_y;
					alignment->direction = 2;
					alignment->length = i_nb_aligned;
					g_alignment_list = g_list_append(g_alignment_list, (gpointer) alignment);
				}
				i_nb_aligned = 0;
			}

		// end of column
		if (i_nb_aligned > 2) {
			alignment = (T_Alignment *)g_malloc (sizeof (T_Alignment));
			alignment->x = start_x;
			alignment->y = start_y;
			alignment->direction = 2;
			alignment->length = i_nb_aligned;
			g_alignment_list = g_list_append (g_alignment_list, (gpointer) alignment);
		}
		i_nb_aligned = 0;
	}

// make a list of horizontal alignments
	i_nb_aligned = 0;

	for (j = 0; j < BOARD_HEIGHT; j++) {
		for (i = 0; i < BOARD_WIDTH; i++)
			if ((gweled_is_part_of_an_alignment (i, j) & 1) == 1) {
				// record the origin of the alignment
				if (i_nb_aligned == 0) {
					start_x = i;
					start_y = j;
				}
				i_nb_aligned++;
			} else {
				// if we found one, let's remember it for later use                             
				if (i_nb_aligned > 2) {
					alignment = (T_Alignment *)g_malloc (sizeof (T_Alignment));
					alignment->x = start_x;
					alignment->y = start_y;
					alignment->direction = 1;
					alignment->length = i_nb_aligned;
					g_alignment_list = g_list_append (g_alignment_list, (gpointer) alignment);
				}
				i_nb_aligned = 0;
			}

		// end of row
		if (i_nb_aligned > 2) {
			alignment = (T_Alignment *) g_malloc (sizeof (T_Alignment));
			alignment->x = start_x;
			alignment->y = start_y;
			alignment->direction = 1;
			alignment->length = i_nb_aligned;
			g_alignment_list = g_list_append (g_alignment_list, (gpointer) alignment);
		}
		i_nb_aligned = 0;
	}

	return (g_list_length (g_alignment_list) != 0);
}

void display_multiply(void)
{
	char buf[64];
	if ((gi_bonus_multiply >>1) > 9)
		snprintf(buf, sizeof(buf), "XX");
	else	
		snprintf(buf, sizeof(buf), "X%d", gi_bonus_multiply >> 1);
	gfx_draw_text(buf, 800 - 48, 480 - 24, 0);
}

void display_time(float num)
{
	SDL_Rect src,dst;
	src.x = 0;
	src.y = 0; 
	src.w = TIME_W * num; 
	src.h = TIME_H;
	dst.x = TIME_X;
	dst.y = TIME_Y; 
	dst.w = src.w;
	dst.h = src.h;
	gfx_clear(TIME_X, TIME_Y, TIME_W, TIME_H);
	SDL_BlitSurface(sge_get_pixbuf(gi_power_pixbuf), &src, screen, &dst);
	gfx_update(TIME_X, TIME_Y, TIME_W, TIME_H);
}

void display_hi_score(void)
{
	char score[64];
	if (game_timer_mode)
		snprintf(score, sizeof(score),"%06d", gi_high_score);
	else	
		snprintf(score, sizeof(score),"%06d", gi_notime_high_score);
	gfx_draw_text(score, 30, 120, 1);
}

void display_current_score(void)
{
	char score[64];
	snprintf(score, sizeof(score),"%06d", gi_current_score);
	gfx_draw_text(score, 30, 200, 1);
}

gboolean board_engine_loop (gpointer data)
{
	static gint x1, y1, x2, y2, time_slice = 0;
	static T_SGEObject *cursor[2] = { NULL, NULL };
	gchar msg_buffer[200];
	board_lock();
//	gint hiscore_rank;
	time_slice++;
	
// progressive score
	if(gi_current_score < gi_score)
	{
		gi_current_score += 10;
		display_current_score();
	}
/* Let's first check if we are in timer mode, and penalize the player if necessary */
#if 1
	if (TIMER_MODE && gi_game_running && (time_slice % 80 == 0))
	{
		gi_total_gems_removed -= g_steps_for_timer;
		if (gi_total_gems_removed <= gi_previous_bonus_at) {
			gweled_draw_message ("time's up #");
			gi_game_running = 0;
			remove(SAVE_PATH);	
	#if 0 // gloomt		
 			hiscore_rank = gnome_score_log ((gfloat) gi_score, "timed", TRUE);
 			if (hiscore_rank > 0)
 				show_hiscores (hiscore_rank);
 			show_hiscores (gi_score);
	#endif		
			g_do_not_score = FALSE;
			gi_state = _IDLE;
		} {/* else
			gtk_progress_bar_set_fraction ((GtkProgressBar *)
						       g_progress_bar,
						       (float)(gi_total_gems_removed -gi_previous_bonus_at)
		*/ // gloomy				       / (float)(gi_next_bonus_at - gi_previous_bonus_at));
			display_time((float)(gi_total_gems_removed -gi_previous_bonus_at)
					       / (float)(gi_next_bonus_at - gi_previous_bonus_at));
		}
	}
#endif
//	fprintf(stderr,"state%d\n", gi_state);
	switch (gi_state) {
	case _IDLE:
		if (gi_gem_clicked) {
			x1 = gi_x_click;
			y1 = gi_y_click;
			gi_state = _FIRST_GEM_CLICKED;
			if (cursor[0])
				sge_destroy_object (cursor[0], NULL);
			cursor[0] = sge_create_object (TILE_WIDTH * x1,
					TILE_HEIGHT * y1,
					2, gi_cursor_pixbuf);
//			fprintf(stderr,"Gem clicked\n");		
			gi_gem_clicked = 0;
			gi_gem_dragged = 0;
		}
		break;

	case _FIRST_GEM_CLICKED:
		if (gi_gem_clicked) {
			x2 = gi_x_click;
			y2 = gi_y_click;
			gi_gem_clicked = 0;
			if (((x1 == x2) && (fabs (y1 - y2) == 1)) || 
			    ((y1 == y2) && (fabs (x1 - x2) == 1))) {
				// If the player clicks an adjacent gem, try to swap
				if (cursor[1])
					sge_destroy_object (cursor[1], NULL);
				cursor[1] = sge_create_object (TILE_WIDTH * x2,
						TILE_HEIGHT * y2,
						2, gi_cursor_pixbuf);
				sge_object_move_to (g_gem_objects[x1][y1],
						x2 * TILE_WIDTH,
						y2 * TILE_HEIGHT);
				sge_object_move_to (g_gem_objects[x2][y2],
						x1 * TILE_WIDTH,
						y1 * TILE_HEIGHT);
				snd_play(SND_SWAP, 1);
				gi_state = _SECOND_GEM_CLICKED;
			} else if((x1 == x2) && (y1 == y2)) {
				if (cursor[1])
					sge_destroy_object (cursor[1], NULL);
				cursor[1] = NULL;					
				// If the player clicks the selected gem, deselect it
				if(cursor[0])
					sge_destroy_object(cursor[0], NULL);
				gi_state = _IDLE;
				gi_gem_clicked = 0;
			} else {
				if (cursor[1])
					sge_destroy_object (cursor[1], NULL);
				cursor[1] = NULL;					
				// If the player clicks anywhere else, make that the first selection
				x1 = x2;
				y1 = y2;
				if (cursor[0])
					sge_destroy_object (cursor[0], NULL);
				cursor[0] = sge_create_object (TILE_WIDTH * x1,
						TILE_HEIGHT * y1,
						2, gi_cursor_pixbuf);
			}
		}else if(gi_gem_dragged) {
			//printf("gem dragged\n");
			if (cursor[1])
					sge_destroy_object (cursor[1], NULL);
				cursor[1] = sge_create_object (TILE_WIDTH * gi_x_drag,
						TILE_HEIGHT * gi_y_drag,
						2, gi_cursor_pixbuf);
		}
		break;

	case _SECOND_GEM_CLICKED:
		if (!sge_object_is_moving (g_gem_objects[x1][y1]) && !sge_object_is_moving (g_gem_objects[x2][y2])) {
			gweled_swap_gems (x1, y1, x2, y2);
			if (!gweled_is_part_of_an_alignment (x1, y1) && !gweled_is_part_of_an_alignment (x2, y2)) {
				//gweled_draw_game_message("illegal move !", 1.0);
				sge_object_move_to (g_gem_objects[x1][y1],
						x2 * TILE_WIDTH,
						y2 * TILE_HEIGHT);
				sge_object_move_to (g_gem_objects[x2][y2],
						x1 * TILE_WIDTH,
						y1 * TILE_HEIGHT);
				gi_state = _ILLEGAL_MOVE;
			} else {
//				fprintf(stderr,"Clearing\n");
				if (cursor[0])
					sge_destroy_object (cursor[0], NULL);
				if (cursor[1])
					sge_destroy_object (cursor[1], NULL);
				cursor[0] = NULL;
				cursor[1] = NULL;
				gi_gems_removed_per_move = 0;
				gi_state = _MARK_ALIGNED_GEMS;
			}
		}
		break;

	case _ILLEGAL_MOVE:
		if (!sge_object_is_moving (g_gem_objects[x1][y1]) && !sge_object_is_moving (g_gem_objects[x2][y2])) {
//			fprintf(stderr,"Illegial move\n");
			if (cursor[0]) {
//				fprintf(stderr, "Destory cursor\n");
				sge_destroy_object (cursor[0], NULL);
			}
			if (cursor[1])
				sge_destroy_object (cursor[1], NULL);
			cursor[0] = NULL;
			cursor[1] = NULL;
			gweled_swap_gems (x1, y1, x2, y2);
			gi_state = _IDLE;
		}
		break;

	case _MARK_ALIGNED_GEMS:
		if (gweled_check_for_alignments () == TRUE) {
			gweled_take_down_deleted_gems ();
			gweled_remove_gems_and_update_score ();
			if (g_do_not_score == FALSE)
				gi_total_gems_removed += gi_gems_removed_per_move;
			if (gi_total_gems_removed <= gi_next_bonus_at)
//				gtk_progress_bar_set_fraction ((GtkProgressBar *) g_progress_bar, (float) (gi_total_gems_removed - gi_previous_bonus_at) / (float) (gi_next_bonus_at - gi_previous_bonus_at));
				display_time((float) (gi_total_gems_removed - gi_previous_bonus_at) / (float) (gi_next_bonus_at - gi_previous_bonus_at));
			else
				display_time(1.0);
//				gtk_progress_bar_set_fraction ((GtkProgressBar *) g_progress_bar, 1.0);
			//sprintf (msg_buffer, "<span weight=\"bold\">%06d</span>", gi_score);
			//gtk_label_set_markup ((GtkLabel *) g_score_label, msg_buffer);
			gweled_refill_board ();
			gweled_gems_fall_into_place ();
			gi_state = _BOARD_REFILLING;
		} else {
			if (gweled_check_for_moves_left (NULL, NULL) == FALSE) {
				if ((gi_next_bonus_at == FIRST_BONUS_AT) || TIMER_MODE) {
					gint i, j;

					gweled_draw_game_message ("no moves left #", 1.0);
					memset (gi_nb_of_tiles, 0, 7 * sizeof (gchar));

					for (i = 0; i < BOARD_WIDTH; i++)
						for (j = 0; j < BOARD_HEIGHT; j++) {
							sge_destroy_object (g_gem_objects[i][j], NULL);
							gpc_game_board[i][j] = get_new_tile();
							gi_nb_of_tiles[gpc_game_board[i][j]]++;
						}
					g_do_not_score = TRUE;
					while(gweled_check_for_alignments ()) {
						gweled_remove_gems_and_update_score ();
						gweled_refill_board();
					};
					g_do_not_score = FALSE;
					for (i = 0; i < BOARD_WIDTH; i++)
						for (j = 0; j < BOARD_HEIGHT; j++) {
							g_gem_objects[i][j] = sge_create_object
								(i * TILE_WIDTH,
							    	(j - BOARD_HEIGHT) * TILE_HEIGHT,
								1, gi_gems_pixbuf[gpc_game_board[i][j]]);
						}
					gweled_gems_fall_into_place ();
					gi_state = _MARK_ALIGNED_GEMS;
				} else {
					gweled_draw_message ("no moves left #");
					gi_game_running = 0;
					remove(SAVE_PATH);	
			#if 0 // gloomy		
					hiscore_rank = gnome_score_log ((gfloat) gi_score, "easy", TRUE);
 					if (hiscore_rank > 0)
 						show_hiscores (hiscore_rank);
			#endif			
					g_do_not_score = FALSE;
					gi_state = _IDLE;
				}
			} else {
				g_do_not_score = FALSE;
				gi_state = _IDLE;
			}
		}
		break;

	case _BOARD_REFILLING:
		if (!sge_objects_are_moving_on_layer (1)) {
			if (gi_total_gems_removed >= gi_next_bonus_at) {
				gi_previous_bonus_at = gi_next_bonus_at;
				gi_next_bonus_at *= 2;
				if (TIMER_MODE)
					g_steps_for_timer = (gi_next_bonus_at - gi_previous_bonus_at) / TOTAL_STEPS_FOR_TIMER + 1;
				gi_bonus_multiply++;
				sprintf (msg_buffer, "bonus x%d", gi_bonus_multiply >> 1);
				gweled_draw_game_message (msg_buffer, 1.0);
				gweled_delete_gems_for_bonus ();
				gweled_take_down_deleted_gems ();
				gweled_remove_gems_and_update_score ();
				if (TIMER_MODE)
					gi_total_gems_removed = (gi_next_bonus_at + gi_previous_bonus_at) / 2;
				display_time((float) (gi_total_gems_removed - gi_previous_bonus_at) /
					(float) (gi_next_bonus_at - gi_previous_bonus_at));
				display_multiply();
//				gtk_progress_bar_set_fraction ((GtkProgressBar *) g_progress_bar,
//					(float) (gi_total_gems_removed - gi_previous_bonus_at) /
//					(float) (gi_next_bonus_at - gi_previous_bonus_at));
				//sprintf (msg_buffer, "<span weight=\"bold\">%06d</span>", gi_score);
//				//gtk_label_set_markup ((GtkLabel *)g_score_label, msg_buffer);
//				sprintf (msg_buffer, "<span weight=\"bold\">x%d</span>", gi_bonus_multiply >> 1);
//				gtk_label_set_markup ((GtkLabel *)g_bonus_label, msg_buffer);
				gweled_refill_board ();
				gweled_gems_fall_into_place ();
				g_do_not_score = TRUE;
			} else {
				gi_state = _MARK_ALIGNED_GEMS;
			}
		}
		break;
	default:
		break;
	}
	board_unlock();
	return TRUE;
}

static T_SGEObject *gweled_draw_character (int x, int y, int layer, char character)
{
	int index = (unsigned char)character;
	if (gpc_font_glyphs[index] != -1)
		return sge_create_object (x, y, layer,
					  gi_small_charset_pixbuf[gpc_font_glyphs
							    [index]]);
	else
		return NULL;
}

void gweled_draw_message_at (gchar * in_message, gint msg_x, gint msg_y)
{
	int i;
	gchar *message;

	message = g_ascii_strup (in_message, -1);

	for (i = 0; i < strlen (message); i++)
		gweled_draw_character (msg_x + i * SMALL_FONT_WIDTH, msg_y, 1,
				       message[i]);

	free (message);
}

void gweled_draw_message (gchar * in_message)
{
	gint msg_x, msg_y, msg_w;

	msg_w = SMALL_FONT_WIDTH * strlen (in_message);
	msg_x = (BOARD_WIDTH * TILE_WIDTH - msg_w) >> 1;
	msg_y = (BOARD_HEIGHT * TILE_HEIGHT - SMALL_FONT_HEIGHT) >> 1;

	gweled_draw_message_at (in_message, msg_x, msg_y);
}

void gweled_draw_game_message (gchar * in_message, double timing)
{
	int i;
	gint msg_x, msg_y, msg_w;
	T_SGEObject *p_object;
	gchar *message;

	msg_w = SMALL_FONT_WIDTH * strlen (in_message);
	msg_x = (BOARD_WIDTH * TILE_WIDTH - msg_w) >> 1;
	msg_y = (BOARD_HEIGHT * TILE_HEIGHT - SMALL_FONT_HEIGHT) >> 1;

	message = g_ascii_strup (in_message, -1);

	for (i = 0; i < strlen (message); i++) {
		int index = (unsigned char)(message[i]);
		if (gpc_font_glyphs[index] != -1) {
			p_object =
			    sge_create_object (msg_x + i * SMALL_FONT_WIDTH,
					       msg_y, 3,
					       gi_small_charset_pixbuf[gpc_font_glyphs[index]]);
			sge_object_set_lifetime (p_object, (int) (50.0 * timing));
		}
	}
	free (message);
}

int board_init(void)
{
	if (board_mutex)
		SDL_DestroyMutex(board_mutex);
	board_mutex = SDL_CreateMutex();
	return (board_mutex)?0:-1;
}

void board_done(void)
{
	if (board_mutex)
		SDL_DestroyMutex(board_mutex);
}
