/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "directionsitemdelegate.h"

#include <QDebug>
#include <QPainter>
#include <QTextItem>

#include "directions.h"

DirectionsItemDelegate::DirectionsItemDelegate(QListView *parent) :
    QStyledItemDelegate(parent)
{
}

void DirectionsItemDelegate::paint( QPainter * painter, const QStyleOptionViewItem & option, const QModelIndex & index ) const {
    QStyleOptionViewItemV4 options = option;
    initStyleOption(&options, index);

    painter->save();

    QColor highlightColor = options.palette.color(QPalette::Highlight);

    /* Call this to get the focus rect and selection background. */
    QStyleOptionViewItemV4 bkgOptions = options;
    bkgOptions.text = "";
    bkgOptions.rect.setLeft(bkgOptions.rect.left()+m_leftMargin);
    bkgOptions.rect.setRight(bkgOptions.rect.right() - m_rightMargin);

    Step *pStep = static_cast<Step *>(index.internalPointer());
    if ( !pStep ) {
        // summary line
        // don't paint selection?
        /*
        QRect r = options.rect;
        r.setLeft(r.left()+m_leftMargin);
        r.setRight(options.rect.right() - m_rightMargin);
        */
        painter->setFont(m_font);
        painter->setPen(highlightColor);

        painter->drawText(bkgOptions.rect, Qt::TextWordWrap | Qt::AlignVCenter, options.text);
        painter->drawLine(options.rect.left(), options.rect.top(),options.rect.right(), options.rect.top());

        painter->restore();
        return;
    }

    bkgOptions.widget->style()->drawControl(QStyle::CE_ItemViewItem, &bkgOptions, painter);

    Q_ASSERT(pStep);

    // text
    QRect r = options.rect;
    r.setLeft(r.left()+m_leftMargin);
    r.setRight(options.rect.right() - m_rightMargin);
    painter->setFont(m_font);
    painter->drawText(r, Qt::TextWordWrap | Qt::AlignVCenter, pStep->plainDescription());

    painter->setPen(Qt::lightGray);

    // step index
    r = options.rect;
    r.setWidth(m_leftMargin);
    painter->setFont(options.font);
    painter->drawText(r, Qt::AlignVCenter | Qt::AlignHCenter, QString::number(index.row()+1));

    r = options.rect;
    r.setLeft(options.rect.right()-m_rightMargin);
    r.setTop(r.top()+m_vertMargin);
    r.setBottom(r.bottom()-m_vertMargin);

    // distance
    painter->setPen(highlightColor); // QColor(255,128,0)
    painter->drawText(r, Qt::AlignTop | Qt::AlignHCenter, pStep->m_distanceStr);

    // time
    painter->setPen(Qt::lightGray);
    painter->drawText(r, Qt::AlignBottom | Qt::AlignHCenter, pStep->m_durationStr);

    if ( index.row()>0 ) {
        painter->setPen(highlightColor);
        painter->drawLine(options.rect.left(), options.rect.top(),options.rect.right(), options.rect.top());
    }

    painter->restore();
}

QSize DirectionsItemDelegate::sizeHint( const QStyleOptionViewItem & option, const QModelIndex & index ) const {
    QListView *pList = dynamic_cast<QListView *>(parent());
    if ( !pList ) return QStyledItemDelegate::sizeHint(option, index);

    int w = pList->viewport()->width();
    w -= m_leftMargin;
    w -= m_rightMargin;

    QRect r(0,0,w,10);
    QFontMetrics metrics(m_font);
    r = metrics.boundingRect(r, Qt::TextWordWrap, index.data().toString());

    int h = r.height();
    if ( h<option.fontMetrics.height()*2 ) h = option.fontMetrics.height()*2;
    h += m_vertMargin*2;

    return QSize(pList->viewport()->width(), h);
}
