/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <time.h>

#include "eventlogger-util.h"
#include "common.h"

/**
 * get_full_status_text:
 *
 * eventlogger status plugin limits status text to 100 characters
 * So we need to get the full text string from el.db
 */
static ContactStatus *
get_contact_status (RTComEl *eventlogger,
                    RTComElQuery *query)
{
  RTComElIter *iter;
  GValueArray *values;
  const gchar *free_text;
  ContactStatus *status;

  iter = rtcom_el_get_events (eventlogger, query);

  if(!(iter && rtcom_el_iter_first(iter))) {
      g_debug("No query result");
      if (iter)
        g_object_unref(iter); /* must unref iter, otherwise the app will crash */
      return NULL;
  }

  values = rtcom_el_iter_get_valuearray (iter, "free-text", NULL);
  if (!values) {
    g_debug("Failed to get the free-text");
    g_object_unref (iter);
    return NULL;
  }
  free_text = g_value_get_string (g_value_array_get_nth(values, 0));

  status = eventlogger_util_contact_status_new ();
  status->text = g_strdup (free_text);

  GValue timeval = {0};
  rtcom_el_iter_get_raw (iter, "start-time", &timeval);
  time_t t = g_value_get_int (&timeval);
  g_value_unset (&timeval);
  status->start_time = get_timestamp (t);

  g_value_array_free (values);
  g_object_unref (iter);

  return status;
}

/**
 * eventlogger_util_get_contact_status_by_id:
 *
 * @event_id: id of the event in el.db
 *
 * Get the status of event identified by event_id
 */
ContactStatus *
eventlogger_util_get_contact_status_by_id (RTComEl *eventlogger,
                                           gint event_id)
{
  RTComElQuery *query;
  ContactStatus *status;

  query = rtcom_el_query_new (eventlogger);
  rtcom_el_query_set_limit (query, 1);

  if(!rtcom_el_query_prepare (
              query,
              "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
              "id", event_id, RTCOM_EL_OP_EQUAL,
              NULL)) {
      g_debug("Failed to get the status message of %d", event_id);
      return NULL;
  }

  status = get_contact_status (eventlogger, query);
  g_object_unref (query);

  return status;
}

/**
 * eventlogger_util_get_last_contact_status:
 *
 * @remote_uid: vcard_field of the contact (aka IM field), e.g. someone@gmail.com
 *
 * Get the last status of a contact
 */
ContactStatus *
eventlogger_util_get_last_contact_status (RTComEl *eventlogger,
                                          const gchar *local_uid,
                                          const gchar *remote_uid)
{
  RTComElQuery *query;
  ContactStatus *status;
  gboolean res;

  query = rtcom_el_query_new (eventlogger);
  rtcom_el_query_set_limit (query, 1);

  if (local_uid) {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "local-uid", local_uid, RTCOM_EL_OP_EQUAL,
                "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL,
                NULL);
  }
  else {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL,
                NULL);
  }

  if(!res) {
    g_debug("Failed to get the last status message of %s", remote_uid);
    return NULL;
  }

  status = get_contact_status (eventlogger, query);
  g_object_unref (query);

  return status;
}

ContactStatus *
eventlogger_util_contact_status_new ()
{
  ContactStatus *status;
  status = g_slice_new0 (ContactStatus);
  status->text = NULL;
  status->start_time = NULL;
  return status;
}

void
eventlogger_util_contact_status_free (ContactStatus *status)
{
  if (status) {
    g_free (status->text);
    g_free (status->start_time);
    g_slice_free (ContactStatus, status);
  }
}
