/*
 * Copyright (C) 2004-2010 Geometer Plus <contact@geometerplus.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#include <ZLOptionsDialog.h>

#include <optionEntries/ZLSimpleOptionEntry.h>
#include <optionEntries/ZLToggleBooleanOptionEntry.h>

#include "OptionsDialog.h"
#include "ScrollingTab.h"

#include "../fbreader/FBReader.h"

MovementToleranceEntry::MovementToleranceEntry(const ZLResource &resource, ZLIntegerOption &sensitivityOption) : myOption(sensitivityOption) {
	myValues.push_back(resource["stylus"].value());
	myValues.push_back(resource["finger"].value());
}

MovementToleranceEntry::Tolerance MovementToleranceEntry::positionToTolerance(Position position) const {
	Tolerance value;

	switch(position) {
		case POS_STYLUS:
			value = STYLUS;
			break;
		case POS_FINGER:
			value = FINGER;
			break;
		default:
			value = STYLUS;
	};

	return value;
}

MovementToleranceEntry::Position MovementToleranceEntry::toleranceToPosition(Tolerance tolerance) const {
	Position value;


	switch(tolerance) {
		case STYLUS:
			value = POS_STYLUS;
			break;
		case FINGER:
			value = POS_FINGER;
			break;
		default:
			value = POS_STYLUS;
	};

	return value;
}

const std::string &MovementToleranceEntry::initialValue() const {
	return myValues[toleranceToPosition((MovementToleranceEntry::Tolerance)myOption.value())];
}

const std::vector<std::string> &MovementToleranceEntry::values() const {
	return myValues;
}

void MovementToleranceEntry::onAccept(const std::string &value) {
	for (size_t index = 0; index != myValues.size(); ++index) {
		if (myValues[index] == value) {
			myOption.setValue(positionToTolerance((MovementToleranceEntry::Position)index));
			break;
		}
	}
}

void OptionsDialog::createScrollingTab() {
	FBReader &fbreader = FBReader::Instance();

	ZLDialogContent &scrollingTab = myDialog->createTab(ZLResourceKey("Scrolling"));
	scrollingTab.addOption(ZLResourceKey("keyLinesToScroll"), new ZLSimpleSpinOptionEntry(fbreader.LinesToScrollOption, 1));
	scrollingTab.addOption(ZLResourceKey("keyLinesToKeep"), new ZLSimpleSpinOptionEntry(fbreader.LinesToKeepOption, 1));
	scrollingTab.addOption(ZLResourceKey("keyScrollDelay"), new ZLSimpleSpinOptionEntry(fbreader.KeyScrollingDelayOption, 50));

	static ZLResourceKey toleranceKey("movementTolerance");
	MovementToleranceEntry *movementToleranceEntry = new MovementToleranceEntry(scrollingTab.resource(toleranceKey), fbreader.MovementToleranceOption);
	scrollingTab.addOption(toleranceKey, movementToleranceEntry);

	const bool hasTouchScreen = 
		ZLBooleanOption(ZLCategoryKey::EMPTY, ZLOption::PLATFORM_GROUP, ZLOption::TOUCHSCREEN_PRESENTED, false).value();
	if (hasTouchScreen) {
		ZLToggleBooleanOptionEntry *enableTapScrollingEntry =
			new ZLToggleBooleanOptionEntry(fbreader.EnableTapScrollingOption);
		scrollingTab.addOption(ZLResourceKey("enableTapScrolling"), enableTapScrollingEntry);
		const bool isFingerTapDetectionSupported = 
			ZLBooleanOption(ZLCategoryKey::EMPTY, ZLOption::PLATFORM_GROUP, ZLOption::FINGER_TAP_DETECTABLE, false).value();
		if (isFingerTapDetectionSupported) {
			ZLOptionEntry *fingerOnlyEntry =
				new ZLSimpleBooleanOptionEntry(fbreader.TapScrollingOnFingerOnlyOption);
			scrollingTab.addOption(ZLResourceKey("fingerOnly"), fingerOnlyEntry);
			enableTapScrollingEntry->addDependentEntry(fingerOnlyEntry);
		}
		ZLToggleBooleanOptionEntry *dragOnlyEntry =
			new ZLToggleBooleanOptionEntry(fbreader.TapScrollingOnDragOnlyOption);
		scrollingTab.addOption(ZLResourceKey("dragOnly"), dragOnlyEntry);
		enableTapScrollingEntry->addDependentEntry(dragOnlyEntry);
		ZLToggleBooleanOptionEntry *variableDragEntry =
			new ZLToggleBooleanOptionEntry(fbreader.VariableDragOption);
		scrollingTab.addOption(ZLResourceKey("variableDrag"), variableDragEntry);
		dragOnlyEntry->addDependentEntry(variableDragEntry);
		ZLOptionEntry *variableDragLengthEntry =
			new ZLSimpleSpinOptionEntry(fbreader.VariableDragLengthOption, 100);
		scrollingTab.addOption(ZLResourceKey("variableDragLength"), variableDragLengthEntry);
		variableDragEntry->addDependentEntry(variableDragLengthEntry);

		variableDragEntry->onStateChanged(variableDragEntry->initialState());
		dragOnlyEntry->onStateChanged(dragOnlyEntry->initialState());
		enableTapScrollingEntry->onStateChanged(enableTapScrollingEntry->initialState());
	}
}
