#include "searchhistorydialog.h"
#include "textlistdelegate.h"

SearchHistoryDialog::SearchHistoryDialog(SearchListModel *model, QWidget *parent) :
    AbstractDialog(parent),
    m_model(model),
    m_listView(new ListView(this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_clearButton(m_buttonBox->addButton(tr("Clear"), QDialogButtonBox::ActionRole)),
    m_filterBox(new FilterBox(this)),
    m_contextMenu(new QMenu(this)),
    m_removeAction(m_contextMenu->addAction(tr("Remove"), this, SLOT(removeSearchTerm()))),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Search history"));
    setAttribute(Qt::WA_DeleteOnClose, false);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(new TextListDelegate(Qt::AlignCenter, m_listView));
    m_filterBox->hide();

    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_clearButton, SIGNAL(clicked()), m_model, SLOT(clearSearches()));
    connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    connect(m_filterBox, SIGNAL(cleared()), this, SLOT(hideFilterBox()));
    connect(this, SIGNAL(rejected()), m_filterBox, SLOT(clear()));

    onOrientationChanged();
}

void SearchHistoryDialog::setLandscapeLayout() {
    setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_listView, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void SearchHistoryDialog::setPortraitLayout() {
    setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_listView, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void SearchHistoryDialog::onItemClicked(const QModelIndex &index) {
    m_filterBox->clear();
    emit queryChosen(index.data().toString());
    accept();
}

void SearchHistoryDialog::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        showFilterBox();
        m_filterBox->setFilterText(event->text());
    }
}

void SearchHistoryDialog::showFilterBox() {
    if (!inPortrait()) {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 2, 0, Qt::AlignBottom);
    }

    m_grid->addWidget(m_filterBox, 1, 0);
    m_filterBox->show();
    m_filterBox->setFocus(Qt::OtherFocusReason);
}

void SearchHistoryDialog::hideFilterBox() {
    if (!inPortrait()) {
        m_filterBox->hide();
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_filterBox->hide();
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void SearchHistoryDialog::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_removeAction);
}

void SearchHistoryDialog::removeSearchTerm() {
    m_model->removeSearch(m_listView->currentIndex().row());
}
