#include "dailymotiongrouplistmodel.h"
#include "feedurls.h"
#include "json.h"
#include <QtNetwork/QNetworkReply>
#include <QtNetwork/QNetworkRequest>

using namespace QtJson;

DailymotionGroupListModel::DailymotionGroupListModel(Session *session, QObject *parent) :
    AbstractGroupListModel(parent),
    m_session(session),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionGroupListModel::setFeed(const QString &afeed) {
    m_feed = afeed + "&fields=" + DAILYMOTION_GROUP_FIELDS;

    if (afeed == DAILYMOTION_GROUPS_FEED) {
        connect(m_session->dailymotion(), SIGNAL(groupJoined(QSharedPointer<GroupItem>)), this, SLOT(onGroupJoined(QSharedPointer<GroupItem>)));
        connect(m_session->dailymotion(), SIGNAL(groupLeft(QSharedPointer<GroupItem>)), this, SLOT(onGroupLeft(QSharedPointer<GroupItem>)));
    }
}

void DailymotionGroupListModel::getGroups(const QString &afeed) {
    if (!afeed.isEmpty()) {
        setFeed(afeed);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->dailymotion()->createReply(feed(), offset());
    connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void DailymotionGroupListModel::search(const QString &aquery, Queries::QueryOrder anorder) {
    if (!aquery.isEmpty()) {
        setQuery(aquery);
        setOrder(anorder);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->dailymotion()->createSearchReply(Queries::Groups, query(), offset(), order());
    connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void DailymotionGroupListModel::addGroups() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        setLoading(false);
        setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(m_session->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        connect(m_session->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        m_session->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            setLoading(false);
            setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                appendGroup(QSharedPointer<GroupItem>(new GroupItem(entries.at(i).toMap())));
            }

            setLoading(false);
            setMoreResults(res.value("has_more").toBool());
            setOffset(res.value("page").toInt() + 1);
        }

        disconnect(m_session->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        disconnect(m_session->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    reply->deleteLater();
}

void DailymotionGroupListModel::onGroupJoined(QSharedPointer<GroupItem> group) {
    if (!group.isNull()) {
        appendGroup(group);
    }
}

void DailymotionGroupListModel::onGroupLeft(QSharedPointer<GroupItem> group) {
    if (!group.isNull()) {
        removeGroup(group.data()->id());
    }
}

void DailymotionGroupListModel::getMoreGroups() {
    if ((moreResults()) && (!loading())) {
        getGroups();
    }
}
