#ifndef URLGRABBER_H
#define URLGRABBER_H

#include "enums.h"
#include <QObject>
#include <QUrl>
#include <QList>
#include <QVariant>
#include <QScriptValue>

class QNetworkAccessManager;
class QNetworkReply;

typedef struct {
    Videos::VideoFormat format;
    QVariant value;
    QString displayName;
} VideoFormat;

class UrlGrabber : public QObject
{
    Q_OBJECT

    Q_PROPERTY(bool busy
               READ busy
               NOTIFY busyChanged)

public:
    explicit UrlGrabber(QObject *parent = 0);
    inline QNetworkAccessManager *networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QList<VideoFormat> youtubeFormats() const { return m_youtubeFormats; }
    inline QList<VideoFormat> dailymotionFormats() const { return m_dailymotionFormats; }
    inline QList<VideoFormat> vimeoFormats() const { return m_vimeoFormats; }
    inline bool busy() const { return m_busy; }
    inline bool cancelled() const { return m_cancelled; }

    static UrlGrabber* instance();

public slots:
    void getVideoUrl(int service, const QString &id);
    void getYouTubeVideoUrl(const QString &id);
    void getDailymotionVideoUrl(const QString &id);
    void getVimeoVideoUrl(const QString &id);
    void getAvailableVideoFormats(int service, const QString &id);
    void getAvailableYouTubeVideoFormats(const QString &id);
    void getAvailableDailymotionVideoFormats(const QString &id);
    void getAvailableVimeoVideoFormats(const QString &id);
    void getSubtitlesUrl(int service, const QString &id, const QString &language);
    void getYouTubeSubtitlesUrl(const QString &id, const QString &language);
    void getDailymotionSubtitlesUrl(const QString &id, const QString &language);
    void getVimeoSubtitlesUrl(const QString &id, const QString &language);
    void getAvailableSubtitles(int service, const QString &id);
    void getAvailableYouTubeSubtitles(const QString &id);
    void getAvailableDailymotionSubtitles(const QString &id);
    void getAvailableVimeoSubtitles(const QString &id);
    void setYouTubeFormats(QSet<int> formats);
    void setDailymotionFormats(QSet<int> formats);
    void setVimeoFormats(QSet<int> formats);
    void cancelCurrentOperation();    

private:
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);
    void setCancelled(bool cancelled) { m_cancelled = cancelled; }
    QScriptValue getYouTubeDecryptionFunction(const QUrl &playerUrl);
    QMap<int, QUrl> getYouTubeVideoUrlMap(QString page, QScriptValue decryptionFunction = QScriptValue());
    void getYouTubeVideoInfoPage(const QString &id, const char* slot);
    void getYouTubeVideoWebPage(const QString &id, const char* slot);

private slots:
    void checkYouTubeVideoInfoPage();
    void checkYouTubeWebPage();
    void parseYouTubeVideoPage(QScriptValue decryptionFunction = QScriptValue(), QString page = QString());
    void parseDailymotionVideoPage();
    void parseVimeoVideoPage();
    void checkYouTubeVideoInfoFormats();
    void checkYouTubeWebPageFormats();
    void parseYouTubeVideoFormats(QScriptValue decryptionFunction = QScriptValue(), QString page = QString());
    void checkDailymotionVideoFormats();
    void checkVimeoVideoFormats();
    void addYouTubeDecryptionFunctionToCache();
    void checkYouTubeSubtitles();
    void checkDailymotionSubtitles();
    void checkVimeoSubtitles();
    void checkAvailableYouTubeSubtitles();
    void checkAvailableDailymotionSubtitles();
    void checkAvailableVimeoSubtitles();

signals:
    void gotVideoUrl(const QUrl &url, Videos::VideoFormat videoFormat);
    void gotVideoFormats(const QVariantList &formats);
    void gotSubtitlesUrl(const QUrl &url);
    void gotSubtitles(const QVariantList &subtitles);
    void busyChanged(bool isBusy);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void error(const QString &errorString);
    void currentOperationCancelled();
    void youtubeDecryptionFunctionReady(QScriptValue decryptionFunction);

private:
    static UrlGrabber *self;
    static QScriptEngine *youtubeDecryptionEngine;
    static QHash<QUrl, QScriptValue> youtubeDecryptionCache;

    QNetworkAccessManager *m_nam;
    QList<VideoFormat> m_youtubeFormats;
    QList<VideoFormat> m_dailymotionFormats;
    QList<VideoFormat> m_vimeoFormats;
    Videos::VideoFormat m_format;
    QString m_language;
    bool m_busy;
    bool m_cancelled;
    QString m_youtubePage;
};

#endif // URLGRABBER_H
