/*
* This file is part of CPCm
*
* Copyright (C) 2010 javicq
*
* This software is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* as published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA
*
*/

#define MAEMO_VERSION 5

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <startup_plugin.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-defines.h>

#if MAEMO_VERSION >= 5
#include <hildon/hildon-button.h>
#include <hildon/hildon-check-button.h>
#include <hildon/hildon-picker-button.h>
#include <hildon/hildon-touch-selector.h>
#include <hildon/hildon-gtk.h>
#else
#include <hildon/hildon-caption.h>
#endif

#include "plugin.h"
#include "gconf.h"
#include "i18n.h"

static GtkWidget * load_plugin(void);
static void unload_plugin(void);
static void write_config(void);
static GtkWidget ** load_menu(guint *);
static void update_menu(void);
static void plugin_callback(GtkWidget * menu_item, gpointer data);

GConfClient * gcc = NULL;
static GameStartupInfo gs;
static GtkWidget * menu_items[3];

static StartupPluginInfo plugin_info = {
	load_plugin,
	unload_plugin,
	write_config,
	load_menu,
	update_menu,
	plugin_callback
};

STARTUP_INIT_PLUGIN(plugin_info, gs, FALSE, TRUE)

static HildonButton* drivea_btn;
static HildonButton* driveb_btn;
static HildonButton* controls_btn;
static HildonButton* launch_btn;
//static HildonCheckButton* sound_check;
static GtkCheckButton* sound_check;

static gchar* drivea_path = NULL;
static gchar* driveb_path = NULL;

static inline void set_rom_label(gchar * text, int drive)
{
	hildon_button_set_value( drive ? driveb_btn : drivea_btn, text );
}

static inline void set_rom_value( gchar* file, int drive ) {
	if( drive ) {
		if( driveb_path ) g_free( driveb_path );
		driveb_path = file;
	} else {
		if( drivea_path ) g_free( drivea_path );
		drivea_path = file;
	}
}

static void set_rom(const char * rom_file, int drive)
{
	gchar * utf8_filename = g_filename_display_basename(rom_file);
	set_rom_label(utf8_filename, drive);
	g_free(utf8_filename);

	gboolean current_rom_file_exists = g_file_test(rom_file,
		G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR);

	if (!rom_file || strlen(rom_file) == 0 || !current_rom_file_exists) {
		set_rom_label(_("<empty>"), drive);
		set_rom_value( NULL, drive );
		return;
	}
	set_rom_value( g_strdup(rom_file), drive );

	save_clear();
}

static inline GtkWindow* get_parent_window() {
	return GTK_WINDOW(gs.ui->hildon_appview);
}

static gboolean blank_dsk( gchar* filename ) {
	FILE* f = fopen( filename, "wb" );
	if( !f ) return FALSE;
	/*FILE* b = fopen( "/opt/cpcm/blank.dsk", "rb" );
	if( !b ) { fclose(f); return FALSE; }
	while( !feof( b ) ) fputc( fgetc(b), f );
	fclose(b);*/
	fclose(f);
	return TRUE;
}

static void select_rom_callback(GtkWidget * button, gpointer data)
{
	GtkWidget * dialog;
	GtkFileFilter * filter;
	gchar * filename = NULL;

	filter = gtk_file_filter_new();
	gtk_file_filter_add_pattern(filter, "*.dsk");
	gtk_file_filter_add_pattern(filter, "*.dsk.gz");
	gtk_file_filter_add_pattern(filter, "*.DSK");
	gtk_file_filter_add_pattern(filter, "*.DSK.gz");

	dialog = hildon_file_chooser_dialog_new_with_properties(
		get_parent_window(),
		"action", GTK_FILE_CHOOSER_ACTION_OPEN,
		"local-only", TRUE,
		"filter", filter,
		NULL);
	hildon_file_chooser_dialog_set_show_upnp(HILDON_FILE_CHOOSER_DIALOG(dialog),
		TRUE);
	gtk_dialog_add_button( GTK_DIALOG(dialog), "Eject", 1 );
	gtk_dialog_add_button( GTK_DIALOG(dialog), "New disk", 2 );

	gchar* file = data ? driveb_path : drivea_path;
	if (file) {
		// By default open showing the last selected file
		gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(dialog),
			file);
	} else if( gconf_client_get_string( gcc, G_CPCM_ROMDIR, NULL) ) {
		gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER(dialog), gconf_client_get_string( gcc, G_CPCM_ROMDIR, NULL) );
	}

	gtk_widget_show_all(GTK_WIDGET(dialog));
	gint response = gtk_dialog_run(GTK_DIALOG(dialog));
	if( response  == GTK_RESPONSE_OK ) {
		filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
		gconf_client_set_string( gcc, G_CPCM_ROMDIR, gtk_file_chooser_get_current_folder( GTK_FILE_CHOOSER(dialog) ), NULL );
	}
	else if( response == 1 )
		filename = NULL;
	else if( response == 2 ) {
		GtkWidget* 	dialog2 = hildon_file_chooser_dialog_new_with_properties(
				get_parent_window(),
				"action", GTK_FILE_CHOOSER_ACTION_SAVE,
				"local-only", TRUE,
				"filter", filter,
				NULL);
		gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER(dialog2), gtk_file_chooser_get_current_folder( GTK_FILE_CHOOSER(dialog) ) );
		if( gtk_dialog_run(GTK_DIALOG(dialog2)) == GTK_RESPONSE_OK ) {
			filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog2));
			if( !g_str_has_suffix( filename, ".dsk" ) ) filename = g_strconcat( filename, ".dsk", 0 );
			if( !blank_dsk( filename ) ) filename = NULL;
		}
		else filename = NULL;
		gtk_widget_destroy(dialog2);
	}

	gtk_widget_destroy(dialog);

	if( filename || response == 1 ) set_rom(filename, (int)data);
	if (filename) g_free(filename);
}

static void controls_item_callback(GtkWidget * button, gpointer data)
{
	controls_dialog(get_parent_window(), GPOINTER_TO_INT(data));
}

static void settings_item_callback(GtkWidget * button, gpointer data)
{
	settings_dialog(get_parent_window());
}

static void about_item_callback(GtkWidget * button, gpointer data)
{
	about_dialog(get_parent_window());
}

static void keys_btn_callback(GtkWidget * button, gpointer data)
{
	keys_dialog(get_parent_window(), GPOINTER_TO_INT(data));
}

static void launch_callback( GtkWidget * button, gpointer data ) {
	DBusError err;
	DBusConnection* bus;
	DBusMessage* m;

	dbus_error_init(&err);

	bus = dbus_bus_get(DBUS_BUS_SESSION, &err);
	if (dbus_error_is_set(&err)) {
		dbus_error_free(&err);
		return;
	}

	write_config();

	m = dbus_message_new_method_call("com.jcq.cpcm",
										"/com/jcq/cpcm",
										"com.jcq.cpcm",
										"game_autorun");

	dbus_connection_send(bus, m, NULL);
	dbus_connection_flush(bus);

	dbus_message_unref(m);
}

#if MAEMO_VERSION >= 5
#include <gdk/gdkx.h>
#include <X11/Xatom.h>

/** Called for each of the play/restart/continue buttons */
static void found_ogs_button_callback(GtkWidget *widget, gpointer data)
{
	/*hildon_gtk_widget_set_theme_size(widget,
		HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_widget_set_size_request(widget, 200, -1);
	gtk_box_set_child_packing(GTK_BOX(data), widget,
		FALSE, FALSE, 0, GTK_PACK_START);*/
}
/** Converts the window into a stackable one */
static void plugin_realized_callback(GtkWidget *widget, gpointer data)
{
	GdkDisplay *display;
    Atom atom;
    unsigned long val = 0;
	GtkWidget* window = gtk_widget_get_ancestor(widget, GTK_TYPE_WINDOW);

    /* Set additional property "_HILDON_STACKABLE_WINDOW",
    	to allow the WM to manage it as a stackable window. */
    display = gdk_drawable_get_display(window->window);
    atom = gdk_x11_get_xatom_by_name_for_display (display,
    				"_HILDON_STACKABLE_WINDOW");
    XChangeProperty(GDK_DISPLAY_XDISPLAY(display),
    				 GDK_WINDOW_XID(window->window),
    				 atom, XA_INTEGER, 32, PropModeReplace,
                     (unsigned char *) &val, 1);
}
#endif

static GtkWidget * load_plugin(void)
{
	g_type_init();
	gcc = gconf_client_get_default();

	GtkWidget* parent = gtk_vbox_new(FALSE, HILDON_MARGIN_DEFAULT);

/* Select ROM button */
	drivea_btn = HILDON_BUTTON(hildon_button_new_with_text(
		HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
		HILDON_BUTTON_ARRANGEMENT_VERTICAL,
		_("Drive A"),
		NULL));
	hildon_button_set_alignment(drivea_btn, 0.0f, 0.5f, 0.9f, 0.2f);

	driveb_btn = HILDON_BUTTON(hildon_button_new_with_text(
		HILDON_SIZE_HALFSCREEN_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
		HILDON_BUTTON_ARRANGEMENT_VERTICAL,
		_("Drive B"),
		NULL));
	hildon_button_set_alignment(driveb_btn, 0.0f, 0.5f, 0.9f, 0.2f);

	// Ugly hacks: resize the Osso-Games-Startup buttons
	GtkBox* button_box =
		GTK_BOX(gtk_widget_get_parent(gs.ui->play_button));
	gtk_box_set_spacing(button_box, HILDON_MARGIN_DEFAULT);
	gtk_container_foreach(GTK_CONTAINER(button_box),
		found_ogs_button_callback, button_box);

	// Ugly hacks: move the select rom button to the left.
	gtk_box_pack_start_defaults(button_box, GTK_WIDGET(drivea_btn));
	gtk_box_reorder_child(button_box, GTK_WIDGET(drivea_btn), 0);


	GtkBox* opt_hbox0 = GTK_BOX(gtk_hbox_new(FALSE, HILDON_MARGIN_DEFAULT));
	gtk_box_pack_start_defaults(opt_hbox0, GTK_WIDGET(driveb_btn));
// First row of widgets
	GtkBox* opt_hbox1 = GTK_BOX(gtk_hbox_new(FALSE, HILDON_MARGIN_DEFAULT));
	/*sound_check = HILDON_CHECK_BUTTON(hildon_check_button_new(
			HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT));
	gtk_button_set_label(GTK_BUTTON(sound_check), _("Sound"));
	gtk_widget_set_size_request(GTK_WIDGET(sound_check), 200, -1);
*/
	sound_check = gtk_check_button_new_with_label( "Sound" );

	controls_btn = HILDON_BUTTON(hildon_button_new(
			HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL));
	gtk_button_set_label(GTK_BUTTON(controls_btn), _("Joysticks…"));
	gtk_widget_set_size_request(GTK_WIDGET(controls_btn), 200, -1);

	launch_btn = HILDON_BUTTON(hildon_button_new(
			HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL));
	gtk_button_set_label(GTK_BUTTON(launch_btn), _("Launch drive A"));
	gtk_widget_set_size_request(GTK_WIDGET(launch_btn), 200, -1);

	gtk_box_pack_start_defaults(opt_hbox1, GTK_WIDGET(sound_check));
	gtk_box_pack_start_defaults(opt_hbox0, GTK_WIDGET(launch_btn));
	gtk_box_pack_start_defaults(opt_hbox0, GTK_WIDGET(controls_btn));
	gtk_box_pack_start(GTK_BOX(parent), GTK_WIDGET(opt_hbox0), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), GTK_WIDGET(opt_hbox1), FALSE, FALSE, 0);

// Second row of widgets

// Load current configuration from GConf
	gtk_toggle_button_set_active(sound_check,
		gconf_client_get_bool(gcc, G_CPCM_SOUND, NULL));

	set_rom(gconf_client_get_string(gcc, G_CPCM_DRIVE_A, NULL), 0);
	set_rom(gconf_client_get_string(gcc, G_CPCM_DRIVE_B, NULL), 1);

	g_signal_connect_after(G_OBJECT(parent), "realize",
						G_CALLBACK(plugin_realized_callback), NULL);

	// Connect signals
	g_signal_connect(G_OBJECT(drivea_btn), "clicked",
					G_CALLBACK(select_rom_callback), 0);
	g_signal_connect(G_OBJECT(driveb_btn), "clicked",
					G_CALLBACK(select_rom_callback), 1);
	g_signal_connect(G_OBJECT(controls_btn), "clicked",
					G_CALLBACK(controls_item_callback), NULL);
	g_signal_connect(G_OBJECT(launch_btn), "clicked",
					G_CALLBACK(launch_callback), NULL);

	game_state_update();

	return parent;
}

static void unload_plugin(void)
{
	game_state_clear();
	//save_clear();
	g_object_unref(gcc);
	if( drivea_path ) g_free( drivea_path );
	if( driveb_path ) g_free( driveb_path );
}

static void write_config(void)
{
/* Write current settings to gconf */
	gconf_client_set_bool(gcc, G_CPCM_SOUND,
		gtk_toggle_button_get_active(sound_check), NULL);

	if( drivea_path )
		gconf_client_set_string( gcc, G_CPCM_DRIVE_A, drivea_path, NULL );
	else
		gconf_client_unset( gcc, G_CPCM_DRIVE_A, NULL );

	if( driveb_path )
		gconf_client_set_string(gcc, G_CPCM_DRIVE_B, driveb_path, NULL);
	else
		gconf_client_unset( gcc, G_CPCM_DRIVE_B, NULL );
}

static GtkWidget **load_menu(guint *nitems)
{
	const HildonSizeType button_size =
		HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH;
	menu_items[0] = hildon_gtk_button_new(button_size);
	gtk_button_set_label(GTK_BUTTON(menu_items[0]), _("Settings…"));
	menu_items[1] = hildon_gtk_button_new(button_size);
	gtk_button_set_label(GTK_BUTTON(menu_items[1]), _("About…"));
	*nitems = 2;

	g_signal_connect(G_OBJECT(menu_items[0]), "clicked",
					G_CALLBACK(settings_item_callback), NULL);
	g_signal_connect(G_OBJECT(menu_items[1]), "clicked",
					G_CALLBACK(about_item_callback), NULL);
/*
	GtkMenu* settings_menu = GTK_MENU(gtk_menu_new());
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_items[0]),
		GTK_WIDGET(settings_menu));

	GtkMenu* controls_menu = GTK_MENU(gtk_menu_new());
	GtkMenuItem* controls_item =
		GTK_MENU_ITEM(gtk_menu_item_new_with_label(_("Controls")));
	gtk_menu_item_set_submenu(controls_item, GTK_WIDGET(controls_menu));
	gtk_menu_append(settings_menu, GTK_WIDGET(controls_item));

	GtkMenuItem* advanced_item =
		GTK_MENU_ITEM(gtk_menu_item_new_with_label(_("Advanced…")));
	gtk_menu_append(settings_menu, GTK_WIDGET(advanced_item));

	GtkMenuItem* player1_item =
		GTK_MENU_ITEM(gtk_menu_item_new_with_label(_("Player 1…")));
	gtk_menu_append(controls_menu, GTK_WIDGET(player1_item));
	GtkMenuItem* player2_item =
		GTK_MENU_ITEM(gtk_menu_item_new_with_label(_("Player 2…")));
	gtk_menu_append(controls_menu, GTK_WIDGET(player2_item));

	g_signal_connect(G_OBJECT(player1_item), "activate",
					G_CALLBACK(controls_item_callback), GINT_TO_POINTER(1));
	g_signal_connect(G_OBJECT(player2_item), "activate",
					G_CALLBACK(controls_item_callback), GINT_TO_POINTER(2));
	g_signal_connect(G_OBJECT(advanced_item), "activate",
					G_CALLBACK(settings_item_callback), NULL);
	g_signal_connect(G_OBJECT(menu_items[1]), "activate",
					G_CALLBACK(about_item_callback), NULL);
*/
	return menu_items;
}

static void update_menu(void)
{
	// Nothing to update in the current menu
}

// From osso-games-startup
#define MA_GAME_PLAY 1
#define MA_GAME_RESTART 2
#define MA_GAME_OPEN 3
#define MA_GAME_SAVE 4
#define MA_GAME_SAVE_AS 5
#define MA_GAME_HELP 6
#define MA_GAME_RECENT_1 7
#define MA_GAME_RECENT_2 8
#define MA_GAME_RECENT_3 9
#define MA_GAME_RECENT_4 10
#define MA_GAME_RECENT_5 11
#define MA_GAME_RECENT_6 12
#define MA_GAME_CLOSE 13
#define MA_GAME_HIGH_SCORES 14
#define MA_GAME_RESET 15
#define MA_GAME_CHECKSTATE 16
#define MA_GAME_SAVEMENU_REFERENCE 17
#define ME_GAME_OPEN     20
#define ME_GAME_SAVE     21
#define ME_GAME_SAVE_AS  22
#define MA_GAME_PLAYING_START 30
#define MA_GAME_PLAYING 31

static void plugin_callback(GtkWidget * menu_item, gpointer data)
{
	switch ((gint) data) {
		case ME_GAME_OPEN:
			save_load(get_parent_window());
			break;
		case ME_GAME_SAVE:
			save_save(get_parent_window());
			break;
		case ME_GAME_SAVE_AS:
			save_save_as(get_parent_window());
			break;
		case MA_GAME_PLAYING_START:
			if (!menu_item) {
				// Avoid duplicate message
				break;
			}
			break;
	}
}

