#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm

import gobject

import gtk
import hildon

import time

import beye

import dataprocessing

def dhmSince(since):
    if since == None:
        return '-'
    now = time.time()
    if since > now - 60:
        return '-'
    return dhm(now - since)

def dhm(timeDuration):
    if timeDuration == None or timeDuration < 60:
        return '-'
    m = int(timeDuration / 60)
    h = m / 60
    m = m % 60
    if (h >= 24):
        d = int(h / 24)
        h = h % 24
        if d == 1:
            days = 'day'
        else:
            days = 'days'
        return "%d %s, %d:%02d" % (d, days, h, m)
    else:
        return "%d:%02d" % (h, m)
   

class Statistics(hildon.StackableWindow):
    def __init__(self, dataStorage):
        hildon.StackableWindow.__init__(self)

        self.dataStorage = dataStorage

        self.connect("delete_event", self.onWindowClose)

        oldest = self.dataStorage.getOldestObservationTime()
        if oldest == None:
            oldest = 0
        self.set_title('Statistics, last %d days' % int((time.time() - oldest) / 86400))

        # Setting a label in the new window
        label = gtk.Label("Loading statistics...")
 
        vbox = gtk.VBox(False, 0)
        vbox.pack_start(label, True, True, 0)
        self.add(vbox)
 
        # This call show the window and also add the window to the stack
        hildon.hildon_gtk_window_set_progress_indicator(self, False)
        gobject.idle_add(self.showStatistics)
        self.show_all()

    def onWindowClose(self, param1, param2):
        hildon.hildon_gtk_window_set_progress_indicator(self, False)

    def showStatistics(self):
        now = time.time()
        chargeData = dataprocessing.getChargeInfo(self.dataStorage,
                                                  self.dataStorage.getOldestObservationTime(),
                                                  now, now)

        # Items shorter than this lenght (in seconds) will be discarded.
        discardTreshold = 300

        # Items longer that this length are very likely to be bad data
        # (e.g.) from changing clock during recording, and are discarded.
        badDataTreshold = 30 * 86400

        # How large amount of data is considered for high and low discharge rates
        # 1 / highLowSelect
        # e.g. for highLowSelect = 4, the highest 1/4 of data is used for high rate,
        # and the lowest 1/4 for low rate
        highLowSelect = 3
        
        discharges = []
        charges = []

        for segment in chargeData:
            discharges.extend((i for i in segment if i[0] == (False, True)
                                                     and i[2] - i[1] > discardTreshold
                                                     and i[2] - i[1] < badDataTreshold))
            charges.extend((i for i in segment if i[0] == (True, False)
                                                  and i[2] - i[1] > discardTreshold
                                                  and i[2] - i[1] < badDataTreshold))

        chargeTime = 0
        maxChargeTime = 0
        for item in charges:
            chargeTime += item[2] - item[1]
            maxChargeTime = max(maxChargeTime, item[2] - item[1])

        dischargeRates = []
        drainedmAh = {'high': 0.0, 'avg': 0.0, 'low': 0.0}
        # Total and rate discharge times are different.
        # Total time is calculated from (charger is disconnected OR break)
        #                          to (charger is connected OR last any observation before break)
        # Time for rate is calculated between actual CHARGE observations, which may be
        # different from charger connect / disconnect times.
        dischargeTimeForTotal = 0
        dischargeTimeForRate = {'high': 0, 'avg': 0, 'low': 0}
        maxDischargeTime = 0
        for item in discharges:
            dischargeTimeForTotal += item[2] - item[1]
            maxDischargeTime = max(maxDischargeTime, item[2] - item[1])

            try:
                startTime, startValue = self.dataStorage.getFirstObservationForType(item[1:3], beye.DataSourceHal.CHARGE)
                endTime, endValue = self.dataStorage.getLastObservationForType(item[1:3], beye.DataSourceHal.CHARGE)
            except:
                # There might not be any observations in the interval.
                continue

            if endValue > startValue:
                # We consider this an instance of bad data.
                continue

            if endTime <= startTime:
                # There's no data here
                continue

            dischargeTimeForRate['avg'] += endTime - startTime
            drainedmAh['avg'] += startValue - endValue
            
            dischargeRates.append(((float(startValue-endValue) / (endTime - startTime)),
                                   endTime-startTime,
                                   startValue - endValue))
            
        dischargeRates.sort()

        try:
            avgChargeTime = int(chargeTime/len(charges))
        except ZeroDivisionError:
            avgChargeTime = None

        try:
            avgDischargeTime = int(dischargeTimeForTotal/len(discharges))
        except ZeroDivisionError:
            avgDischargeTime = None

        try:
            lastChargeEnd = charges[-1][2]
        except IndexError:
            lastChargeEnd = None

        rate = {'high': 0.0, 'avg': 0.0, 'low': 0.0}

        try:
            rate['avg'] = drainedmAh['avg'] / dischargeTimeForRate['avg']
        except ZeroDivisionError:
            pass

        if len(dischargeRates) >= highLowSelect:
            lowbound = len(dischargeRates) / highLowSelect
            highbound = (highLowSelect-1)*len(dischargeRates) / highLowSelect
            for rates, key in ((dischargeRates[0:lowbound], 'low'), (dischargeRates[highbound:], 'high')):
                for r in rates:
                    dischargeTimeForRate[key] += r[1]
                    drainedmAh[key] += r[2]
                rate[key] = drainedmAh[key] / dischargeTimeForRate[key]

        maxCharge = self.dataStorage.getMaxCharge()
        if (maxCharge == None):
            maxCharge = 1300

        try:
            t, currentCharge = self.dataStorage.getLastObservationForType((now-7200, now), beye.DataSourceHal.CHARGE)   
        except:
            currentCharge = None
            
        batteryLife = {}
        batteryRemain = {}
        for key in ('high', 'avg', 'low'):
            try:
                batteryLife[key] = int(maxCharge / rate[key])
            except:
                batteryLife[key] = None
            try:
                batteryRemain[key] = int(currentCharge / rate[key])
            except:
                batteryRemain[key] = None

        if currentCharge == None:
            currentCharge = '?'
        else:
            currentCharge = str(currentCharge)

        parea = hildon.PannableArea()
        hbox = gtk.HBox(False, 0)
        vboxKeys = gtk.VBox(False, 0)
        vboxValues = gtk.VBox(False, 0)
        hbox.pack_start(gtk.VBox(False, 0))
        hbox.pack_start(vboxKeys)
        hbox.pack_start(vboxValues)
        for k, v in (('', ''),
                     ('Time since last charge', '%s' % dhmSince(lastChargeEnd)),
                     ('Number of charges (>%ds) recorded' % discardTreshold, '%d' % len(charges)),
                     ('Total charge time', '%s' % dhm(chargeTime)),
                     ('Average charge time', '%s' % dhm(avgChargeTime)),
                     ('Longest charge time', '%s' % dhm(maxChargeTime)),
                     ('', ''),
                     ('Number of discharges (>%ds) recorded' % discardTreshold, '%d' % len(discharges)),
                     ('Total discharge time', '%s' % dhm(dischargeTimeForTotal)),
                     ('Average time between charges', '%s' % dhm(avgDischargeTime)),
                     ('Longest time without charge', '%s' % dhm(maxDischargeTime)),
                     ('Total ampere-hours discharged', '%.2f Ah' % (drainedmAh['avg'] / 1000)),
                     ('Discharge current (mAh/hour)', ''),
                     ('- Low use (lowest 1/%d of data)' % highLowSelect, '%.1f mA' % (rate['low'] * 3600)),
                     ('- Average use (all data)', '%.1f mA' % (rate['avg'] * 3600)),
                     ('- High use (highest 1/%d of data)' % highLowSelect, '%.1f mA' % (rate['high'] * 3600)),
                     ('', ''),
                     ('*Battery life for full charge (%d mAh)' % maxCharge, ''), 
                     ('- Low use', '%s' % dhm(batteryLife['low'])), 
                     ('- Average use', '%s' % dhm(batteryLife['avg'])), 
                     ('- High use', '%s' % dhm(batteryLife['high'])), 
                     ('*Battery life left now (%s mAh)' % currentCharge, ''),
                     ('- Low use', '%s' % dhm(batteryRemain['low'])),
                     ('- Average use', '%s' % dhm(batteryRemain['avg'])),
                     ('- High use', '%s' % dhm(batteryRemain['high'])),
                     ('', ''),
                     ('(*) Values are extrapolated from average, low and', ''),
                     ('      high discharge currents, accuracy may vary.', '')
                    ):
            label = gtk.Label(k)
            label.set_alignment(0, 0)
            vboxKeys.pack_start(label, False, False, 0)
 
            label = gtk.Label(v)
            label.set_alignment(0, 0)
            vboxValues.pack_start(label, False, False, 0)
           

        self.remove(self.get_children()[0])
        parea.add_with_viewport(hbox)
        self.add(parea)
        self.show_all()

        hildon.hildon_gtk_window_set_progress_indicator(self, False)
