from youamp.ui.detailswindow import DetailsWindow
import gtk
import gtk.gdk
import time

from youamp.ui.window import Window
from youamp.ui.preferences import Preferences
from youamp.ui.searchview import SearchView

from youamp.ui.playlist import PlaylistView
from youamp.ui.popupmenu import SongMenu, PlaylistMenu

from youamp.ui.elements import Controls, GladeAdapter, AboutDialog
from youamp.ui import xml_escape

from youamp import VERSION, data_path

class UserInterface:
    NOTIFY_STR = "{0} <i>{1}</i>\n{2} <i>{3}</i>"

    def __init__(self, controller):
        player = controller.player
        config = controller.config
        library = controller.library
        scrobbler = controller.scrobbler
        self._player = player
        self._config = config
        self.library = library
        self._press_time = {"F6": None, "F7": None, "F8": None}
        
        self.song_meta = controller.song_meta
        
        self._controller = controller
        
        # Build Interface
        xml = GladeAdapter(data_path + "interface.glade")

        # Create Views
        dw = DetailsWindow(controller.song_meta, xml)
        smenu = SongMenu(config, player, dw, xml)
        plmenu = PlaylistMenu(xml)
        self.smenu = smenu
        self.plmenu = plmenu
        self.vmenu = xml.get_object("view_menu")

        self._view = [SearchView(controller.main_list, controller, config, smenu, self.vmenu)]
        self._cur_view = self._view[0]
        
        self._view += [PlaylistView(None, controller, smenu, plmenu, self.vmenu)]
        self._view[1].connect("model-deleted", self._remove_playlist)
        
        lists = library.get_playlists()
  
        if len(lists) > 0:
            controller.cur_pl = lists[0]
            self._view[1].replace(lists[0])
  
        sw = self._view[0]
        sw.shuffle.set_active(self._config["shuffle"])
        
        self.lists = lists # keep reference
        
        new = gtk.RadioMenuItem(label=_("New Playlist"))
        
        pl_sub = xml.get_object("playlist_sel_menu")
        self._pl_sub = pl_sub
        
        self._pl_grp = None
        self._plis = {}
        
        for m in lists:
            pli = self._get_pl_item(m)
            
            if self._pl_grp is None:
                pli.set_active(True)
                self._pl_grp = pli
                
            pl_sub.prepend(pli)
        
        # Windows
        self.window = Window(player, dw, self.song_meta, xml)
        self.window._w.set_menu(xml.get_object("main_menu"))
        
        about = AboutDialog(self.window._w)

        prefs = Preferences(config, scrobbler, xml)

        # Controls
        ctls = Controls(player, config, xml)
        
        # Menu
        if config["is-browser"]:
            xml.get_object("view_browse").set_active(True)
        else:
            xml.get_object("view_search").set_active(True)

        if not config["shuffle"]:
            xml.get_object("order_"+config["order-by"]).set_active(True)
               
        # Add {Search, Playlist}Views        
        self.nb = xml.get_object("notebook1")
        
        for v in self._view:
            self.nb.append_page(v)

        # disable implicit playlist change
        self.nb.connect("switch-page", self._change_playlist, player)

        # Signals
        xml.connect_signals({"show-preferences": lambda *args: prefs.cshow(),
                             "show-about": lambda *args: about.show(),
                             "order-album": lambda caller: controller.order_changed("album"),
                             "order-date": lambda caller: controller.order_changed("date"),
                             "order-playcount": lambda caller: controller.order_changed("playcount"),
                             "hide-on-delete": gtk.Widget.hide_on_delete,
                             "toggle": lambda caller: player.toggle(),
                             "previous": lambda caller: player.previous(),
                             "next": lambda caller: player.next(),
                             "view-search": lambda caller: self._search_mode(),
                             "view-browse": lambda caller: self._browse_mode(),
                             "view-playlist": lambda caller: self.playlist_mode(),
                             "select-current": lambda caller: self._cur_view.restore(),
                             "new-playlist": lambda caller: self._add_playlist(library.get_new_playlist())})
        
        self.window._w.connect("key-press-event", self._handle_keypress)
        self.window._w.connect("key-release-event", self._handle_keyrelease)
        self.window._w.connect("destroy", controller.quit)
        ctls.pos.connect("change-value", lambda caller, *a: player.seek_to(caller.get_value()))
        
        self._toggle = xml.get_object("playback_item")
        self._thndl = self._toggle.connect("toggled", lambda caller: player.toggle())
        
        player.connect("toggled", self._watch_toggled)
        player.connect("song-changed", self._on_song_changed)
        player.playlist.connect("list-switched", self._switch_to)
        
        # change to library on browsing
        config.notify_add("is-browser", lambda *args: self.nb.set_current_page(0))
    
    def _rename_pli(self, caller, new_title, pli):
        del self._plis[pli.get_child().get_text()]   
        pli.get_child().set_text(new_title)
        self._plis[new_title] = pli
        
    def _get_pl_item(self, m):
        pli = gtk.RadioMenuItem(group=self._pl_grp, label=m.title)
                
        pli.connect("activate", self.set_current_pl, m)
        m.connect("renamed", self._rename_pli, pli)
        
        self._plis[m.title] = pli
        
        return pli

    def set_current_pl(self, caller, pl):
        self._controller.cur_pl = pl
        self._view[1].replace(pl)
    
    def _remove_playlist(self, caller, obj): 
        self.lists.remove(obj)
        self._plis[obj.title].destroy()
        del self._plis[obj.title]
        self._search_mode()
        
        if len(self.lists) > 0:
            self._view[1].replace(self.lists[-1])
    
    def playlist_mode(self, *args):
        if len(self._view) < 2:
            self._add_playlist(self.library.get_new_playlist())
        else:
            self.nb.set_current_page(1)
    
    def _browse_mode(self):
        self.nb.set_current_page(0)
        self._view[0].browse_mode()
    
    def _search_mode(self):
        self.nb.set_current_page(0)
        self._view[0].search_mode()
        
    def _on_song_changed(self, player, newsong):
        # move cursor to new position       
        self._cur_view.playlist.set_cursor(player.playlist.pos)

    def _switch_to(self, caller, model):
        i = [v.playlist.get_model() for v in self._view].index(model)
        n = self.nb.page_num(self._view[i])
        self.nb.set_current_page(n)

    def _add_playlist(self, playlist):
        self.lists.append(playlist)
        self.set_current_pl(None, playlist)
        self._view[1].label.just_created()
        self.playlist_mode()
        
        pli = self._get_pl_item(playlist)   
        self._pl_sub.prepend(pli)
        pli.set_active(True)
        pli.show_all()

    def _change_playlist(self, nb, page, num, player):
        self._cur_view = nb.get_nth_page(num)
        player.playlist.set(self._cur_view.playlist.get_model())

    def show_notification(self, song):
        return # no notifications on maemo for now...
        body = self.NOTIFY_STR.format(
                                    _("by"),
                                    xml_escape(song["artist"]),
                                    _("from"),
                                    xml_escape(song["album"]))
        self._notify.update(xml_escape(song["title"]), body)

        path = self.song_meta.get_cover_path(song)
        
        if path is None:
            cover = gtk.icon_theme_get_default().load_icon("audio-x-generic", 128, 0)
        else:
            cover = gtk.gdk.pixbuf_new_from_file_at_size(path, 128, 128)
            
        self._notify.set_icon_from_pixbuf(cover)
        self._notify.show()
    
    def restore(self):
        for v in self._view:
            v.restore()
    
    def _watch_toggled(self, caller, state):
        self._toggle.handler_block(self._thndl)
        self._toggle.set_active(state)
        self._toggle.handler_unblock(self._thndl)
                  
    def _handle_keypress(self, widget, event):
        key = gtk.gdk.keyval_name(event.keyval)
        
        if key == "Right":
            self._player.next()
            return True # event handled
        elif key == "Left":
            self._player.previous()
            return True # event handled
        elif key in ("F6", "F7", "F8"):
            self._press_time[key] = time.time()

    def _handle_keyrelease(self, widget, event):
        key = gtk.gdk.keyval_name(event.keyval)
        
        if key == "F6":
            if time.time() - self._press_time[key] < 1:
                self.window.toggle_fullscreen()
            else:
                self._player.toggle()
        elif key == "F7":
            if time.time() - self._press_time[key] < 1:
                self._on_vol_up()
            else:
                self._player.next()
        elif key == "F8":
            if time.time() - self._press_time[key] < 1:
                self._on_vol_down()
            else:
                self._player.previous()

    def _on_vol_down(self):
        vol = self._config["volume"]
        self._config["volume"] = max(0.0, vol - 0.1) # cut to range
    
    def _on_vol_up(self):
        vol = self._config["volume"]
        self._config["volume"] = min(1.0, vol + 0.1) # cut to range
